/*
 * DEVICE.LIB - written by Jan Kriesten 
 *
 * (c) 1994 by Jan Kriesten, Friedhofstr. 25 b, 35043 Marburg, Germany
 *     EMail:    Maus GI
 *               90:400/1002@nest.ftn
 *               51:601/103@atarinet.ftn
 *               2:244/4344@fidonet.org
 *
 * Routines for using the ports - supporting HSModem, FastSerial and
 * MiNT block device routines.
 * Including fast sending routines when no block device routines are
 * availbable.
 */

#ifndef __DEVICE_H__
#define __DEVICE_H__

/*--- includes              ---*/

/*--- defines               ---*/

#define     IO_I_BUFFER     1L      /* Inbound buffer       */
#define     IO_O_BUFFER     2L      /* Outbound buffer      */
#define     IO_BUFFERS      3L      /* Both buffers         */

/*--- types                 ---*/

typedef struct _dev_list
{
    BYTE                *name;      /* Device name          */
    long                curr_dte;   /* current DTE rate     */
    struct _dev_list    *next;      /* Pointer to next dev  */
} DEV_LIST;

/*--- variables             ---*/

/*--- prototypes            ---*/

/*
 * InitDevices:
 *      To be called at the beginning of the program to initialize
 *      the all routines.
 *
 * Parameters:
 *      pause_normal: Routine to spent some time or NULL.
 *      pause_super : same as normal, but will be called from 
 *                    supervise modus!
 */
DEV_LIST *InitDevices( void *pause_normal, void *pause_super );

/*
 * TermDevices:
 *      Frees all memory allocated during initialization. Devicelist
 *      is no longer valid after calling this routine.
 *      Just to be called before exiting the program.
 */
void TermDevices( void );

/*
 * Open-/CloseDevice:
 *      The given port will be initialized and the speedlist will be
 *      created. Returns TRUE if everything went well and the Device
 *      could be opened. FALSE is returned if the Device is already in
 *      use or memory allocation failed ...
 */
BOOLEAN OpenDevice( DEV_LIST *port );
void CloseDevice( DEV_LIST *port );

/*
 * GetBiosNr:
 *      Returns the corresponding bios number for the device or -1
 *      if the device isn't known by the bios. Should only be used
 *      if you have to transfer the bios number as a parameter to
 *      another program.
 */
int GetBiosNr( DEV_LIST *port );

/*
 * GetSpeedList:
 *      Returns the speedlist for the corresponding device. The 
 *      return value is only defined for _open_ devices!
 *      The speedlist is an array of long values (from higher dte-
 *      rates to lower). The end of the list is specified by a
 *      speed of -1L.
 */
long *GetSpeedList( DEV_LIST *port );

/*
 * SetDTESpeed:
 *      Sets the DTE speed to `speed'. If `speed' isn't available,
 *      the current DTE is returned.
 *      In all cases the new DTE speed is returned by the function.
 *      Just call this function for open devices!
 */
long SetDTESpeed( DEV_LIST *port, long speed );

/*
 * PortSendByte:
 *      Sends character `c' to the port `port'.
 */
BOOLEAN PortSendByte( DEV_LIST *port, BYTE c );

/*
 * PortSendBlock:
 *      Sends `len' characters from `block' to device `port'. 
 *      If `tst_dcd' is TRUE, the carrier is checked during sending.
 */
BOOLEAN PortSendBlock( DEV_LIST *port, BYTE *block, long len, BOOLEAN tst_dcd );

/*
 * PortGetByte:
 *      Returns the next available character. While no character is
 *      available the function pause_1 is called.
 *      The routine will not pause, if there's no carrier available!
 */
int PortGetByte( DEV_LIST *port );

/*
 * PortPeekByte:
 *      Returns the next available character _without_ actually read
 *      it from the port (i.e. the character is still available from
 *      port).
 *      If no char is available, -1 is returned.
 */
int PortPeekByte( DEV_LIST *port );

/*
 * OutIsEmpty:
 *      Checks, if the outbound iorec buffer is empty.
 */
BOOLEAN OutIsEmpty( DEV_LIST *port );

/*
 * WaitOutEmpty:
 *      Waits, until the outbound iorec buffer is empty. If tst_dcd
 *      is set to TRUE, the carrier is checked while waiting.
 *      If wait_hundreds is != 0, WaitOutEmpty waits a maximum time
 *      of wait_hundreds hundreds seconds. If the outbound buffer is
 *      not empty  in this period, the outbound buffer is cleared.
 */
BOOLEAN WaitOutEmpty( DEV_LIST *dev, BOOLEAN tst_dcd, unsigned int wait_hundreds );

/*
 * CharAvailable:
 *      Returns TRUE if a char is available from `port', else FALSE.
 */
BOOLEAN CharAvailable( DEV_LIST *port );

/*
 * ClearIOBuffer:
 *      Clears the IO buffer `which' from device port.
 *      IO_I_BUFFER:    Inbound buffer;
 *      IO_O_BUFFER:    Outbound buffer;
 *      IO_BUFFERS:     Both buffers;
 */
void ClearIOBuffer( DEV_LIST *port, long which );

/*
 * DtrOn:
 * DtrOff:
 *      Sets/Unsets the DTR of port `port'.
 */
void DtrOn( DEV_LIST *port );
void DtrOff( DEV_LIST *port );

/*
 * IsCarrier:
 *      Returns TRUE if carrier on port `port' is available, else
 *      FALSE;
 */
BOOLEAN IsCarrier( DEV_LIST *port );

/*
 * StartReceiver:
 * StopReceiver:
 *      Enables/disables the reiceiver. 
 *      StopReceiver should be used before disk access and
 *      StartReceiver after finished disk i/o.
 */
void StartReceiver( DEV_LIST *port );
void StopReceiver( DEV_LIST *port );

/*
 * Set/Get*xBuffer:
 *      Sets/returns the size of the inbound/outbound iorec
 *      buffers.
 */
int SetRxBuffer( DEV_LIST *port, int size );
int GetRxBuffer( DEV_LIST *port );
int SetTxBuffer( DEV_LIST *port, int size );
int GetTxBuffer( DEV_LIST *port );

/*
 * Misceallaneous functions:
 */

/*
 * get_tos:
 *      returns TOS version
 */
int get_tos( void );

/*
 * getcookie:
 *      searches `cookie' in the cookie jar. If found TRUE is
 *      returned. `value' then holds the value of the found
 *      cookie.
 */
BOOLEAN getcookie( long cookie, long *value );

/*
 * Get200Hz:
 *      returns the value of the 200Hz hardware counter.
 */
unsigned long Get200Hz( void );

/*
 * Calc200Hz:
 *      returns the value the 200Hz hardware counter will have
 *      after `time_to_set' hundreds seconds.
 */
unsigned long Calc200Hz( unsigned long time_to_set );

/*--- End of device.h module  ---*/
#endif

