#include <stdio.h>
#include <io.h>
#include <sys\stat.h>
#include <fcntl.h>
#include <malloc.h>
#include <memory.h>
#include "syms.h"
#include "structs.h"
#include "global.h"

#define MAXBF 5
#define BSIZE 1000

typedef struct buf
    {
    struct buf *pNext;
    char        acBuffer[BSIZE];
    } DATABUF;

typedef struct file
    {
    char    *sName;
    int      iHandle;
    int      iFlags;
    DATABUF *pFirstBuffer;
    DATABUF *pLastBuffer;
    long     lFileLen;
    long     lReadPtr;
    int      iBufOffset;
    char     acBuffer[BSIZE];
    } FILEINFO;
    
FILEINFO IOFiles[MAXBF];

int iActiveFiles = 0;

#define F_READING 1
#define F_FILEIO  2
#define F_LOCKED  4

static void t_freef(FILEINFO *pFile);
static struct buf *t_dequeue(FILEINFO *pFile);
static void t_wfull(FILEINFO *pFile);
static void t_rfill(FILEINFO *pFile);
static void t_putc(FILEINFO *pFile, char c);
static char t_getc(FILEINFO *pFile);
static void t_enqueue(FILEINFO *pFile, struct buf *pBuffer);

/****************************************************************************
 *                                                                          *
 * Open the named temporary file                                            *
 *                                                                          *
 ****************************************************************************/

int t_open(char *sFile)

{
int iTemp;

if(iActiveFiles >= MAXBF)               /* Check for the maximum number of  */
    fatal("Too many b-files");          /* files and panic if we hit it     */

iTemp = iActiveFiles++;                 /* Chalk up another used file       */
IOFiles[iTemp].sName = sFile;           /* Store the file name              */

return iTemp;                           /* Return with the file number      */
}


/****************************************************************************
 *                                                                          *
 * Remove any disk based temporary files and zero the used file count       *
 *                                                                          *
 ****************************************************************************/

void t_exit(void)

{
int iCount;

for(iCount = 0; iCount < iActiveFiles; iCount++) /* Check all file entries  */
    {
    if(IOFiles[iCount].iFlags & (F_FILEIO != 0))  /* If it is a real file     */
                                                /* Then pick the correct    */
    #ifdef TOS                                  /* delete function to use   */
        remove(IOFiles[iCount].sName);
    #else
        unlink(IOFiles[iCount].sName);
    #endif
    }

iActiveFiles = 0;                               /* Say all files are free   */
}


/****************************************************************************
 *                                                                          *
 * Flush one RAM based temporary file to disk and free up memory            *
 *                                                                          *
 ****************************************************************************/

int t_frees(void)

{
int iCount;

for(iCount = 0; iCount < iActiveFiles; iCount++)    /* Check all temp files */
    {
    /*
     * Only do this for non disk files with at least one buffer block
     * allocated to them
     */

    if(((IOFiles[iCount].iFlags & F_FILEIO) == 0) && IOFiles[iCount].pFirstBuffer)
        {
        t_freef(&IOFiles[iCount]);      /* Flush the current file           */
        return(1);                      /* and indicate the fact on return  */
        }
    }

return(0);                              /* This means we didn't free any    */
}


/****************************************************************************
 *                                                                          *
 * Turf a RAM resident temp file out into the cruel world of the disk I/O   *
 * system.                                                                  *
 *                                                                          *
 ****************************************************************************/

static void t_freef(FILEINFO *pFile)

{
int      iFileHndl;
DATABUF *pBuffer;

/*
 * Announce to world if verbose flag is on
 */

if(vflag)
    printf("Using disk temporary file for %s\n", pFile->sName);

pFile->iFlags |= F_FILEIO;               /* Mark file as being disk based   */
iFileHndl = open(pFile->sName, O_RDWR|O_BINARY); /* Open file for read/write access */

/*
 * If we couldn't open it we assume it's not there and try to create it.
 */

if(iFileHndl < 0)
    iFileHndl = open(pFile->sName, O_CREAT|O_TRUNC|O_BINARY, S_IREAD | S_IWRITE);

if(iFileHndl < 0)                                /* File failure so exit    */
    fatals("Can't open tmp file", pFile->sName);

pFile->iHandle = iFileHndl;              /* Store file handle for later     */

/*
 * Walk list of buffers for this file, write each one to disk and then
 * release the memory allocated to the buffer.
 */

while(pFile->pFirstBuffer)
    {
    pBuffer = t_dequeue(pFile);         /* Remove first buffer from list    */

    if(write(iFileHndl,pBuffer->acBuffer, BSIZE) != BSIZE) /* Write to disk */
        fatal("write error");                              /* Well, almost! */

    free(pBuffer);                      /* Finally, free the buffer memory  */
    }

/*
 * Should this be a seek to the last read position ??
 */

if(pFile->iFlags & F_READING)
    lseek(iFileHndl, 0L, 0);
}


/****************************************************************************
 *                                                                          *
 * Jump to start of previously written file and prepare for reading.        *
 *                                                                          *
 ****************************************************************************/

void t_rewind(int iIndex)

{
FILEINFO *pFile;

pFile = &IOFiles[iIndex];               /* Get address of file info block   */
if(pFile->iFlags & F_READING)           /* Panic if rewind whilst reading   */
    fatal("t_rewind called twice");

if(pFile->iBufOffset)                   /* Flush any remaining write data   */
    t_wfull(pFile);

if(pFile->iFlags & F_FILEIO)            /* Do physical seek to BOF if disk  */
    lseek(pFile->iHandle, 0L, 0);

pFile->iFlags |= F_READING;             /* Mark file as reading now         */
if(pFile->lFileLen)                     /* Finally, if file is non empty we */
    t_rfill(pFile);                     /* pre fetch first block            */
}


/****************************************************************************
 *                                                                          *
 * Write a block of data to a temporary file                                *
 *                                                                          *
 ****************************************************************************/

void t_write(int iIndex, char *pcBuff, int iCount)

{
FILEINFO *pFile;

pFile = &IOFiles[iIndex];                /* Get address of file info block  */
if(pFile->iFlags & F_READING)            /* Sanity check, see if operation  */
    fatal("writet called while READING"); /* is allowed at this time         */

while(iCount-- > 0)                      /* Write block out, byte at a time */
    t_putc(pFile, *pcBuff++);
}


/****************************************************************************
 *                                                                          *
 * Write a single byte to a temporary file, flushing buffer as required.    *
 *                                                                          *
 ****************************************************************************/

static void t_putc(FILEINFO *pFile, char cData)

{
if(pFile->iBufOffset >= BSIZE)              /* If buffer is full then flush */
    t_wfull(pFile);                         /* to disk or RAM as needed     */

pFile->acBuffer[pFile->iBufOffset++] = cData; /* Place character in buffer  */
pFile->lFileLen++;                            /* and mark up another one    */
}


/****************************************************************************
 *                                                                          *
 * Read a block of data from a temporary file                               *
 *                                                                          *
 ****************************************************************************/

void t_read(int iIndex, char *pcBuff, int iCount)

{
FILEINFO *pFile;

pFile = &IOFiles[iIndex];               /* Get address of file info block   */
if((pFile->iFlags & F_READING) == 0)    /* Sanity check, see if operation   */
    fatal("read called while WRITING"); /* is allowed at this time          */

while(iCount-- > 0)                     /* Fetch the vblock of data one     */
    *pcBuff++ = t_getc(pFile);          /* byte at a time                   */
}


/****************************************************************************
 *                                                                          *
 * Fetch a single byte from a temporary file, prefetching blocks as needed  *
 *                                                                          *
 ****************************************************************************/

static char t_getc(FILEINFO *pFile)

{
if(pFile->lReadPtr >= pFile->lFileLen)  /* Catch attempt to read past EOF   */
    fatal("b-file read past end");

if(pFile->iBufOffset >= BSIZE)          /* Fetch next block if the current  */
    t_rfill(pFile);                     /* one is empty                     */

pFile->lReadPtr++;                      /* Advance read pointer by one      */

return(pFile->acBuffer[pFile->iBufOffset++]);  /* Return the requested char */
}

/****************************************************************************
 *                                                                          *
 * Flush a temporary files working buffer either to disk or to RAM          *
 *                                                                          *
 ****************************************************************************/

static void t_wfull(FILEINFO *pFile)

{
int iBytes;
struct buf *pBuffer;

iBytes = pFile->iBufOffset;         /* See how much data is in the buffer   */
again:
if(pFile->iFlags & F_FILEIO)        /* Is it a disk file or a RAM file      */
    {
    /*
     * Write block to disk and reset buffer count, panic if write fails
     */

    if(write(pFile->iHandle, pFile->acBuffer, iBytes) != iBytes)
        fatal("write error");

    pFile->iBufOffset = 0;
    }
else
    {
    /*
     * Try to allocate a RAM buffer to add to the files chain of buffers.
     * If we can't allocate enough memory then we try flushing a temporary
     * file to disk (possibly this one!) and try again. If all else fails
     * panic and exit the program with an error message.
     */

    pBuffer = (struct buf *)malloc(sizeof(struct buf));
    if(pBuffer == NULL)
        {
        if(t_frees())
            goto again;

        fatal("out of memory");
        }

    memcpy(pBuffer->acBuffer, pFile->acBuffer, iBytes); /* copy buffer over */
    t_enqueue(pFile, pBuffer);                          /* add to buf chain */
    pFile->iBufOffset = 0;                              /* and reset offset */
    }
}


/****************************************************************************
 *                                                                          *
 * Add a memory block to the end of a RAM temporary file chain.             *
 *                                                                          *
 ****************************************************************************/

static void t_enqueue(FILEINFO *pFile, struct buf *pBuffer)

{
pBuffer->pNext = 0;                 /* Mark the buffer as the end of chain  */
if(pFile->pLastBuffer)              /* If there is an existing end of chain */
    {
    pFile->pLastBuffer->pNext = pBuffer; /* Link current end of chain to    */
    pFile->pLastBuffer = pBuffer;        /* new block and update EOC ptr    */
    }
else                                                    /* Else block is    */
    pFile->pFirstBuffer = pFile->pLastBuffer = pBuffer; /* first and last   */
}


/****************************************************************************
 *                                                                          *
 * Remove current first block from list                                     *
 *                                                                          *
 ****************************************************************************/

static struct buf *t_dequeue(FILEINFO *pFile)

{
struct buf *pBuffer;

pBuffer = pFile->pFirstBuffer;              /* Get address of first buffer  */
if(!pBuffer)                                /* Panic if list is empty       */
    fatal("internal buffer list error");

pFile->pFirstBuffer = pBuffer->pNext;       /* Unlink first buf from chain  */
if(pFile->pFirstBuffer == 0)                /* If chain now empty clear out */
    pFile->pLastBuffer = 0;                 /* the last buffer pointer      */

return pBuffer;                             /* Return address of freed buf  */
}


/****************************************************************************
 *                                                                          *
 * Fetch a block of data for reading from a temporary file                  *
 *                                                                          *
 ****************************************************************************/

static void t_rfill(FILEINFO *pFile)

{
struct buf *pBuffer;
int  iCount;
long lTemp;

if(pFile->iFlags & F_FILEIO)            /* Check for disk based file        */
    {
    lTemp = pFile->lFileLen - pFile->lReadPtr;  /* Calc bytes remaining     */
    if(lTemp > BSIZE)                           /* If more than a buf full  */
        iCount = BSIZE;                         /* just fetch a buffer full */
    else
	iCount = lTemp;                         /* Otherwise fetch them all */

    if(read(pFile->iHandle, pFile->acBuffer, iCount) != iCount) /* Panic if */
        fatal("read error");                                    /* bad read */
    }
else                                    /* It's a RAM based temporary file  */
    {
    pBuffer = t_dequeue(pFile);                         /* Pluck block from */
    memcpy(pFile->acBuffer, pBuffer->acBuffer, BSIZE);  /* chain and then   */
    free(pBuffer);                                      /* free up memory   */
    }

pFile->iBufOffset = 0;                  /* Point to beginning of work buf   */
}
