/*
 * Copyright (c) 1991 by Sozobon, Limited.  Author: Johann Ruegg
 *
 * Permission is granted to anyone to use this software for any purpose
 * on any computer system, and to redistribute it freely, with the
 * following restrictions:
 * 1) No charge may be made other than reasonable charges for reproduction.
 * 2) Modified versions must be clearly marked as such.
 * 3) The authors are not responsible for any harmful consequences
 *    of using this software, even if they result from defects in it.
 */

#include <stdio.h>
#include <io.h>
#include <fcntl.h>
#include <sys\stat.h>
#include "syms.h"
#include "structs.h"
#include "global.h"

#ifndef UNIXHOST
#define SWAPW(a,b)
#define SWAPL(a,b)
#else
#define SWAPW(a,b)  swapw(a,b)
#define SWAPL(a,b)  swapl(a,b)
#endif
extern OFILEINFO *ohead;
long textsize, datasize, bsssize, comsize;
extern int mflag, symout, only8, vflag;
extern char *oname;
extern int tb, db, trelb, drelb;

#define LBUFSZ 1000
#define LBUFCNT (LBUFSZ/2)
short *lbuf, *rbuf;

int ofd;
unsigned long ulProgCounter;

static void calc_sizes(void);
static void sec_offs(void);
static void see_map(void);
static void endsyms(void);
static void wrhdr(void);
static void rewinds(void);
static void mkbufs(void);
static void flink(OFILEINFO *op, int dat);
static void link(OFILEINFO *op, int iSize, short *pData, short *pRelInfo);

/****************************************************************************
 *                                                                          *
 * Build the target file from the assembled peices                          *
 *                                                                          *
 ****************************************************************************/

void pass2(void)

{
OFILEINFO *pRover;

if(ohead == NULL)                           /* Complain if nothing to build */
    fatal("no object files included");

chk_undefs();               /* Make sure all references have been satisfied */
calc_sizes();               /* Calculate code, data and bss segment sizes   */
sec_offs();                 /* Adjust segment positions for each obj file   */

if(mflag)                   /* Produce linkage map listing if required      */
    see_map();

adjsyms();                  /* Fix up offsets for the defined symbols       */

comsize = mkcomm(textsize + datasize + bsssize); /* Put commons at bss end  */

endsyms();                  /* Add in linker produced segment end symbols   */

ofd = open(oname, O_WRONLY|O_BINARY);   /* Open output file for writing             */
if(ofd < 0)                                          /* Failed to open it   */
    ofd = open(oname, O_CREAT|O_TRUNC|O_BINARY, S_IREAD|S_IWRITE|S_IEXEC); /* So try to create it */

if(ofd < 0)                             /* If all else fails, give up       */
    fatals("Can't create", oname);

wrhdr();                        /* Write out the executable file header     */
rewinds();                      /* Step back to the beginning of temp files */
relstart();                     /* Initalise relocation variables           */
mkbufs();                       /* Set up buffers                           */

ulProgCounter = 0L;
for(pRover = ohead; pRover; pRover = pRover->pNext) /* Handle code segment  */
    flink(pRover, 0);                               /* first                */

for(pRover = ohead; pRover; pRover = pRover->pNext) /* Then take care of    */
    flink(pRover, 1);                               /* the data segment     */

if(symout)                          /* Write out symbol table if requested  */
    wrsyms(ofd);

relend();                           /* Finally write out relocation info    */
}


/****************************************************************************
 *                                                                          *
 * Allocate working buffers for output processing                           *
 *                                                                          *
 ****************************************************************************/

static void mkbufs(void)

{
rbuf = (short *)mmalloc(LBUFSZ + 4);            /* Relocatioon info buffer  */
lbuf = (short *)mmalloc(LBUFSZ + 4);            /* Linkage data buffer      */
}

#ifndef MINIX

char fill[HDRFILL];

/****************************************************************************
 *                                                                          *
 * Generate and write header for execuatble target file - CP/M-68K format   *
 *                                                                          *
 ****************************************************************************/

static void wrhdr(void)

{
short   iMagic;
LENINFO Lengths;

iMagic = OMAGIC;                        /* Write magic number with correct  */
SWAPW((char *)&iMagic, 1);              /* byte order                       */
wwrite(ofd, (char *)&iMagic, 2);

Lengths.tsize = textsize;               /* Fill in segment sizes            */
Lengths.dsize = datasize;
Lengths.bsize = bsssize + comsize;

if(symout)                              /* Set number of symbols correctly  */
    Lengths.syms = countsyms();
else
    Lengths.syms = 0;

SWAPL((char *)&Lengths, sizeof(Lengths) / 4);   /* Ensure byte order is ok  */
wwrite(ofd, (char *)&Lengths, sizeof(Lengths)); /* Write lengths out        */
wwrite(ofd, (char *)fill, HDRFILL);             /* Write out filler bytes   */
}

#else

struct mhdr
    {
    long magic, magic2;
    long tsize, dsize, bsize;
    long fill;
    long totsize;
    long syms;
    } mh;

#define MMAGIC 0x4100301L
#define MMAGIC2 0x20
#define MINSTK 4096

extern long mstack;


/****************************************************************************
 *                                                                          *
 * Generate and write header for execuatble target file - Minix format      *
 *                                                                          *
 ****************************************************************************/

wrhdr()
{
mh.magic  = MMAGIC;
mh.magic2 = MMAGIC2;
mh.tsize  = textsize;
mh.dsize  = datasize;
mh.bsize  = bsssize + comsize;

if(symout)
    mh.syms = countsyms();
else
    mh.syms = 0;

if(mstack <= 0)
    mstack = 0x10000L - (mh.dsize + mh.bsize);

if(mstack < MINSTK)
    mstack = MINSTK;

mh.totsize = mstack + mh.tsize + mh.dsize + mh.bsize;
SWAPL(&mh, sizeof(mh) / 4);
wwrite(ofd, &mh, sizeof(mh));
}

#endif


/****************************************************************************
 *                                                                          *
 * Rewind temporary files to beginning                                      *
 *                                                                          *
 ****************************************************************************/

static void rewinds(void)

{
t_rewind(tb);                                       /* Code segment         */
t_rewind(db);                                       /* Data segment         */
t_rewind(trelb);                                    /* Code relocation info */
t_rewind(drelb);                                    /* Data relocation info */
}


/****************************************************************************
 *                                                                          *
 * Add some special symbols to indicate segment ends                        *
 *                                                                          *
 ****************************************************************************/

static void endsyms(void)

{
end_sym("_etext", textsize, (F_TEXT | F_GLBL | F_DEF));
end_sym("_edata", textsize + datasize, (F_DATA | F_GLBL | F_DEF));
end_sym("_end",   textsize + datasize + bsssize + comsize, (F_BSS | F_GLBL | F_DEF));
}


/****************************************************************************
 *                                                                          *
 * Tot up the sizes of the different segments                               *
 *                                                                          *
 ****************************************************************************/

static void calc_sizes(void)

{
OFILEINFO *pRover;

pRover = ohead;                         /* Start at root of list and work   */
while(pRover)                           /* our way forwards to the end      */
    {
    pRover->tbase = textsize;           /* For each object file we record   */
    pRover->dbase = datasize;           /* where in the segment the current */
    pRover->bbase = bsssize;            /* object file lies                 */

    textsize += pRover->oh.tsize;       /* We then add the sizes of each    */
    datasize += pRover->oh.dsize;       /* segment in the object file to    */
    bsssize  += pRover->oh.bsize;       /* the running totals               */

    pRover = pRover->pNext;             /* Finally select next list entry   */
    }
}


/****************************************************************************
 *                                                                          *
 * Adjust the segment offsets assuming a contigious code,data and bss model *
 *                                                                          *
 ****************************************************************************/

static void sec_offs(void)

{
OFILEINFO *pRover;

pRover = ohead;                         /* Start at root of list and work   */
while(pRover)                           /* our way forwards to the end      */
    {
    pRover->dbase += textsize;             /* Data seg follows code seg     */
    pRover->bbase += textsize + datasize;  /* bss seg follows code and data */

    pRover = pRover->pNext;
    }
}


/****************************************************************************
 *                                                                          *
 * Produce linkage map listing                                              *
 *                                                                          *
 ****************************************************************************/

static void see_map(void)

{
OFILEINFO *pRover;

pRover = ohead;                         /* Start at root of list and work   */
while(pRover)                           /* our way forwards to the end      */
    {
    printf("%s", pRover->finfo->sName);         /* Object or lib file name  */
    if(pRover->sFileName[0])                    /* Print object file name   */
        printf("(%.14s)", pRover->sFileName);   /* if it comes from library */

    /*
     * Print size and offset of each segment used in object file
     */

    printf(":\t");
    if(pRover->oh.tsize)
        printf("T %lx @%lx  ", pRover->oh.tsize, pRover->tbase);

    if(pRover->oh.dsize)
        printf("D %lx @%lx  ", pRover->oh.dsize, pRover->dbase);

    if(pRover->oh.bsize)
        printf("B %lx @%lx  ", pRover->oh.bsize, pRover->bbase);

    putchar('\n');                              /* Finally end the line     */

    pRover = pRover->pNext;
    }
}


/****************************************************************************
 *                                                                          *
 * Link file chunk by chunk                                                 *
 *                                                                          *
 ****************************************************************************/

static void flink(OFILEINFO *pCurObj, int iDataSeg)

{
long  lSegSize;
int   iCount;
int   HndlContents;
int   HndlRelocInfo;
char *sSegName;

/*
 * Exit early if segment does not contain anything
 */

lSegSize = iDataSeg ? pCurObj->oh.dsize : pCurObj->oh.tsize;
if(lSegSize == 0)
    return;

/*
 * Pick the input files and section name to use for the segment type
 */

if(iDataSeg)
    {
    HndlContents  = db;
    HndlRelocInfo = drelb;
    sSegName      = "data";
    }
else
    {
    HndlContents  = tb;
    HndlRelocInfo = trelb;
    sSegName      = "text";
    }

if(vflag)                               /* Print some details if requested  */
    {
    printf("load %s from %s", sSegName, pCurObj->finfo->sName);
    if(pCurObj->sFileName[0])
        printf("(%.14s)", pCurObj->sFileName);

    putchar('\n');
    }

while(lSegSize > 0)                     /* Loop 'till complete segment done */
    {
    if(lSegSize > LBUFSZ)               /* Request either a full buffer or  */
        iCount = LBUFSZ;                /* whatever is left over            */
    else
        iCount = lSegSize;

    t_read(HndlRelocInfo, (char *)rbuf, iCount);    /* Get relocation info  */
    SWAPW((char *)rbuf, iCount / 2);                /* Correct byte order   */

    /*
     * If we are not at the end othe segment and the last item in the
     * current block is the first word of a long word reference then we
     * must read in second word of the relocation information now.
     */

    if(iCount < lSegSize && rbuf[LBUFCNT - 1] == 5)
        {
        t_read(HndlRelocInfo, (char *)&rbuf[LBUFCNT], 2);
        SWAPW((char *)&rbuf[LBUFCNT], 1);
        iCount += 2;                                /* Remember extra bytes */
        }

    t_read(HndlContents, (char *)lbuf, iCount);     /* Get segment contents */
    link(pCurObj, iCount, lbuf, rbuf);              /* Link current chunk   */
    wwrite(ofd, (char *)lbuf, iCount);              /* Write seg contents   */
    relout(iCount, rbuf);                           /* Save relocation info */

    lSegSize -= iCount;                             /* Ready for next loop  */
    }
}


/****************************************************************************
 *                                                                          *
 * Link a chunk of code or data                                             *
 *                                                                          *
 ****************************************************************************/

static void link(OFILEINFO *pCurObj, int iSize, short *pData, short *pRelInfo)

{
int   iWords;
long  lValue;
unsigned short int uiValue;
short iReloc;
int   iSkip;
#ifdef UNIXHOST
long  lTemp;
#endif

iWords = iSize / 2;                     /* See how many 16 bit words to do  */
while(iWords--)
    {
    iReloc = *pRelInfo++;               /* Fetch a relocation entry         */
    switch(iReloc & 7)                  /* and decide whjat to do with it   */
        {
        case 7:                         /* Start of instruction word        */
        case 0:                         /* Absolute reference               */
            pData++;                    /* These don't require relocation   */
            ulProgCounter += 2;
            break;                      /* so just skip over them           */

        case 5:                         /* First word of a long value       */
            iReloc = *pRelInfo;         /* Second reloc value holds type    */
            
            /*
             * If the bottom 3 bits are == 4 then this is a reference to a
             * symbol and we go and sort this out elsewhere
             */

            if((iReloc & 7) == 4)
                lValue = findx(iReloc, pCurObj->pSymbols, pCurObj->havex, pRelInfo);
            else
                {
                iSkip = FALSE;                  /* Assume a fixup required  */
                switch(iReloc)
                    {
                    case 0:                     /* Absolute long, no fixup  */
                        iSkip = TRUE;
                        break;

                    case 1:                      /* Data seg relative long  */
                        lValue = pCurObj->dbase;
                        break;

                    case 2:                      /* Code seg relative long  */
                        lValue = pCurObj->tbase;
                        break;

                    case 3:                      /* BSS seg relative long   */
                        lValue = pCurObj->bbase;
                        break;

                    default:                           /* None of the above */
                        fatal("Bad relocation value");
                    }
                }

            if(iSkip == FALSE)                  /* See if fixup is required */
                {
                #ifndef UNIXHOST
                *(long *)pData += lValue;       /* Add fixup to data        */
                #else                           /* If in little endian      */
                lTemp = crossl((char *)pData);  /* world it is trickier     */
                lTemp += lValue;
                *(long *)pData = crossl((char *)&lTemp);
                #endif
                }

            pData += 2;                         /* Skip over long (2 words) */
            ulProgCounter += 4;
            pRelInfo++;                         /* Skip over second reloc   */
            iWords--;                           /* and drop from count also */
            break;

        case 6:                                 /* 16 bit pc Relative       */
            lValue = findx(iReloc, pCurObj->pSymbols, pCurObj->havex, &uiValue);
            lValue -= ulProgCounter;
            uiValue = (unsigned int)lValue;
            SWAPW((char *)&uiValue, 1);
            *(unsigned int *)pData = uiValue;
            pData += 2;
            ulProgCounter += 2;
            *(pRelInfo-1) = 0;
            break;

        default:                                /* We have some bad news    */
            fatal("bad relocation value");
        }
    }
}
