/* ICMP-related user commands
 *
 * Modifications:
 * 08 Nov 91    HT1     hash_ping returns _positive_ value
 */

#include <stdio.h>
#include "global.h"
#include "icmp.h"
#include "mbuf.h"
#include "netuser.h"
#include "iface.h"
#include "internet.h"
#include "ip.h"
#include "timer.h"
#include "ping.h"
#include "icmpcmd.h"

/* local prototypes */
static pingem(int32 dest,int16 seq);
static int16 hash_ping(int32 dest);
static void del_ping(struct ping *pp);
static struct ping *add_ping(int32 dest);

int
doicmpstat(void)
{
        extern struct icmp_errors icmp_errors;
        extern struct icmp_stats icmp_stats;
        extern char *icmptypes[];
        register int i;

        printf("ICMP: chksum err %u no space %u icmp %u bdcsts %u\n",
         icmp_errors.checksum,icmp_errors.nospace,icmp_errors.noloop,
         icmp_errors.bdcsts);
        printf("type  rcvd  sent\n");
        for(i=0;i<ICMP_TYPES;i++){
                if(icmp_stats.input[i] == 0 && icmp_stats.output[i] == 0)
                        continue;
                printf("%-6u%-6u%-6u",i,icmp_stats.input[i],
                        icmp_stats.output[i]);
                if(icmptypes[i] != NULLCHAR)
                        printf("  %s",icmptypes[i]);
                printf("\n");
        }
        return 0;
}

/* Hash table list heads */
struct ping *ping[PMOD];

/* 1 Hz counter for generating seq numbers */
static int16 iclk;

/* Increment counter -- called by low level clock tick */
void 
icmpclk(void)
{
        iclk++;
}

/* Send ICMP Echo Request packets */
int
doping(int argc,char *argv[])
{
        int32 dest;
        struct ping *pp1;
        register struct ping *pp;
        int16 hval;
        int i;

        if(argc < 2){
                printf("Host                Sent    Rcvd   %%   Avg RTT\n");
                for(i=0;i<PMOD;i++){
                        for(pp = ping[i];pp != NULLPING;pp = pp->next){
                                printf("%-16s",inet_ntoa(pp->remote));
                                printf("%8ld%8ld",pp->count,pp->echoes);
                                printf("%4ld%10ld\n",
                                 (long)pp->echoes * 100 / pp->count,
                                 pp->echoes != 0 ? 
                                 (long)pp->ttotal * MSPTICK / pp->echoes : 0);
                        }
                }
                return 0;
        }
        if(strcmp(argv[1],"clear") == 0){
                for(i=0;i<PMOD;i++){
                        for(pp = ping[i];pp != NULLPING;pp = pp1){
                                pp1 = pp->next;
                                del_ping(pp);
                        }
                }
                return 0;
        }
        if((dest = resolve(argv[1])) == 0){
                printf("Host %s unknown\n",argv[1]);
                return 1;
        }
        /* See if dest is already in table */
        hval = hash_ping(dest);
        for(pp = ping[hval]; pp != NULLPING; pp = pp->next){
                if(pp->remote == dest){
                        break;
                }
        }
        if(argc > 2){
                /* Inter-ping time is specified; set up timer structure */
                if(pp == NULLPING)
                        pp = add_ping(dest);
                pp->timer.start = atoi(argv[2]);
                pp->timer.func = ptimeout;
                pp->timer.arg = (char *)pp;
                pp->remote = dest;
                start_timer(&pp->timer);
                pp->count++;
        }
        pingem(dest,iclk);
        return 0;
}

/* Called by ping timeout */
void
ptimeout(char *p)
{
        register struct ping *pp;

        /* Send another ping */
        pp = (struct ping *)p;
        pp->count++;
        pingem(pp->remote,iclk);
        start_timer(&pp->timer);
}

/* Send ICMP Echo Request packet */
static
pingem(int32 dest,int16 seq)
{
        struct mbuf *bp;
        struct icmp icmp;
        extern struct icmp_stats icmp_stats;

        icmp_stats.output[ECHO]++;
        icmp.type = ECHO;
        icmp.code = 0;
        icmp.args.echo.seq = seq;
        icmp.args.echo.id = 0;
        bp = htonicmp(&icmp,NULLBUF);
        ip_send(ip_addr,dest,ICMP_PTCL,0,0,bp,len_mbuf(bp),0,0);
        return 0;
}

/* Called with incoming Echo Reply packet */
void
echo_proc(int32 source,int32 dest,struct icmp *icmp)
{
        register struct ping *pp;
        int16 hval;
        int16 rtt;

        rtt = iclk - icmp->args.echo.seq;
        hval = hash_ping(source);
        for(pp = ping[hval]; pp != NULLPING; pp = pp->next)
                if(pp->remote == source)
                        break;
        if(pp == NULLPING){
                printf("%s: echo reply id %d seq %d, %lu ms\n",
                 inet_ntoa(source),
                 icmp->args.echo.id,icmp->args.echo.seq,
                 (long)rtt * MSPTICK);
                 fflush(stdout);
        } else {
                /* Repeated poll, just keep stats */
                pp->ttotal += rtt;
                pp->echoes++;
        }
}

static
int16
hash_ping(int32 dest)
{
        int16 hval;

        hval = (hiword(dest) ^ loword(dest)) % PMOD;
        return abs(hval);       /* HT1: abs */
}
/* Add entry to ping table */
static
struct ping *
add_ping(int32 dest)
{
        struct ping *pp;
        int16 hval;

        pp = (struct ping *)calloc(1,sizeof(struct ping));
        if(pp == NULLPING)
                return NULLPING;

        hval = hash_ping(dest);
        pp->prev = NULLPING;
        pp->next = ping[hval];
        if(pp->next != NULLPING)
                pp->next->prev = pp;
        ping[hval] = pp;
        return pp;
}

/* Delete entry from ping table */
static void
del_ping(struct ping *pp)
{
        int16 hval;

        stop_timer(&pp->timer);

        if(pp->next != NULLPING)
                pp->next->prev = pp->prev;
        if(pp->prev != NULLPING) {
                pp->prev->next = pp->next;
        } else {
                hval = hash_ping(pp->remote);
                ping[hval] = pp->next;
        }
        free((char *)pp);       
}

