MODULE RamDisk ;

(* --------------------------------------------------------------------------

               RamDisk : Ram disk driver for TDI Modula-2/ST

               Based on various "C" public domain versions

   This module is included to demonstrate the use of several low-level
   features of Modula-2.

   It is desinged to be a TOS type file, and not use many libraries because
   it has to remain resident in RAM. After linking RamDisk, rename the
   file RAMDISK.PRG to RAMDISK.TOS using the file menu "Show Info" option.

   --------------------------------------------------------------------------*)

(* ---------------------------------------------------------------------- *)
(* (c) Copyright 1986, 1987  Modula-2 Software Ltd.  All Rights Reserved. *)
(* ---------------------------------------------------------------------- *)
(* (c) Copyright 1986, 1987  TDI Software, Inc.      All Rights Reserved. *)
(* ---------------------------------------------------------------------- *)

(* 
        Version :       1.00a    12-Mar-86    PRC M2SW
                        Original version

*)

(*$T-*)(*$S-*)(*$A+*)


FROM SYSTEM IMPORT ADDRESS, ADR, SETREG, CODE, REGISTER ;
FROM GEMX   IMPORT BasePageAddress, BasePageType ;
FROM BIOS   IMPORT BPB ;
FROM XBIOS  IMPORT SuperExec ;
FROM GEMDOS IMPORT ConWS, ConRS, ConOut, ConIn, Alloc, TermRes ;
FROM Conversions IMPORT ConvertFromString, ConvertToString ;



CONST
  RamDiskDeviceID = 3 ;         (* drive D *)
  BytesPerCluster = 1024 ;

  (* Because we dont know what registers the BIOS is using we must use
     the following opcodes to save the registers *)
  MOVEMDEC = 48E7H ;    (* 68000 opcode for MOVEM <regs>,-(A7) *)
  MOVEMINC = 4CDFH ;    (* 68000 opcode for MOVEM (A7)+,<regs> *)
  SAVEREGS = 07FFCH ;   (* Registers D1..A5 for DEC *)
  RESTREGS = 03FFEH ;   (* Registers D1..A5 for INC *)
  RTS = 04E75H ;        (* 68000 return from subroutine opcode *)

TYPE
  (* Procedure types to mimic correct sequence for "C" BIOS routines *)

  CBPBProc     = PROCEDURE ( CARDINAL ) ;
  CMediaChProc = PROCEDURE ( CARDINAL ) ;
  CRWAbsProc   = PROCEDURE ( CARDINAL, CARDINAL, CARDINAL, ADDRESS, CARDINAL );

VAR

  (* BIOS variables : These can only be accessed with the 68000 in supervisor
     mode. The Modula-2 language allows you to fix the location of variables *)

  HDBPB     [0472H] : ADDRESS ;       (* hard disk get Bios Parameter Block *)
  HDRWAbs   [0476H] : ADDRESS ;       (* hard disk read/write abs *)
  HDMediaCh [047EH] : ADDRESS ;       (* hard disk media change *)
  DriveBits [04C4H] : BITSET ;        (* disk drives present map *)

  (* The following are saved copies of the BIOS variables so that the real
     hard disk routines can be called if a hard disk access is requested. *)

  SaveHDBPB      : CBPBProc ;     (* hard disk get Bios Parameter Block *)
  SaveHDRWAbs    : CRWAbsProc ;   (* hard disk read/write abs *)
  SaveHDMediaCh  : CMediaChProc ; (* hard disk media change *)

  (* RAM disk control *)

  RamDiskSize : LONGCARD ;    (* size in bytes *)
  RamDiskBuffer : ADDRESS ;   (* address of RAM disk *)
  RamDiskBPB  : BPB ;         (* BIOS Parameter block for RAM disk *)

PROCEDURE MoveMemory ( From, To : ADDRESS ; Bytes : LONGCARD ) ;
(* This routine shows how time critical portions of code can be optimised to
   run faster. It relys on the code generation rules of the compiler which 
   can be checked by dis-assembling the link file with DecLnk.*)

CONST
  MOVEB = 12D8H ;       (*      MOVE.B  (A0)+,(A1)+     *)
  MOVEL = 22D8H ;       (*      MOVE.L  (A0)+,(A1)+     *)
  A0    = 0+8 ;         (* register A0 *)
  A1    = 1+8 ;         (* register A1 *)

BEGIN
  SETREG(A0,From) ;             (* load From pointer into A0 *)
  SETREG(A1,To) ;               (* load To pointer into A1 *)
  
  IF ( ODD(From) OR ODD(To) ) THEN      (* must do bytes *)
    WHILE ( Bytes <> 0 ) DO
      CODE(MOVEB) ;
      DEC(Bytes) ;
    END ;
  ELSE (* even addresses so can do long moves *)
    WHILE ( Bytes > 3 ) DO
      CODE(MOVEL) ;
      DEC(Bytes,4) ;
    END ;
    WHILE ( Bytes <> 0 ) DO
      CODE(MOVEB) ;             (* clean up remainder *)
      DEC(Bytes) ;
    END ;
  END ;
END MoveMemory ;

(* The following procedures mimic the disk handling routines called by the
   BIOS. Their procedure declarations have been written to mimic the "C"
   calling sequence. *)

PROCEDURE RDRWAbs ( Device, RecordNum, SectorCount : CARDINAL ;
                    Buffer : ADDRESS ; Flag : CARDINAL ) ;
(* NB. It is assumed that GEMDOS wont call this routine with out of range
   parameters *)
CONST D0 = 0 ;
VAR numBytes : LONGCARD ;
BEGIN
  CODE(MOVEMDEC,SAVEREGS) ;     (* save registers on stack *)
  IF ( Device = RamDiskDeviceID ) THEN (* is RAM disk *)
    numBytes := LONGCARD(SectorCount) * 512 ;
    IF ( Flag = 0 ) OR ( Flag = 2 ) (* read *)  THEN
      MoveMemory( RamDiskBuffer + ADDRESS(RecordNum) * 512,
                   Buffer,numBytes) ;
      SETREG(D0,0) ;
    ELSIF ( Flag = 1 ) OR ( Flag = 3 ) THEN
      MoveMemory( Buffer,RamDiskBuffer + ADDRESS(RecordNum) * 512,numBytes) ;
      SETREG(D0,0) ;
    ELSE
      SETREG(D0,LONGINT(-15)) ;
    END ;
  ELSE (* not RAM Disk *)
    SaveHDRWAbs (Device,RecordNum,SectorCount,Buffer,Flag) ;
  END ;
  CODE(MOVEMINC,RESTREGS) ;     (* Restore registers from stack *)
END RDRWAbs ;

PROCEDURE RDMediaCh ( Device : CARDINAL ) ;
CONST D0 = 0 ;
BEGIN
  CODE(MOVEMDEC,SAVEREGS) ;     (* save registers on stack *)
  IF ( Device = RamDiskDeviceID ) THEN (* is RAM disk *)
    SETREG(D0,0) ;               (* "C" uses D0 as return location *)
  ELSE (* not RAM Disk *)
    SaveHDMediaCh(Device) ;
  END ;
  CODE(MOVEMINC,RESTREGS) ;     (* Restore registers from stack *)
END RDMediaCh ;

PROCEDURE RDBPB ( Device : CARDINAL ) ;
CONST D0 = 0 ;
BEGIN
  CODE(MOVEMDEC,SAVEREGS) ;     (* save registers on stack *)
  IF ( Device = RamDiskDeviceID ) THEN (* is RAM disk *)
    SETREG(D0,ADR(RamDiskBPB)) ;     (* "C" uses D0 as return location *)
  ELSE (* not RAM Disk *)
    SaveHDBPB(Device) ;
  END ;
  CODE(MOVEMINC,RESTREGS) ;     (* Restore registers from stack *)
END RDBPB ;
    
(* ----------------------------------------------------------------------- *)

PROCEDURE WriteString ( VAR s : ARRAY OF CHAR ) ;
BEGIN
  ConWS(s) ;
END WriteString ;

PROCEDURE WriteLn ;
BEGIN
  ConOut(15C) ; ConOut(12C) ;
END WriteLn ;

PROCEDURE WriteLine ( VAR s : ARRAY OF CHAR ) ;
BEGIN
  WriteString(s) ;
  WriteLn ;
END WriteLine ;

PROCEDURE WaitSpace ;
VAR ch : CHAR ;
BEGIN
  WriteLine("Type space to contine:") ;
  REPEAT
    ConIn(ch) ;
  UNTIL ( ch = ' ' ) ;
END WaitSpace ;

PROCEDURE ReadString ( VAR s : ARRAY OF CHAR ) ;
VAR i : CARDINAL ;
BEGIN
  i := 0 ;
  REPEAT
    ConIn(s[i]) ;
    INC(i) ;
  UNTIL (i>HIGH(s)) OR (s[i-1]<' ');
  s[i-1] := 0C ;
END ReadString ;


PROCEDURE Initialise () : BOOLEAN ;
(* returns TRUE is Ram disk is to be installed *)
CONST
  DeviceSlop = 7 + 2 ; (* num K used by GEMDOS on device = 7 dirs + 2 boot *)
VAR
  availableRAM : ADDRESS ; numStr : ARRAY [0..79] OF CHAR ;
  done : BOOLEAN ; ptr : POINTER TO LONGCARD ; size : LONGCARD ;
BEGIN
  WriteLine("RamDisk: Modula-2/ST RAM disk driver 1.00a");
  WriteLine("This module installs a new disk device 'D' as a RAM disk.");
  WriteLine("You will have to use the GEM 'Install Drive' menu option");
  WriteLine("to access the RAM disk from the GEM desktop.") ;
  WriteLn ;
  (* Determine available RAM by using GEMDOS Alloc function *)
  Alloc(0FFFFFFFFH,availableRAM) ;
  (* scale to multiple of cluster size *)
  availableRAM := availableRAM DIV BytesPerCluster * BytesPerCluster ;
  WriteString("Available RAM for RamDisk = ") ;
  ConvertToString(LONGCARD(availableRAM DIV 1024-DeviceSlop)
                  ,10,FALSE,numStr,done) ;
  WriteString(numStr) ; WriteLine("K."); 
  IF ( availableRAM < 32*1024 ) THEN (* Ram disk less than 16K not viable *)
    WriteLine("There is not enough RAM to build a viable RAM disk.") ;
    WaitSpace ;
    RETURN FALSE ;
  END ;
  WriteString("How many K RAM do you wish to allocate to the RAM disk ? ") ;
  ReadString(numStr) ;
  WriteLn ;
  ConvertFromString(numStr,10,FALSE,LONGCARD(availableRAM) DIV 1024,
                    RamDiskSize,done) ;
  IF (NOT done) OR ( RamDiskSize < 32 ) THEN
    WriteLine("Invalid RAM disk size.") ;
    WaitSpace ;
    RETURN FALSE ;
  END ;
  RamDiskBPB.numcl := CARDINAL(RamDiskSize); (* set number of data clusters *)
  RamDiskSize := (RamDiskSize+DeviceSlop) * 1024;
  Alloc(RamDiskSize,RamDiskBuffer) ;
  IF ( RamDiskBuffer = 0 ) THEN
    WriteLine("**ERROR: Unable to allocate RAM disk") ;
    WaitSpace ;
    RETURN FALSE ;
  END ;
  (* preset disk to zero *)
  ptr := RamDiskBuffer ; size := RamDiskSize ;
  REPEAT
    ptr^ := 0 ;
    (* Turning off range checking allows pointer maths *)
    (*$T-*) INC(ptr,4) ; (*$T=*)
    DEC(size,4) ;
  UNTIL ( size = 0 ) ;
  WITH RamDiskBPB DO
    recsiz := 512 ;
    clsiz := 2 ;
    clsizb := 2 * 512 ;
    rdlen := 7 ;
    fsiz := 5 ;
    fatrec := 6 ;
    datrec := 18 ;
    bflags := 1 ;
  END ;
  RETURN TRUE ;
END Initialise ;

(* The following compiler directive stops the compiler from generating the
   normal Modula-2 entry/exit code for the next procedure. This is needed as
   this routine is called in supervisor mode by the BIOS function to install
   the BIOS vectors. *)
(*$P- Stop entry/exit code for next procedure *)
PROCEDURE InstallVectors ;
BEGIN
  (* First save the current hard disk vectors *)
  SaveHDBPB := CBPBProc(HDBPB) ;
  SaveHDRWAbs := CRWAbsProc(HDRWAbs) ;
  SaveHDMediaCh := CMediaChProc(HDMediaCh) ;
  (* Now set the BIOS vectors to our routines *)
  HDBPB := ADDRESS(RDBPB) ;
  HDRWAbs := ADDRESS(RDRWAbs) ;
  HDMediaCh := ADDRESS(RDMediaCh) ;
  INCL(DriveBits,RamDiskDeviceID) ;           (* set new drive D *)
  CODE(RTS) ;                   (* code to return to calling BIOS function *)
END InstallVectors ;

BEGIN   (* body of module *)
  IF Initialise() THEN
    SuperExec(PROC(InstallVectors)) ; (* install the RAM disk *)
    WriteLine("RAM disk installed.") ;
    WaitSpace ;
    WITH BasePageAddress^ DO
      (* Tell GEMDOS to terminate us but let us remain in memory *)
      TermRes(CodeLen+BssLen+LONGCARD(CodeBase-ADDRESS(BasePageAddress)),0) ;
    END ;
  END ;
END RamDisk.
