/*
 * NOTE: This file only works if sizeof(int) == 2!
 * UNLESS: you have an ANSI compiler and use prototypes
 *
 * Copyright 1991,1992 Eric R. Smith. This file may be re-distributed
 * as long as this notice remains intact.
 */

#ifndef _filesys_h
#define _filesys_h

#define NAME_MAX 32
#define PATH_MAX 128

struct filesys;		/* forward declaration */
struct devdrv;		/* ditto */

typedef struct f_cookie {
	struct filesys *fs;	/* filesystem that knows about this cookie */
	unsigned int	dev;		/* device info (e.g. Rwabs device number) */
	unsigned int	aux;		/* extra data that the file system may want */
	long	index;		/* this+dev uniquely identifies a file */
} fcookie;

/* structure for opendir/readdir/closedir */
typedef struct dirstruct {
	fcookie fc;		/* cookie for this directory */
	unsigned int	index;		/* index of the current entry */
	unsigned int	flags;		/* flags (e.g. tos or not) */
#define TOS_SEARCH	0x01
	char	fsstuff[60];	/* anything else the file system wants */
				/* NOTE: this must be at least 45 bytes */
} DIR;

/* structure for getxattr */
typedef struct xattr {
	unsigned int	mode;
/* file types */
#define S_IFMT	0170000		/* mask to select file type */
#define S_IFCHR	0020000		/* BIOS special file */
#define S_IFDIR	0040000		/* directory file */
#define S_IFREG 0100000		/* regular file */
#define S_IFIFO 0120000		/* FIFO */
#define S_IMEM	0140000		/* memory region or process */
#define S_IFLNK	0160000		/* symbolic link */

/* special bits: setuid, setgid, sticky bit */
#define S_ISUID	04000
#define S_ISGID 02000
#define S_ISVTX	01000

/* file access modes for user, group, and other*/
#define S_IRUSR	0400
#define S_IWUSR 0200
#define S_IXUSR 0100
#define S_IRGRP 0040
#define S_IWGRP	0020
#define S_IXGRP	0010
#define S_IROTH	0004
#define S_IWOTH	0002
#define S_IXOTH	0001
#define DEFAULT_DIRMODE (0777)
#define DEFAULT_MODE	(0666)
	long	index;
	unsigned int	dev;
	unsigned int	reserved1;
	unsigned int	nlink;
	unsigned int	uid;
	unsigned int	gid;
	long	size;
	long	blksize, nblocks;
	int	mtime, mdate;
	int	atime, adate;
	int	ctime, cdate;
	int	attr;
	int	reserved2;
	long	reserved3[2];
} XATTR;

typedef struct fileptr {
	int				links;	/* number of copies of this descriptor */
	unsigned int	flags;	/* file open mode and other file flags */
	long			pos;	/* position in file */
	long			devinfo;	/* device driver specific info */
	fcookie			fc;		/* file system cookie for this file */
	struct devdrv 	*dev; 	/* device driver that knows how to deal with this */
	struct fileptr 	*next;	/* link to next fileptr for this file */
} FILEPTR;

/* lock structure */
struct flock {
	int l_type;			/* type of lock */
#define F_RDLCK		O_RDONLY
#define F_WRLCK		O_WRONLY
#define F_UNLCK		3
	int l_whence;			/* SEEK_SET, SEEK_CUR, SEEK_END */
	long l_start;			/* start of locked region */
	long l_len;			/* length of locked region */
	int l_pid;			/* pid of locking process
						(F_GETLK only) */
};

/* LOCK structure used by the kernel internally */

typedef struct ilock {
	struct flock l;
	struct ilock *next;
	long  reserved[4];
} LOCK;

typedef struct devdrv{
	long cdecl	(*mcopen)	(FILEPTR *f);
	long cdecl	(*mcwrite)	(FILEPTR *f,const char *buf, long bytes);
	long cdecl	(*mcread)	(FILEPTR *f, char *buf, long bytes);
	long cdecl	(*mclseek)	(FILEPTR *f, long where, int whence);
	long cdecl	(*mcioctl)	(FILEPTR *f, int mode, void *buf);
	long cdecl	(*mcdatime)	(FILEPTR *f, int *timeptr, int rwflag);
	long cdecl	(*mcclose)	(FILEPTR *f, int pid);
	long cdecl	(*mcselect)	(FILEPTR *f, long proc, int mode);
	void cdecl	(*mcunselect) (FILEPTR *f, long proc, int mode);
	long	reserved[3];	/* reserved for future use */
} DEVDRV;

typedef struct filesys {
	struct	filesys	*next;	/* link to next file system on chain */
	long	fsflags;
#define FS_KNOPARSE	0x01	/* kernel shouldn't do parsing */
#define FS_CASESENSITIVE	0x02	/* file names are case sensitive */
#define FS_NOXBIT	0x04	/* if a file can be read, it can be executed */
#define	FS_LONGPATH	0x08	/* file system understands "size" argument to */

	long	cdecl	(*root) (int drv, fcookie *fc);
	long	cdecl	(*lookup) (fcookie *dir, const char *name, fcookie *fc);
	long	cdecl	(*creat) (fcookie *dir, const char *name, unsigned mode,int attrib, fcookie *fc);
	DEVDRV * cdecl	(*getdev) (fcookie *fc, long *devspecial);
	long	cdecl	(*getxattr) (fcookie *fc, XATTR *xattr);
	long	cdecl	(*chattr) (fcookie *fc, int attr);
	long	cdecl	(*chown) (fcookie *fc, int uid, int gid);
	long	cdecl	(*chmode) (fcookie *fc, unsigned int mode);
	long	cdecl	(*mkdir) (fcookie *dir, const char *name, unsigned mode);
	long	cdecl	(*rmdir) (fcookie *dir, const char *name);
	long	cdecl	(*remove) (fcookie *dir, const char *name);
	long	cdecl	(*getname) (fcookie *root, fcookie *dir,char *pathname,int size);
	long	cdecl	(*rename) (fcookie *olddir, char *oldname,fcookie *newdir, const char *newname);
	long	cdecl	(*opendir) (DIR *dirh, int tosflag);
	long	cdecl	(*readdir) (DIR *dirh, char *nm, int nmlen, fcookie *fc);
	long	cdecl	(*rewinddir) (DIR *dirh);
	long	cdecl	(*closedir) (DIR *dirh);
	long	cdecl	(*pathconf) (fcookie *dir, int which);
	long	cdecl	(*dfree) (fcookie *dir, long *buf);
	long	cdecl	(*writelabel) (fcookie *dir, const char *name);
	long	cdecl	(*readlabel) (fcookie *dir, char *name, int namelen);
	long	cdecl	(*symlink) (fcookie *dir,const char *name,const char *to);
	long	cdecl	(*readlink) (fcookie *dir, char *buf, int len);
	long	cdecl	(*hardlink) (fcookie *fromdir,fcookie *todir,const char *fromname,	const char *toname);
	long	cdecl	(*fscntl) (fcookie *dir,const char *name,int cmd,long arg);
	long	cdecl	(*dskchng) (int drv);
	long	cdecl	(*release) (fcookie *);
	long	cdecl	(*dupcookie) (fcookie *new, fcookie *old);
} FILESYS;

/*
 * this is the structure passed to loaded file systems to tell them
 * about the kernel
 */

typedef long (*_LongFunc)();

struct kerinfo {
	short	maj_version;	/* kernel version number */
	short	min_version;	/* minor kernel version number */
	unsigned	default_perm;	/* default file permissions */
	short	reserved1;	/* room for expansion */

/* OS functions */
	_LongFunc	*bios_tab;	/* pointer to the BIOS entry points */
	_LongFunc	*dos_tab;	/* pointer to the GEMDOS entry points */

/* media change vector */
	void	cdecl (*drvchng) (unsigned);

/* Debugging stuff */
	void	cdecl (*trace) (const char *, ...);
	void	cdecl (*debug) (const char *, ...);
	void	cdecl (*alert) (const char *, ...);
	void	cdecl (*fatal) (const char *, ...);

/* memory allocation functions */
	void *	cdecl (*kmalloc) (long);
	void	cdecl (*kfree) (void *);
	void *	cdecl (*umalloc) (long);
	void	cdecl (*ufree) (void *);

/* utility functions for string manipulation */
	int	cdecl (*strnicmp) (const char *, const char *, int);
	int	cdecl (*stricmp) (const char *, const char *);
	char *	cdecl (*strlwr) (char *);
	char *	cdecl (*strupr) (char *);
	int	cdecl (*sprintf) (char *, const char *, ...);

/* utility functions for manipulating time */
	void	cdecl (*millis_time) (unsigned long, short *);
	long	cdecl (*unixtim) (unsigned, unsigned);
	long	cdecl (*dostim) (long);

/* utility functions for dealing with pauses, or for putting processes
 * to sleep
 */
	void	cdecl (*nap) (unsigned);
	void	cdecl (*sleep) (int que, long cond);
	void	cdecl (*wake) (int que, long cond);
	void	cdecl (*wakeselect) (long param);

/* file system utility functions */
	int	cdecl (*denyshare) (FILEPTR *, FILEPTR *);

/* reserved for future use */
	LOCK *	cdecl (*denylock) (LOCK *, LOCK *);
	long	res2[9];
};

#ifndef NEWWAY
typedef struct _context {
	long	regs[15];	/* registers d0-d7, a0-a6 */
	long	usp;		/* user stack pointer (a7) */
	short	sr;		/* status register */
	long	pc;		/* program counter */
	long	ssp;		/* supervisor stack pointer */
	long	term_vec;	/* GEMDOS terminate vector (0x102) */
/*
 * AGK: if running on a TT and the user is playing with the FPU then we
 * must save and restore the context. We should also consider this for
 * I/O based co-processors, although this may be difficult due to
 * possibility of a context switch in the middle of an I/O handshaking
 * exchange.
 */
	unsigned char	fstate[216];	/* FPU internal state */
	long	fregs[3*8];	/* registers fp0-fp7 */
	long	fctrl[3];	/* FPCR/FPSR/FPIAR */
	char	ptrace;		/* trace exception is pending */
	char	pad1;		/* junk */
	long	iar;		/* unused */
	short	res[4];		/* unused, reserved */
/*
 * Saved CRP and TC values.  These are necessary for memory protection.
 */

	/*crp_reg crp;*/			/* 64 bits */
	long	crp[2];
	/*tc_reg tc;*/			/* 32 bits */
	long	tc;
/*
 * AGK: for long (time-wise) co-processor instructions (FMUL etc.), the
 * FPU returns NULL, come-again with interrupts allowed primitives. It
 * is highly likely that a context switch will occur in one of these if
 * running a mathematically intensive application, hence we must handle
 * the mid-instruction interrupt stack. We do this by saving the extra
 * 3 long words and the stack format word here.
 */
	unsigned short	sfmt;	/* stack frame format identifier */
	short	internal[42];	/* internal state -- see framesizes[] for size */
} CONTEXT;

#define PROC_CTXTS	2
#define SYSCALL		0	/* saved context from system call	*/
#define CURRENT		1	/* current saved context		*/
#define PNAMSIZ		8	/* no. of characters in a process name */

#define DOM_TOS		0	/* TOS process domain */
#define DOM_MINT	1	/* MiNT process domain */
#define MIN_HANDLE	(-5)	/* minimum handle number		*/
#define MIN_OPEN	6	/* 0..MIN_OPEN-1 are reserved for system */
#define MAX_OPEN	32	/* max. number of open files for a proc	*/
#define	NSIG		31		/* number of signals recognized */
#define SSTKSIZE	8000	/* size of supervisor stack (in bytes) 	*/
#define ISTKSIZE	2000	/* size of interrupt stack (in bytes)	*/
#define STKSIZE		(ISTKSIZE + SSTKSIZE)

typedef struct proc {
	long	sysstack;		/* must be first		*/
	CONTEXT	ctxt[PROC_CTXTS];	/* must be second		*/

/* this is stuff that the public can know about */
	long	magic;			/* validation for proc struct	*/

/*	BASEPAGE *base;*/			/* process base page		*/
	long	 *base;			/* process base page		*/
	short	pid, ppid, pgrp;
	short	ruid;			/* process real user id 	*/
	short	rgid;			/* process real group id 	*/
	short	euid, egid;		/* effective user and group ids */

	unsigned
	unsigned memflags;		/* e.g. malloc from alternate ram */
	short	pri;			/* base process priority 	*/
	short	wait_q;			/* current process queue	*/

/* note: wait_cond should be unique for each kind of condition we might
 * want to wait for. Put a define below, or use an address in the
 * kernel as the wait condition to ensure uniqueness.
 */
	long	wait_cond;		/* condition we're waiting on	*/
					/* (also return code from wait) */

#define WAIT_MB		0x3a140001L	/* wait_cond for p_msg call	*/
#define WAIT_SEMA	0x3a140003L	/* wait_cond for p_semaphore	*/

	/* (all times are in milliseconds) */
	/* usrtime must always follow systime */
	unsigned long	systime;		/* time spent in kernel		*/
	unsigned long	usrtime;		/* time spent out of kernel	*/
	unsigned long	chldstime;		/* children's kernel time 	*/
	unsigned long	chldutime;		/* children's user time		*/

	unsigned long	maxmem;			/* max. amount of memory to use */
	unsigned long	maxdata;		/* max. data region for process */
	unsigned long	maxcore;		/* max. core memory for process */
	unsigned long	maxcpu;			/* max. cpu time to use 	*/

	short	domain;			/* process domain (TOS or UNIX)	*/

	short	curpri;			/* current process priority	*/
#define MIN_NICE -20
#define MAX_NICE 20

/* EVERYTHING BELOW THIS LINE IS SUBJECT TO CHANGE:
 * programs should *not* try to read this stuff via the U:\PROC dir.
 */

	char	name[PNAMSIZ+1];	/* process name			*/
/*	TIMEOUT	*alarmtim;	*/	/* alarm() event		*/
	long	*alarmtim;		/* alarm() event		*/
	short	slices;			/* number of time slices before this
					   process gets to run again */

	short	bconmap;		/* Bconmap mapping		*/
	FILEPTR *midiout;		/* MIDI output			*/
	FILEPTR *midiin;		/* MIDI input			*/
	FILEPTR	*prn;			/* printer			*/
	FILEPTR *aux;			/* auxiliary tty		*/
	FILEPTR	*control;		/* control tty			*/
	FILEPTR	*handle[MAX_OPEN];	/* file handles			*/

	unsigned char	fdflags[MAX_OPEN];	/* file descriptor flags	*/

	unsigned	num_reg;		/* number of memory regions allocated */
/*	MEMREGION **mem;	*/	/* allocated memory regions	*/
	long	 **mem;		/* allocated memory regions	*/
/*	virtaddr *addr;		*/	/* addresses of regions		*/
	long	*addr;			/* addresses of regions		*/

	unsigned long	sigpending;		/* pending signals		*/
	unsigned long	sigmask;		/* signals that are masked	*/
	unsigned long	sighandle[NSIG];	/* signal handlers		*/
	unsigned	sigflags[NSIG];		/* signal flags			*/
	unsigned long	sigextra[NSIG];		/* additional signals to be masked
					   on delivery 	*/
	char	*mb_ptr;		/* p_msg buffer ptr		*/
	long	mb_long1, mb_long2;	/* p_msg storage		*/
	long	mb_mbid;		/* p_msg id being waited for	*/
	short	mb_mode;		/* p_msg mode being waiting in	*/
	short	mb_writer;		/* p_msg pid of writer of msg	*/

	short	curdrv;			/* current drive		*/
	fcookie root[NUM_DRIVES];	/* root directories		*/
	fcookie	curdir[NUM_DRIVES];	/* current directory		*/

	long	usrdata;		/* user-supplied data		*/
	unsigned	umask;			/* file creation mask		*/

	DTABUF	*dta;			/* current DTA			*/
#define NUM_SEARCH	6		/* max. number of searches	*/
	DTABUF *srchdta[NUM_SEARCH];	/* for Fsfirst/next		*/
	DIR	srchdir[NUM_SEARCH];	/* for Fsfirst/next		*/
	long	srchtim[NUM_SEARCH];	/* for Fsfirst/next		*/
	
	DIR	*searches;		/* open directory searches	*/

	long	txtsize;		/* size of text region (for fork()) */

	long    cdecl		 (*criticerr) (long l); /* critical error handler	*/
	void	*logbase;		/* logical screen base		*/

	struct proc *ptracer;		/* process which is tracing this one */
	short	ptraceflags;		/* flags for process tracing */
	short	starttime;		/* time and date when process	*/
	short	startdate;		/* was started			*/

	void	*page_table;		/* rounded page table pointer	*/
	void	*pt_mem;		/* original kmalloc'd block for above */

	unsigned long	exception_pc;		/* pc at time of bombs		*/
	unsigned long	exception_ssp;		/* ssp at time of bomb (e.g. bus error)	*/
	unsigned long	exception_tbl;		/* table in use at exception time */
	unsigned	exception_mmusr;	/* result from ptest insn	*/
	unsigned long	exception_addr;		/* access address from stack	*/

	struct	proc *q_next;		/* next process on queue	*/
	struct 	proc *gl_next;		/* next process in system	*/
	char	stack[STKSIZE+4];	/* stack for system calls	*/
} PROC;
#endif

/* flags for open() modes */
#define O_RWMODE  	0x03	/* isolates file read/write mode */
#	define O_RDONLY	0x00
#	define O_WRONLY	0x01
#	define O_RDWR	0x02
#	define O_EXEC	0x03	/* execute file; used by kernel only */

#define O_SHMODE	0x70	/* isolates file sharing mode */
#	define O_COMPAT	0x00	/* compatibility mode */
#	define O_DENYRW	0x10	/* deny both read and write access */
#	define O_DENYW	0x20	/* deny write access to others */
#	define O_DENYR	0x30	/* deny read access to others */
#	define O_DENYNONE 0x40	/* don't deny any access to others */

#define O_NOINHERIT	0x80	/* this is currently ignored by MiNT */

#define O_NDELAY	0x100	/* don't block for i/o on this file */
#define O_CREAT		0x200	/* create file if it doesn't exist */
#define O_TRUNC		0x400	/* truncate file to 0 bytes if it does exist */
#define O_EXCL		0x800	/* fail open if file exists */

#define O_USER		0x0fff	/* isolates user-settable flag bits */

#define O_GLOBAL	0x1000	/* for Fopen: opens a global file handle */

/* kernel mode bits -- the user can't set these! */
#define O_TTY		0x2000	/* FILEPTR refers to a terminal */
#define O_HEAD		0x4000	/* FILEPTR is the master side of a fifo */
#define O_LOCK		0x8000	/* FILEPTR has had locking Fcntl's performed */


/* GEMDOS file attributes */

/* macros to be applied to FILEPTRS to determine their type */
#define is_terminal(f) (f->flags & O_TTY)

/* lseek() origins */
#define	SEEK_SET	0		/* from beginning of file */
#define	SEEK_CUR	1		/* from current location */
#define	SEEK_END	2		/* from end of file */

/* The requests for Dpathconf() */
#define DP_IOPEN	0	/* internal limit on # of open files */
#define DP_MAXLINKS	1	/* max number of hard links to a file */
#define DP_PATHMAX	2	/* max path name length */
#define DP_NAMEMAX	3	/* max length of an individual file name */
#define DP_ATOMIC	4	/* # of bytes that can be written atomically */
#define DP_TRUNC	5	/* file name truncation behavior */
#	define	DP_NOTRUNC	0	/* long filenames give an error */
#	define	DP_AUTOTRUNC	1	/* long filenames truncated */
#	define	DP_DOSTRUNC	2	/* DOS truncation rules in effect */

#define DP_MAXREQ	5	/* highest legal request */

/* Dpathconf and Sysconf return this when a value is not limited
   (or is limited only by available memory) */

#define UNLIMITED	0x7fffffffL

/* various character constants and defines for TTY's */
#define MiNTEOF 0x0000ff1a	/* 1a == ^Z */

/* defines for tty_read */
#define RAW	0
#define COOKED	0x1
#define NOECHO	0
#define ECHO	0x2
#define ESCSEQ	0x04		/* cursor keys, etc. get escape sequences */

/* constants for various Fcntl commands */
/* constants for Fcntl calls */
#define F_DUPFD		0		/* handled by kernel */
#define F_GETFD		1		/* handled by kernel */
#define F_SETFD		2		/* handled by kernel */
#	define FD_CLOEXEC	1	/* close on exec flag */

#define F_GETFL		3		/* handled by kernel */
#define F_SETFL		4		/* handled by kernel */
#define F_GETLK		5
#define F_SETLK		6

#define FSTAT		(('F'<< 8) | 0)	/* handled by kernel */
#define FIONREAD	(('F'<< 8) | 1)
#define FIONWRITE	(('F'<< 8) | 2)
#define TIOCGETP	(('T'<< 8) | 0)
#define TIOCSETP	(('T'<< 8) | 1)
#define TIOCSETN	TIOCSETP
#define TIOCGETC	(('T'<< 8) | 2)
#define TIOCSETC	(('T'<< 8) | 3)
#define TIOCGLTC	(('T'<< 8) | 4)
#define TIOCSLTC	(('T'<< 8) | 5)
#define TIOCGPGRP	(('T'<< 8) | 6)
#define TIOCSPGRP	(('T'<< 8) | 7)
#define TIOCFLUSH	(('T'<< 8) | 8)
#define TIOCSTOP	(('T'<< 8) | 9)
#define TIOCSTART	(('T'<< 8) | 10)
#define TIOCGWINSZ	(('T'<< 8) | 11)
#define TIOCSWINSZ	(('T'<< 8) | 12)
#define TIOCGXKEY	(('T'<< 8) | 13)
#define TIOCSXKEY	(('T'<< 8) | 14)

#define PPROCADDR	(('P'<< 8) | 1)
#define PBASEADDR	(('P'<< 8) | 2)
#define PCTXTSIZE	(('P'<< 8) | 3)

/* terminal control constants (tty.sg_flags) */
#define T_CRMOD		0x0001
#define T_CBREAK	0x0002
#define T_ECHO		0x0004
#define T_RAW		0x0010
#define T_TOS		0x0080
#define T_TOSTOP	0x0100
#define T_XKEY		0x0200		/* Fread returns escape sequences for
					   cursor keys, etc. */

/* the following are terminal status flags (tty.state) */
/* (the low byte of tty.state indicates a part of an escape sequence still
 * hasn't been read by Fread, and is an index into that escape sequence)
 */
#define TS_ESC		0x00ff
#define TS_HOLD		0x1000		/* hold (e.g. ^S/^Q) */
#define TS_COOKED	0x8000		/* interpret control chars */

/* structures for terminals */
struct tchars {
	char t_intrc;
	char t_quitc;
	char t_startc;
	char t_stopc;
	char t_eofc;
	char t_brkc;
};

struct ltchars {
	char t_suspc;
	char t_dsuspc;
	char t_rprntc;
	char t_flushc;
	char t_werasc;
	char t_lnextc;
};

struct sgttyb {
	char sg_ispeed;
	char sg_ospeed;
	char sg_erase;
	char sg_kill;
	unsigned int sg_flags;
};

struct winsize {
	int	ws_row;
	int	ws_col;
	int	ws_xpixel;
	int	ws_ypixel;
};

struct xkey {
	int	xk_num;
	char	xk_def[8];
};

struct tty {
	int		pgrp;		/* process group of terminal */
	int		state;		/* terminal status, e.g. stopped */
	int		use_cnt;	/* number of times terminal is open */
	int		res1;		/* reserved for future expansion */
	struct sgttyb 	sg;
	struct tchars 	tc;
	struct ltchars 	ltc;
	struct winsize	wsiz;
	long		rsel;		/* selecting process for read */
	long		wsel;		/* selecting process for write */
	char		*xkey;		/* extended keyboard table */
	long		rsrvd[3];	/* reserved for future expansion */
};

/* defines and declarations for Dcntl operations */

#define DEV_INSTALL	0xde02
#define DEV_NEWBIOS	0xde01
#define DEV_NEWTTY	0xde00

struct dev_descr {
	DEVDRV	*driver;
	int	dinfo;
	int	flags;
	struct tty *tty;
	long	reserved[4];
};

/* defines for TOS attribute bytes */
#ifndef FA_RDONLY
#define	       FA_RDONLY	       0x01
#define	       FA_HIDDEN	       0x02
#define	       FA_SYSTEM	       0x04
#define	       FA_LABEL		       0x08
#define	       FA_DIR		       0x10
#define	       FA_CHANGED	       0x20
#endif

#endif _filesys_h
