/*
  Reads and saves the plugin configuration.

  Following values are saved in the config file (they're in 
  sidplugin_config structure), under the SIDPLAY section.

    Key              Type      Comment
    =================================================================
    bits             int       8/16
    mixing_freq      int       In hertzes
    output_mode      int       
    center_autopan   boolean  
    fragments        int      
    fragment_size    int      
    clock_speed      int      
    playsid_compat   boolean  
    filters          boolean  
    force_clock_sync boolean  
    sid_model        int      
    =================================================================

 */

#include <gtk/gtk.h>
#include <glib.h>
#include <iostream.h> // debug aid
#include <fstream.h>
#include <xmms/plugin.h>
#include <xmms/configfile.h>
#include "sid_plugin.hh"
#include "built_iface.h"

ConfigFile *cfg;

extern struct pluginconfig sidplugin_config;
extern struct emuConfig myEmuConfig;

void load_config(void)
{
  gint tmp;
  gboolean tmpb;
  gchar *filename;

#ifdef DEBUG
  cout << "SID: Loading settings from XMMS config file." << endl;
#endif

  filename = g_strconcat(g_get_home_dir(), "/.xmms/config", NULL);
  
  if(cfg = xmms_cfg_open_file(filename)) {
    // Already saved configuration found, loading normally

#ifdef DEBUG
    cout << "SID: Found config from XMMS settings file. Restoring..."
	 << endl;
#endif
    
    // First, we set defaults (in case something is missing)
    set_sidconfig_defaults();

#ifdef DEBUG
    cout << "SID: Now that we have the defaults, off we go..."
	 << endl;
#endif   

    // 8 bit vs 16 bit
    if(xmms_cfg_read_int(cfg, "SIDPLAY", "bits", &tmp)) {
      if(tmp == 8)
	myEmuConfig.bitsPerSample = SIDEMU_8BIT;
      else
	myEmuConfig.bitsPerSample = SIDEMU_16BIT;
    }

#ifdef DEBUG
    cout << "SID: We have BITS! (" <<
      (myEmuConfig.bitsPerSample==SIDEMU_8BIT?"8":"16") << ")"
	 << endl;
#endif   

    // Mixing frequency
    if(xmms_cfg_read_int(cfg, "SIDPLAY", "mixing_freq", &tmp))
      myEmuConfig.frequency = tmp;

#ifdef DEBUG
    cout << "SID: We have Frequency! (" << myEmuConfig.frequency << ")"
	 << endl;
#endif   

    // Mono/Stereo/Surround
    if(xmms_cfg_read_int(cfg, "SIDPLAY", "output_mode", &tmp)) {
      switch((enum sidOutMode) tmp) {
      case sidMono:
      default:
	myEmuConfig.channels = SIDEMU_MONO;
	myEmuConfig.volumeControl = SIDEMU_NONE;
	break;
      case sidStereo:
	myEmuConfig.channels = SIDEMU_STEREO;
	myEmuConfig.volumeControl = SIDEMU_NONE;
	break;
      case sidStereoSurround:
	myEmuConfig.channels = SIDEMU_STEREO;
	myEmuConfig.volumeControl = SIDEMU_STEREOSURROUND;
	break;
      }
    }

#ifdef DEBUG
    cout << "SID: We have What We Hear! (" <<
      (myEmuConfig.channels==SIDEMU_MONO?"Mono":
       (myEmuConfig.volumeControl==SIDEMU_NONE?"Stereo":
	"Stereo Surround")) << ")"
	 << endl;
#endif   

    // Centered autopan
    if(xmms_cfg_read_boolean(cfg, "SIDPLAY", "center_autopan", &tmpb)) {
      if(tmpb) {
	myEmuConfig.autoPanning = SIDEMU_CENTEREDAUTOPANNING;
      } else {
	myEmuConfig.autoPanning = SIDEMU_NONE;
      }
    }

#ifdef DEBUG
    cout << "SID: We " <<
      (myEmuConfig.autoPanning==SIDEMU_NONE?"do not":"do")
	 << " have Autopan!" << endl;
#endif   
    
    // Fragments/FragmentSize
    // Hmmmh... do we need this crap? It's not used anywhere...
    if(xmms_cfg_read_int(cfg, "SIDPLAY", "fragments", &tmp))
      sidplugin_config.fragments = (int)tmp;
    if(xmms_cfg_read_int(cfg, "SIDPLAY", "fragment_size", &tmp))
      sidplugin_config.fragSizeBase = (int)tmp;

#ifdef DEBUG
    cout << "SID: We have " << sidplugin_config.fragments <<
      " fragments of " << sidplugin_config.fragSizeBase << " each." << endl;
#endif   

    // PAL vs. NTSC
    if(xmms_cfg_read_int(cfg, "SIDPLAY", "clock_speed", &tmp)) {
      if((enum sidCS) tmp == PAL) {
	myEmuConfig.clockSpeed = SIDTUNE_CLOCK_PAL;
      } else {
	myEmuConfig.clockSpeed = SIDTUNE_CLOCK_NTSC;
      }
    }

#ifdef DEBUG
    cout << "SID: We live in world of "
	 << (myEmuConfig.clockSpeed == SIDTUNE_CLOCK_PAL ?
	     "European" : "American") << " speed tastes." << endl;
#endif   

    // PlaySID compatibility mode
    if(xmms_cfg_read_boolean(cfg, "SIDPLAY", "playsid_compat", &tmpb)) {
      if(tmpb) {
	myEmuConfig.memoryMode = MPU_PLAYSID_ENVIRONMENT;
      } else {
	myEmuConfig.memoryMode = MPU_BANK_SWITCHING;
      }
    }

#ifdef DEBUG
    cout << "SID: We "
	 << (myEmuConfig.memoryMode==MPU_PLAYSID_ENVIRONMENT?"":"DON'T ")
	 << "care of Playsid..." << endl;
#endif   

    // Filter emulation
    if(xmms_cfg_read_boolean(cfg, "SIDPLAY", "filters", &tmpb)) {
      if(tmpb) {
	myEmuConfig.emulateFilter = TRUE;
      } else {
	myEmuConfig.emulateFilter = FALSE;
      }
    }

#ifdef DEBUG
    cout << "SID: We "
	 << (myEmuConfig.emulateFilter==TRUE?"":"don't")
	 << "emulate filters." << endl;
#endif   

    // Force clock speed = song speed
    if(xmms_cfg_read_boolean(cfg, "SIDPLAY", "force_clock_synch", &tmpb)) {
      if(tmpb) {
	myEmuConfig.forceSongSpeed = TRUE;
      } else {
	myEmuConfig.forceSongSpeed = FALSE;
      }
    }

#ifdef DEBUG
    cout << "SID: We "
	 << (myEmuConfig.forceSongSpeed == TRUE?"":"don't ")
	 << "force clock speed to song speed." << endl;
#endif   
    
    // SID model
    if(xmms_cfg_read_int(cfg, "SIDPLAY", "sid_model", &tmp)) {
      if((enum sidMdl) tmp == sidMOS6581) {
	myEmuConfig.mos8580 = FALSE;
      } else {
	myEmuConfig.mos8580 = TRUE;
      }
    }

#ifdef DEBUG
    cout << "SID: We emulate "
	 << (myEmuConfig.mos8580==FALSE?
	     "6581 (old SID)":
	     "8580 (new SID)") << "." << endl;
    cout << "SID: Done loading." <<endl;
#endif   

  } else {
#ifdef DEBUG
    cout << "SID: No configuration found, using defaults!" << endl; 
#endif

    set_sidconfig_defaults(); // get defaults.
    g_free(filename);  // get rid of...
    xmms_cfg_free(cfg); // ...this vast unnecessary crap...
    save_config(); // ...and save the defaults to file.
    return;
  }

#ifdef DEBUG
  cout << "SID: Freeing filename." << endl; 
#endif  
  g_free(filename);
    
#ifdef DEBUG
    cout << "SID: Freeing configuration file object." << endl; 
#endif  
    xmms_cfg_free(cfg);

    /*
#ifdef DEBUG
    cout << "SID: Re-saving configuration (just in case, "
	 <<"'cuz we're PaRaNoId)." << endl; 
#endif  
    save_config(); // Loaded, then saved. Paranoid, aren't we?
    */

#ifdef DEBUG
    cout << "SID: Here's what I now have:" << endl; 
    dump_settings_to_stdout();
#endif
}

/*
  Save config to config file.

  This code has become WAY better. A lot less Microsoftitis.
  Know what? I think Microsoft coders had been using too much Assembly
  in their previous lives. =)
 */

void save_config(void)
{

#ifdef DEBUG
  cout << "SID: Dammit, *how* can it crash *here*?." << endl;
#endif

  gint tmp;
  gboolean tmpb;
  gchar *filename;

#ifdef DEBUG
  cout << "SID: Saving settings to XMMS config file." << endl;
#endif

  filename = g_strconcat(g_get_home_dir(), "/.xmms/config", NULL);
  cfg = xmms_cfg_open_file(filename);
  if (!cfg)
    cfg = xmms_cfg_new();

  // 8 vs 16 Bits
  xmms_cfg_write_int(cfg, "SIDPLAY", "bits",
		     ((myEmuConfig.bitsPerSample==SIDEMU_8BIT)
		      ? 8 : 16));

  // Mixing frequency
  xmms_cfg_write_int(cfg, "SIDPLAY", "mixing_freq", myEmuConfig.frequency);


  // Reteo / Mono / Surynd
  // I could make that into a ?: way, but that'd not be Readable.
  if(myEmuConfig.channels == SIDEMU_MONO) {
    xmms_cfg_write_int(cfg, "SIDPLAY", "output_mode", (gint) sidMono);
  } else
    if(myEmuConfig.volumeControl == SIDEMU_STEREOSURROUND) {
      xmms_cfg_write_int(cfg, "SIDPLAY", "output_mode",
			 (gint) sidStereoSurround);
    } else {
      xmms_cfg_write_int(cfg, "SIDPLAY", "output_mode",
			 (gint) sidStereo);
    }

  // Centered autopewn
  xmms_cfg_write_boolean(cfg, "SIDPLAY", "center_autopan",
			 myEmuConfig.autoPanning);

  // Fragments & Fragsize
  xmms_cfg_write_int(cfg, "SIDPLAY", "fragments",
		     (gint) sidplugin_config.fragments);
  xmms_cfg_write_int(cfg, "SIDPLAY", "fragment_size",
		     (gint) sidplugin_config.fragSizeBase);

  // Clockspeed
  xmms_cfg_write_int(cfg, "SIDPLAY", "clock_speed",
		     ((myEmuConfig.clockSpeed==SIDTUNE_CLOCK_PAL)
		      ? (gint) PAL : (gint) NTSC));

  // playsid compatibility
  xmms_cfg_write_boolean(cfg, "SIDPLAY", "playsid_compat",
			 ((myEmuConfig.memoryMode==MPU_PLAYSID_ENVIRONMENT)
			  ? TRUE : FALSE));

  // Filter emulation
  xmms_cfg_write_boolean(cfg, "SIDPLAY", "filters",
			 myEmuConfig.emulateFilter);

  // Force song speed = clock speed
  xmms_cfg_write_boolean(cfg, "SIDPLAY", "forceclock_sync",
			 myEmuConfig.forceSongSpeed);

  // SID model
  xmms_cfg_write_int(cfg, "SIDPLAY", "sid_model",
		     ((myEmuConfig.mos8580==FALSE)
		      ? (gint)sidMOS6581 : (gint)sidMOS8580));

  xmms_cfg_write_file(cfg, filename);

  xmms_cfg_free(cfg);
}


void set_sidconfig_defaults(void)
{

  myEmuConfig.bitsPerSample = SIDEMU_8BIT; // Bits
  myEmuConfig.frequency = 22050;           // Frequency
  myEmuConfig.channels = SIDEMU_MONO;      // Mono...
  myEmuConfig.volumeControl = SIDEMU_NONE; //   ...never has surround!

  myEmuConfig.autoPanning = SIDEMU_NONE;   // No centered autopan.

  // Fragments - still kept in sidplugin_config 'til I decide
  // if I shall bury it or praise it.
  sidplugin_config.fragments = 16;
  sidplugin_config.fragSizeBase = 12;

  myEmuConfig.clockSpeed = SIDTUNE_CLOCK_PAL; // PAL vs NTSC
  myEmuConfig.memoryMode = MPU_BANK_SWITCHING; // PlaySID compatibility
  myEmuConfig.emulateFilter = TRUE;
  myEmuConfig.forceSongSpeed = FALSE;
  myEmuConfig.mos8580 = FALSE;

#ifdef DEBUG
  cout << "SID: Defaults set." << endl;
#endif
  // internal_to_sidplay(); // No longer, our data *is* internal!
}



/*  void gtk_toggle_button_set_active( GtkToggleButton *toggle_button,
                                         gint             state );
 */
void internal_to_config(void)
{
#ifdef DEBUG
  cout << "SID: Passed internal settings to config dialog." << endl;
#endif
}


/*
  Read values from the configuration dialog and store them to internal
  values.
*/

void config_to_internal(void)
{
#ifdef DEBUG
  cout << "SID: Taken config dialog settings into internal settings." << endl;
#endif
}


/*
  Just dumps the settings to stdout... This is for purely debuggational
  purposes.
*/

void dump_settings_to_stdout(void) {
#ifndef OLD_AND_OBSOLETE_STRUCTS
#ifdef DEBUG
  cout << "SID: Uses antiquated SID settings dumper. Will be fixed"
       << " someday." << endl;
#endif
#else
  cout
    << "SID settings dump" << endl
    << "============================================================" << endl
    << "Bits per sample         : " << sidplugin_config.bitsPerSample << endl
    << "Mixing frequency        : " << sidplugin_config.mixingFrequency << endl
    << "output mode             : ";

  switch(sidplugin_config.outMode) {
  default:
    cout << "Default? Huh????" << endl;
    break;
  case sidMono:
    cout << "Mono" << endl;
    break;
  case sidStereo:
    cout << "Stereo" << endl;
    break;
  case sidStereoSurround:
    cout << "Dolby Surround -- Where Available(tm)" << endl;
    break;
  }

  cout
    << "Centered autopan        : " <<
    ((sidplugin_config.centeredAutoPan)?"On":"Off") << endl
    << "Fragments               : " << sidplugin_config.fragments << endl
    << "Fragment Size Base      : " << sidplugin_config.fragSizeBase
    << endl
    << "Clock Speed             : " <<
    ((sidplugin_config.clockSpeed==PAL)?"PAL":"NTSC") << endl
    << "PlaySID Compatibility   : " <<
    ((sidplugin_config.playSidComp)?"On":"Off") << endl
    << "Filter emulation        : " <<
    ((sidplugin_config.filterEmulation)?"On":"Off") << endl
    << "Force SongSpd=ClockSpd  : " <<
    ((sidplugin_config.forceSongSpeedClockSpeed)?"On":"Off") << endl
    << "SID Model               : " <<
    ((sidplugin_config.sidModel==sidMOS6581)?"MOS 6581":"MOS 8580") << endl
    << "============================================================" << endl;
#endif
}



void validate_settings(void) {
#ifdef DEBUG
  cout << "SID: Sanitizing settings." << endl;
#endif
  if(myEmuConfig.autoPanning==SIDEMU_CENTEREDAUTOPANNING &&
     myEmuConfig.channels==SIDEMU_MONO) {
    myEmuConfig.channels = SIDEMU_STEREO; // Sanity setting
  }
  // internal_to_sidplay();
}
