/*
 * Test program for new 3D renderer.
 * Copyright 1995 Atari Corporation.
 * All Rights Reserved.
 */
 
 /* Simple demo by TNG dec-04, released -05*/

#include "olist.h"
#include "blit.h"
#include "stdlib.h"
#include "n3d.h"
#include "n3dintern.h"

/****************************************************************
 *	Defines							*
 ****************************************************************/
#define BASE		0xF00000L	/* TOM Internal Register Base */
#define BG		((volatile short *)(BASE+0x58))    /* Background Color */

/* camera width and height */
#define CAMWIDTH 320
#define CAMHEIGHT 200

/* the width and height of the screen object */
#define OBJWIDTH 320
#define OBJHEIGHT 200
#define WIDFLAG WID320		/* blitter flags corresponding to OBJWIDTH */

/* length in bytes of a screen line (2 data buffers+1 Z buffer ->
   3*2 = 6 bytes/pixel */
#define LINELEN (OBJWIDTH*6L)

/* 1/100th of the clock speed */
#define MHZ 265900L

/****************************************************************
 *	Type definitions					*
 ****************************************************************/

/* structure that records the rotation and
 * position of an object
 */
struct angles {
	short alpha, beta, gamma;		/* rotations */
	short xpos, ypos, zpos;		/* position */
};

/*
 * a renderer consists of:
 * (1) The name of the renderer
 * (2) The GPU package to load
 * (3) Entry point for that
 * (4) A flag specifying whether textures should be
 *     in normal format (0) or relative to 0x80 (1)
 */

typedef struct renderer {
	char *name;
	long *gpucode;
	void (*gpuenter)();
	short texflag;
	short null;		/* here for padding only */
} RENDERER;

/*
 * a model consists of:
 * (1) a pointer for the data for the model
 * (2) starting X,Y,Z coordinates for the model
 */
typedef struct model {
	N3DObjdata *data;
	int initx, inity, initz;
} MODEL;

/****************************************************************
 *	External functions					*
 ****************************************************************/

extern void	VIDon(int);				/* turns video on */
extern void	VIDsync(void);				/* waits for a vertical blank interrupt */
extern void	mkMatrix(Matrix *, struct angles *); /* builds a matrix */
extern void	GPUload(long *);			/* loads a package into the GPU */
extern void	GPUrun(void (*)());		/* runs a GPU program */
extern void	SetUpPart2();				/* Setup stuff for switching to scene 2, tha tunnel! =) */
extern	long	Xpos;						/* Current Xpos for tunnel */
extern	long	Ypos;						/* Current ypos for tunnel */
extern void	SetVolume(short int vol);	/* Module volume. Parameters not implemented*/
extern void	LoadModule(void);			/* Load module subroutine*/
extern	void	ReadJoypads();				/* Just for some interaction in demo*/
extern void	ParseInput();				/* Reads joypad and manipulate tunnel movement*/

/****************************************************************
 *	External variables					*
 ****************************************************************/

/* library count of number of 300ths of a second elapsed */
extern long _timestamp;

/* long-aligned parameter block for the GPU */
extern long params[];

/* the 2 screen buffers */
extern short DISPBUF0[], DISPBUF1[];
#define DATA1 ((char *)DISPBUF0)
#define DATA2 ((char *)DISPBUF1)

/* 3D object data */
/* fuji_3ddata is naturally the Atari (rest in peace) fuji and */
/* tngdata is the TNG 3d logo */
extern N3DObjdata fuji_3ddata, tngdata;

/****************************************************************
 *	External GPU references					*
 ****************************************************************/

extern long gstexcode[], texcode[], gourcode[], wfcode[], flattexcode[];
extern void gstexenter(), texenter(), gourenter(), wfenter(), flattexenter();

/****************************************************************
 *	Initialized Data					*
 ****************************************************************/

/* renderers supported */
RENDERER rend[] = {
	{"Wire Frames", wfcode, wfenter, 0},
	{"Gouraud Only", gourcode, gourenter, 0},
	{"Plain Textures", texcode, texenter, 0},
	{"Flat Shaded Textures", flattexcode, flattexenter, 1},
	{"Gouraud Shaded Textures", gstexcode, gstexenter, 1},
};
#define maxrenderer (sizeof(rend)/sizeof(RENDERER))

/* models we can draw */
MODEL models[] = {
	{ &fuji_3ddata, 0, 65, 1000 }, /* Initialization coords*/
	{ &tngdata, 600, 370, 2000 },

};
#define maxmodel (sizeof(models)/sizeof(MODEL))

/* object list for first screen */
union olist buf1_olist[2] =
{
	{{OL_BITMAP,	/* type */
	 20+(320-OBJWIDTH)/2, 20+(240-OBJHEIGHT),		/* x, y */
	 0L,		/* link */
	 DATA1,		/* data */
	 OBJHEIGHT, OBJWIDTH*3/4, OBJWIDTH/4,		/* height, dwidth, iwidth */
	 4, 3, 0, 0, 0,	/* depth, pitch, index, flags, firstpix */
	 0,0,0}},		/* scaling stuff */

	{{OL_STOP}}
};

/* object list for second screen */
union olist buf2_olist[2] =
{
	{{OL_BITMAP,	/* type */
	 20+(320-OBJWIDTH)/2, 20+(240-OBJHEIGHT),		/* x, y */
	 0L,		/* link */
	 DATA2,		/* data */
	 OBJHEIGHT, OBJWIDTH*3/4, OBJWIDTH/4,		/* height, dwidth, iwidth */
	 4, 3, 0, 0, 0,	/* depth, pitch, index, flags, firstpix */
	 0,0,0}},		/* scaling stuff */

	{{OL_STOP}}
};

/* Bitmaps for the two screens */
Bitmap scrn1 = {
	CAMWIDTH, CAMHEIGHT,
	PIXEL16|PITCH3|ZOFFS2|WIDFLAG,
	(void *)(DATA1 + ((OBJWIDTH-CAMWIDTH)*3L) + (((OBJHEIGHT-CAMHEIGHT)/2)*LINELEN) ),
};

/* initial data for camera corresponding to second screen buffer */
Bitmap scrn2 = {
	CAMWIDTH, CAMHEIGHT,
	PIXEL16|PITCH3|ZOFFS1|WIDFLAG,
	(void *)(DATA2 + ((OBJWIDTH-CAMWIDTH)*3) + (((OBJHEIGHT-CAMHEIGHT)/2)*LINELEN) ),
};

/*Light for the fuji*/
Lightmodel lightf = {
	0x0000,
	1,
	{
	  { 0, 0x1000, -0x4000, 0x4000 },
	  { 0, 0, 0, 0xC000 },
	  { 0x4000, 0, 0, 0x4000 },
	  { 0, 0x4000, 0, 0x4000 },
	  { 0, 0, 0x4000, 0x4000 },
	}
};
/*light for the TNG logo*/
Lightmodel lightl = {
	0x0000,
	1,
	{
	  { -0x24f3, -0x24f3, -0x24f3, 0x4000 },
	  { 0, 0, 0, 0xC000 },
	  { 0x4000, 0, 0, 0x4000 },
	  { 0, 0x4000, 0, 0x4000 },
	  { 0, 0, 0x4000, 0x4000 },
	}
};


/****************************************************************
 *	Global variables					*
 ****************************************************************/

/* flag for current texture state */
int texturestate;

/* pointer to temporary storage for transformed points */
TPoint	*tpoints;

/* storage for packed object lists */
int packed_olist1[160];
int packed_olist2[160];

struct angles camangles[2], objangles[2];

N3DObject testobj[2];

short cc = 0x78ff;	/*Current BG color, start white*/

/****************************************************************
 *	Functions						*
 ****************************************************************/

/****************************************************************
 *	N3Dclrbuf(buf)						*
 * Clears the bitmap pointed to by "buf", filling its data with *
 * a solid color, and its Z buffer with a null value		*
 ****************************************************************/

void
N3Dclrbuf(Bitmap *buf)
{
/*	long bgcolor = 0x27a027a0;		*//* fill color, duplicated */
	/*Ugly fast fix by spock, take cc as "parameter"*/
	long bgcolor = cc;		/* fill color, duplicated */
	long zvalue = 0xffffffff;		/* Z value (16.16 fraction) */
	bgcolor = (bgcolor <<16)|cc;

	B_PATD[0] = bgcolor;
	B_PATD[1] = bgcolor;
	B_Z3 = zvalue;
	B_Z2 = zvalue;
	B_Z1 = zvalue;
	B_Z0 = zvalue;
	A1_BASE = (long)buf->data;
	A1_STEP = 0x00010000L | ((-buf->width) & 0x0000ffff);
	A1_FLAGS = buf->blitflags|XADDPHR;
	A1_PIXEL = 0;
	A1_CLIP = 0;
	B_COUNT = ((long)buf->height << 16) | (buf->width);
	B_CMD = UPDA1|DSTWRZ|PATDSEL;
}

/****************************************************************
 *	N3Drender(window, obj, cam, lmodel, rend)		*
 * Render an object into a bitmap. The parameters are:		*
 *	window: the destination bitmap				*
 *	obj:	the N3D object to render			*
 *	cam:	the viewing matrix				*
 *	lmodel:	the lighting model				*
 *	rend:	the renderer to use (wireframe, gouraud,	*
 *		or texture mapped				*
 ****************************************************************/

void
N3Drender(Bitmap *window, N3DObject *obj, Matrix *cam, Lightmodel *lmodel, RENDERER *rend)
{
	/* load GPU code */
	GPUload(rend->gpucode);

	/* allocate temporary storage */
	tpoints = malloc(sizeof(TPoint)*obj->data->numpoints);

	params[0] = (long)obj->data;
	params[1] = (long)&obj->M;
	params[2] = (long)window;
	params[3] = (long)cam;
	params[4] = (long)lmodel;
	params[5] = (long)tpoints;

	GPUrun(rend->gpuenter);

	free(tpoints);
}

/****************************************************************
 *	fixtexture( texture )					*
 * Adjust all intensities in a texture so that they are		*
 * relative to 0x80. This is done for renderers that do		*
 * shading on textures; because of the way the shading is	*
 * done, the source data must with intensities as signed	*
 * offsets to a base intensity (namely 0x80). So before using	*
 * a renderer that does shading on textures, this function	*
 * must be called on all textures in the model to be rendered.	*
 *								*
 * Note that because of the implementation, calling this	*
 * function twice yields the original texture back. This is	*
 * handy because it means that we can switch from unshaded	*
 * textures to shaded ones and then back again, calling this	*
 * function each time we switch.				*
 ****************************************************************/

void
fixtexture( Bitmap *texture )
{
	long *lsrc;
	long numpixs;
	long i;

	numpixs = ((long)texture->width * (long)texture->height)/4;
	lsrc = (long *)texture->data;

	for (i = 0; i < numpixs; i++) {
		*lsrc++ ^= 0x00800080L;
		*lsrc++ ^= 0x00800080L;
	}
}

/*
 * Fix up all textures in all models.
 * This is called when switching between renderers;
 * if the new renderer uses a different texture
 * shading model than the old renderer, then we
 * call fixtexture() on every texture in every
 * model.
 */

void
fixalltextures(int newrender)
{
	int i, j;
	N3DObjdata *curobj;
	Bitmap *map;

	if (texturestate != newrender) {
		for (i = 0; i < maxmodel; i++) {
			curobj = models[i].data;
			for (j = 0; j < curobj->nummaterials; j++) {
				map = curobj->materials[j].tmap;
				if (map)
					fixtexture(map);
			}
		}
		texturestate = newrender;
	}
}



/****************************************************************
 *	main()							*
 * The main demo loop.						*
 ****************************************************************/
	Bitmap *curwindow;		/* pointer to output bitmap */

int
main()
{
	int drawbuf;			/* flag: 0 means first buffer, 1 means second */
	Bitmap *curwindow;		/* pointer to output bitmap */
	Matrix cammatrix[2];		/* camera matrix */
	struct angles *curangles;	/* which set of angles (viewer or camera) are being manipulated */
	int currender;			/* current renderer in use (index into table) */
	int curmodel;			/* current model in use (index into table) */
	/*char buf[256];*/			/* scratch buffer for sprintf */
	
	volatile short *BCOLOR;
	BCOLOR = (short *)BG;
	*BCOLOR = cc;				/* Assign BG colour*/
	/* build packed versions of the two object lists */
	/* (output is double buffered)			 */
	OLbldto(buf1_olist, packed_olist1);
	OLbldto(buf2_olist, packed_olist2);

	/* initialize the video */
	OLPset(packed_olist2);
	VIDon(0x6c1);			/* 0x6c1 = CRY; 0x6c7 = RGB */

	/* wait for video sync (paranoid code) */
	VIDsync();
	
/* Music module load*/
	LoadModule();
	VIDsync();
/* Music volume set*/
	SetVolume(200); /*Function ignores parameter right now*/
	
	/* clear the drawing area to white */
	memset(DATA1, 0x78ff, OBJWIDTH*(long)OBJHEIGHT*2L*3);	/* clear screen to white */

	drawbuf = 0;			/* draw on buffer 1, while displaying buffer 2 */
	currender = maxrenderer - 1;	/* render package to use (Gourand Shaded Textures) */

	curmodel = 0;			/* model to draw */
	

	/* initialize the test object */
	memset(&testobj[0], 0, sizeof(testobj[0]));
	memset(&testobj[1], 0, sizeof(testobj[1]));
	testobj[0].data = models[0].data;
	testobj[1].data = models[1].data;
	objangles[0].xpos = models[0].initx;	/* get initial position */
	objangles[0].ypos = models[0].inity;
	objangles[0].zpos = models[0].initz;
	objangles[1].xpos = models[1].initx;	/* get initial position */
	objangles[1].ypos = models[1].inity;
	objangles[1].zpos = models[1].initz;
	/* no rotation, initially */
	objangles[0].alpha = objangles[0].beta = objangles[0].gamma = 0;
	objangles[1].alpha = objangles[1].beta = objangles[1].gamma = 0;

	/* set up the viewer's position */
	camangles[0].alpha = 180;
	camangles[0].beta = camangles[0].gamma = 0;
	camangles[1].alpha = camangles[1].beta = camangles[1].gamma = 0;
	camangles[0].xpos = camangles[0].ypos = 0;
	camangles[0].zpos = 0;
	camangles[1].xpos = camangles[1].ypos = camangles[1].zpos = 0;
	objangles[1].alpha = 510;
	
	
	/* initially all rotation and movement is applied to the object,
	   not the viewer
	 */
	curangles = &objangles[0];


	/* initially textures are unshaded */
	texturestate = 0;


	/* set up the textures for the first renderer */
	/*(Not used in demo)*/
	fixalltextures(rend[currender].texflag);
	
	/* loop forever */
	for(;;) {
		/* select bitmap for drawing */
		curwindow = (drawbuf) ? &scrn2 : &scrn1;

		/* generate transformation matrices from angles */
		mkMatrix(&testobj[0].M, &objangles[0]);
		mkMatrix(&testobj[1].M, &objangles[1]);
		mkMatrix(&cammatrix[0], &camangles[0]);
		mkMatrix(&cammatrix[1], &camangles[1]);
	
		/* clear the current draw buffer */
		N3Dclrbuf(curwindow);

		/* now draw the object*/
	/*Fuji*/
		currender = 1; /* Gouraud only*/
		N3Drender(curwindow, &testobj[0], &cammatrix[0], &lightf, &rend[currender]);
	/*TNG logo*/
		/*currender = maxrenderer - 1; *//* use textures & gouraud*/
		N3Drender(curwindow, &testobj[1], &cammatrix[1], &lightl, &rend[currender]);
		if (camangles[0].alpha > 0)
		  camangles[0].alpha -= 1;
		else if (objangles[0].beta >4100)
			break;/*Ends eternal for statement. Much nicer if I use for, I know! =)*/
		objangles[0].beta += 10;
		/*objangles[1].alpha =1800;*/
		objangles[1].beta -= 20;
		if (cc > 0x7800)
		  *BCOLOR = --cc;			/* Fade to black */
		/* display the buffer we just drew */
		OLPset(drawbuf ? packed_olist2 : packed_olist1);

		/* wait for vblank */
		VIDsync();

		/* switch drawing buffers */
		drawbuf = !drawbuf;
	}
	/*Could have made the transition between part1 & part2 nicer, but ran out of time*/
	SetUpPart2();			/*SETUP tunnel part(2)*/
	/*abort();*//* return to debugger (illegal)*/
	for(;;){ /*loop forever, part2 (tunnel)*/
	  ReadJoypads();
	  ParseInput();
	  VIDsync();
	  Xpos+=1;
	  Ypos+=1;
	}
	return 0;
	
}
