;**********************************************************************
; Jaguar UART registers and bits

u_clk		equ	$f10034
u_ctrl		equ	$f10032
u_stat		equ	$f10032
u_data		equ	$f10030

u_err		equ	15
u_rbf		equ	7
u_tbe		equ	8

ctl_init	equ	$0001		; No parity, no ints, active high o/ps
clrerr		equ	$0040		; clear errors

PCLK		equ	26593900	; CPU clock (good enough for PAL/NTSC)
;BAUD           equ     9600            ; RS232 clock rate - original rate from test software code
;BAUD           equ     57600           ; RS232 clock rate
BAUD           equ     115200          ;overdrive RS232 speed rate
;BAUD            equ     300             ; slow down for testing!
N_ACLK		equ	((PCLK/(16*BAUD))-1)	; Prescaler value

U_TIMEOUT	equ	30000		; about 30ms
U_MAX		equ	5000		; max characters in a single string
                .globl   uart_init
                .globl   _uart_send
                .globl   uart_send
                .globl   uart_get
                .globl   uart_shex
; DO: we don't want this                .extern  main
;**********************************************************************
uart_init:
	move.w	#N_ACLK,u_clk		; baud rate
	move.w	#clrerr,u_ctrl		; clear all errors
	move.w	u_data,d0		; read any old data
	move.w	u_data,d0		; ... both buffers
	move.w	#ctl_init,u_ctrl	; initialise
	rts

;*******************************************************************
; uart_send	send a message to the terminal
;
; Gets:		a0 = pointer to message, terminated with NULL
;
; Returns: 	d0 =  0 if successful
; 		d0 = err_u1 if timeout on send
; 		d1 = trashed
; 		d2 = trashed
;		a0 = next byte after the NULL
;
;will wait for previous character to finish sending first
;*******************************************************************
_uart_send:
    move.l  4(sp),a0        ; for C-call
uart_send:
;
;  let's print a message that must be read a longword at a time
;

	move.w	#U_MAX,d2	; maximum number of characters
;
	move.l	a0,d1		;check message ptr for alignment
	andi.w	#3,d1		; are we on long boundary?
	beq.s	.ulp3		; br if so, don't need start fringe
;
	add.w	d1,d2		;fake out max char count for this case	
	move.l	a0,d0		; else, pick up chars at nearest long
	andi.w	#~3,d0
	move.l	d0,a0

	move.l	(a0)+,d0	;get all 4 tho less than that needed
	swap	d2		; we need upper half of d2 for counter
	move.w	d1,d2		;d2.w = 1,2,or 3
	not.w	d2
	andi.w	#3,d2		;d2.w = 2,1,or 0 (dbra count)
;	
	lsl.w	#3,d1		;1,2, or 3 * 8 for char alignment
	lsl.l	d1,d0		;align 4-some
	bra.s	.ulp0		; enter main loop with excess header trimmed
;
;   go for next 4-char block
;
.ulp3:
	swap	d2		;use upper half of d2 for counter
	move.w	#3,d2		;4 chars at a time
	move.l	(a0)+,d0	;pick up next 4
.ulp0:
	move.w	#U_TIMEOUT,d1	;time-out count set for each char
	rol.l	#8,d0		;get next char in LSByte
	tst.b	d0
	beq	.msgov		;if all shifted out, we're done
.ulp1:
	swap	d1		;use upper d1 for temp
	move.w	u_stat,d1	;pick up transmit buffer empty status
	swap	d1
	btst.l	#u_tbe+16,d1	;add 16 for upper word access
	bne	.ulp2		;br if transmit buffer empty
	dbra	d1,.ulp1	;test until time-out
	bra	.uerr		;br to time-out error (bogus)
.ulp2:
	move.w	d0,d1		;copy current char to temp
	andi.w	#$ff,d1		;so we can mask off 15:08
	move.w	d1,u_data	;send char
;
	move.w	#$400,d1
.tbewait:
	swap	d1
	move.w	u_stat,d1	;wait for transmit buffer empty 
	swap	d1
	btst.l	#u_tbe+16,d1	;  to go away
	beq.s	.tbegone
	dbra	d1,.tbewait
.tbegone:	
;
;	move.w	#$400,d1
;.uldly:	
;	dbra	d1,.uldly	;delay for a bit
;
	dbra	d2,.ulp0	;go for all 4, this long
;
	swap	d2		;now go for max char count
	subq.w	#4,d2		
	bpl	.ulp3
.uerr:
	move.b	#err_u1,d0
	jmp	(a6)
.msgov:
	suba.w	d2,a0
	moveq	#0,d0
	rts
;
;
;*******************************************************************
; uart_get	gets a character from the terminal
;
; Gets:		
;
; Returns:	d0 = error code
; 		d1 = character received
;
;will wait forever for a character to be received
;
;*******************************************************************
uart_get:
	move.w	u_stat,d0	; Get status
	btst	#u_err,d0
	bne	.uget_err
	btst	#u_rbf,d0	; data?
	beq	uart_get	; ..no, loop again
	move.w	u_data,d1	; ..yes, get the data
	andi.w	#$00ff,d1

.ulp3:
	move.w	#U_TIMEOUT,d0
.ulp1:
	swap	d0
	move.w	u_stat,d0
	swap	d0
	btst	#u_tbe+16,d0
	bne	.ulp2		; transmit buffer empty
	dbra	d0,.ulp1
	bra	.uget_err
.ulp2:
	cmpi.b	#27,d1		;don't echo <esc>
	beq	.ulp4
	move.w	d1,u_data	; echo it
.ulp4:
	sub.l	d0,d0		; ..and clear error flag
	rts
.uget_err:
	move.l	#err_u2,d0
	rts
;
;
;
;*************************************************************************
; uart_shex	send up to 8 nibbles of hex ASCII
; Gets:		d0.x	= hex value
;		d2.w	= nibbles to send (1 to 8)
;
; Returns	d0-d3	trashed
;*************************************************************************
uart_shex:
	move.w	u_stat,d1	; Get status
	btst	#u_rbf,d1	; Check to see if a XOFF character was received
	beq	.ulp2		; Nope continue with transmission
.ulp3:
	move.w	u_data,d1	; Get data
	andi.w	#$00ff,d1	;  mask low byte
	cmp.b	#$1b,d1		; Is it an <esc> character?
	beq	shex_exit	; Yes, then restart program
	cmp.b	#$11,d1		; Is it a XON character?
	bne	.ulp3		; Nope, look again
.ulp2:
	move.l	d0,d1		; save value in d1
	subq.w	#1,d2
	move.w	d2,d3		; d3 = characters to send -1
	lsl.w	#2,d2		; d2 = bits to send - 4

;	lea.l	hextab,a0	; point to ASCII table
.ulp1:
	move.w	u_stat,d0
	btst	#u_tbe,d0
	beq	.ulp1		; wait forever for transmit buffer empty

	move.l	d1,d0		; convert to ASCII
	lsr.l	d2,d0		; ..shift down the nibble
	and.l	#$0f,d0		; ..mask it off
	move.b	hextab(PC,d0),d0	; ..and look up the ASCII

	move.w	d0,u_data	; send it
	lsl.l	#4,d1		; shift next nibble into place
	dbra	d3,.ulp1	; and loop
	rts
shex_exit:
; DO: we don't want this	jmp	main
    rts

hextab:
dc.b	'0123456789ABCDEF'

    .data
datstart:
;*********************************************************
; Error data structures
;
; dc.l	xxxxxxxx	; address of error handler code, or ASCII message
; dc.b	xx		; 0 = address is custom error handler code
;			;-1 = address is ASCII message
; dc.b	xx		; 0 = return immediately after displaying message
;			;-1 = wait for any key after displaying  message
; dc.b	xx,xx		; not used for now
;*********************************************************
err_pass	equ	0	; test passed. No error
err_r1		equ	1
err_r2		equ	2
err_r3		equ	3
err_r4		equ	4
err_r5		equ	5
err_u1		equ	6
err_u2		equ	7
err_mna		equ	8
err_inv		equ	9
err_term	equ	10
