/* This program pads a 24 bit TGA file out to a multiple of 8 in
 * height and width (so it can be JAGpeg'd).
 *
 * This is generic ANSI C, and should compile with any ANSI compliant
 * compiler (e.g. gcc, Borland, Microsoft, or Lattice).
 *
 * History:
 * 1.1		Added -clip option
 * 1.0		First version
 */

#define VERSION "1.1"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "tgadefs.h"

Pixel *
rescale(Pixel *oldpix, unsigned old_w, unsigned old_h, unsigned new_w, unsigned new_h, int filter_type, int aspect);

Pixel *
crop(Pixel *oldpix, unsigned old_w, unsigned old_h, unsigned clip_x, unsigned clip_y, unsigned clip_w, unsigned clip_h);

#ifndef PATHMAX
#define PATHMAX 256
#endif

char *infilename;				/* input file name */
char *outfilename;				/* output file name */
Pixel *srcfile;				/* buffer holding loaded file */

unsigned int image_w;			/* width of image in pixels from TGA header */
unsigned int image_h;			/* height of image in pixels from TGA header */

int bits_per_pixel;			/* bits per pixel from TGA file header */
int bytes_in_name;			/* bytes in filename at end of TGA header */
int tga_flags;				/* TGA file flags */
int cmap_type;				/* color map type */
int sub_type;				/* TGA file sub type */
int cmap_len;				/* length of color map */

int pad_boundary;			/* what kind of boundary to pad on */
unsigned int new_width;			/* new width for image */
unsigned int new_height;		/* new height for image */
unsigned int clip_x, clip_y,		/* clipping rectangle variables */
	clip_w, clip_h;

#define hflip_flag 0
int vflip_flag = 0;

#define PROGNAME "tgastretch"
char *progname;				/* name the program was invoked with */


void
usage(void)
{
	printf("%s Version %s\n\n", progname, VERSION);
	printf("Usage: %s {options} [-o outfile] file.tga\n", progname);
	printf("This program resizes a Targa file. It's default operation is to stretch the file\n");
	printf("to the next multiple of 8 in height and width (i.e. a 17x31 file becomes 24x32).\n");
	printf("The -resize and -pad options can override this:\n");
	printf("\t-pad n         Pad to an n pixel boundary (default is 8)\n");
	printf("\t-resize w,h    Gives an explicit size (width,height) for resizing\n");
	printf("\t-crop x,y,w,h  Clip the input image before resizing; x,y is the upper left corner of the clip rectangle\n");
	exit(2);
}



void
err_eof(void)
{
	fprintf(stderr, "Error: unexpected EOF\n");
	exit(1);
}

/* State info for reading RLE-coded pixels; both counts must be init to 0 */
static int block_count;		/* # of pixels remaining in RLE block */
static int dup_pixel_count;	/* # of times to duplicate previous pixel */
static void (*read_pixel)(FILE *fhandle, Pixel *place);
static char tga_pixel[4];

/*
 * read_rle_pixel: read a pixel from an RLE encoded .TGA file
 */
static void
read_rle_pixel(FILE *fhandle, Pixel *place)
{
	int i;

	/* if we're in the middle of reading a duplicate pixel */
	if (dup_pixel_count > 0) {
		dup_pixel_count--;
		place->blue = tga_pixel[0];
		place->green = tga_pixel[1];
		place->red = tga_pixel[2];
		return;
	}
	/* should we read an RLE block header? */
	if (--block_count < 0) {
		i = fgetc(fhandle);
		if (i < 0) err_eof();
		if (i & 0x80) {
			dup_pixel_count = i & 0x7f;	/* number of duplications after this one */
			block_count = 0;		/* then a new block header */
		} else {
			block_count = i & 0x7f;		/* this many unduplicated pixels */
		}
	}
	place->blue = tga_pixel[0] = fgetc(fhandle);
	place->green = tga_pixel[1] = fgetc(fhandle);
	place->red = tga_pixel[2] = fgetc(fhandle);
}

/*
 * read a pixel from an uncompressed .TGA file
 */
static void
read_norm_pixel(FILE *fhandle, Pixel *place)
{
	int i;

	place->blue = fgetc(fhandle);
	place->green = fgetc(fhandle);
	i = fgetc(fhandle);
	if (i < 0) err_eof();
	place->red = i;
}

void
read_row(FILE *fhandle, Pixel *place)
{
	int i;
	void (*rpixel)(FILE *, Pixel *);

	rpixel = read_pixel;
	if (hflip_flag) {
		place += image_w;
		for (i = 0; i < image_w; i++) {
			place--;
			(*rpixel)(fhandle, place);
		}
	} else {
		for (i = 0; i < image_w; i++) {
			(*rpixel)(fhandle, place);
			place++;
		}
	}
}

void
read_file(char *infile)
{
	FILE *fhandle;
	int c;
	Pixel *row_pixels;
	long i;

	block_count = dup_pixel_count = 0;
	fhandle = fopen(infile, "rb");
	if (!fhandle) {
		perror(infile);
		exit(1);
	}
	bytes_in_name = fgetc(fhandle);
	cmap_type = fgetc(fhandle);
	sub_type = fgetc(fhandle);
	c = fgetc(fhandle);			/* skip bytes 3 and 4 */
	c = fgetc(fhandle);
	if (c < 0) err_eof();

	cmap_len = fgetc(fhandle) + ((unsigned)fgetc(fhandle) << 8);
	c = fgetc(fhandle);			/* skip bytes 7 through 11 */
	c = fgetc(fhandle);
	c = fgetc(fhandle);
	c = fgetc(fhandle);
	c = fgetc(fhandle);
	if (c < 0) err_eof();

	image_w = fgetc(fhandle) + ((unsigned)fgetc(fhandle) << 8);
	image_h = fgetc(fhandle) + ((unsigned)fgetc(fhandle) << 8);

	if (new_width == 0 || new_height == 0) {
		if (clip_w && clip_h) {
			new_width = clip_w;
			new_height = clip_h;
		} else {
			new_width = (image_w + pad_boundary - 1) & ~(pad_boundary - 1);
			new_height = (image_h + pad_boundary - 1) & ~(pad_boundary - 1);
		}
	}

	bits_per_pixel = fgetc(fhandle);
	if (bits_per_pixel < 0) err_eof();
	tga_flags = fgetc(fhandle);

	if (cmap_type != 0) {
		fprintf(stderr, "ERROR: Targa files with color maps not supported\n");
		exit(1);
	}
	if (bits_per_pixel != 24) {
		fprintf(stderr, "ERROR: Only 24 bit Targa files are supported\n");
		exit(1);
	}
	if ((tga_flags & 0x20) == 0) {		/* this picture is bottom-up */
		vflip_flag = !vflip_flag;
	}
	if (tga_flags & 0xc0) {
		fprintf(stderr, "ERROR: Interlaced Targa files are not supported\n");
		exit(1);
	}
/* figure out how to read source pixels */
	if (sub_type > 8) {
	/* an RLE-coded file */
		read_pixel = read_rle_pixel;
		sub_type -= 8;
	} else {
		read_pixel = read_norm_pixel;
	}

	if (sub_type == 1) {
		fprintf(stderr, "ERROR: Colormapped Targa files not supported\n");
		exit(1);
	} else if (sub_type == 2) {
		/* everything is OK */ ;
	} else {
		fprintf(stderr, "ERROR: Invalid or unsupported Targa file\n");
		exit(1);
	}

/* skip the image name */
	for (i = 0; i < bytes_in_name; i++) {
		c = fgetc(fhandle);
		if (c < 0) err_eof();
	}

	srcfile = calloc((size_t)image_w*(size_t)image_h, sizeof(Pixel));

	if (!srcfile) {
		fprintf(stderr, "ERROR: insufficient memory for image\n");
		exit(1);
	}

	if (vflip_flag) {
		row_pixels = srcfile + image_h*(long)image_w;
		for (i = 0; i < image_h; i++) {
			row_pixels -= image_w;
			read_row(fhandle, row_pixels);
		}
	} else {
		row_pixels = srcfile;
		for (i = 0; i < image_h; i++) {
			read_row(fhandle, row_pixels);
			row_pixels += image_w;
		}
	}

	fclose(fhandle);

	/* clip image */
	if (clip_w != 0 && clip_h != 0) {
		if ((clip_x + clip_w > image_w) || (clip_y + clip_h > image_h)) {
			fprintf(stderr, "Clipping rectangle too large!\n");
			exit(1);
		}
		row_pixels = crop(srcfile, image_w, image_h, clip_x, clip_y, clip_w, clip_h);
		free(srcfile);
		srcfile = row_pixels;
		image_w = clip_w;
		image_h = clip_h;
	}

	/* now resize it */
	row_pixels = rescale(srcfile, image_w, image_h, new_width, new_height, FILTER_MITCH, 0);
	free(srcfile);
	srcfile = row_pixels;
}

static void
write_file(char *outfile)
{
	FILE *fhandle;
	static unsigned char tgaheader[18];
	int i, j;
	Pixel *ptr;

	fhandle = fopen(outfile, "wb");
	if (!fhandle) {
		perror(outfile);
		exit(1);
	}

/* initialize the TARGA file header */
	tgaheader[12] = (new_width & 0xff);
	tgaheader[13] = (new_width >> 8);
	tgaheader[14] = (new_height & 0xff);
	tgaheader[15] = (new_height >> 8);

	tgaheader[17] = 0x20;	/* Top-down, non-interlaced */
	tgaheader[2] = 2;	/* uncompressed RGB */
	tgaheader[16] = 24;	/* bits per pixel */

	if (fwrite(tgaheader, 1, 18, fhandle) != 18) {
		perror(outfile);
		exit(1);
	}

	/* now write the data, in BGR order */
	ptr = srcfile;
	for (j = 0; j < new_height; j++) {
		for (i = 0; i < new_width; i++) {
			putc(ptr->blue,fhandle);
			putc(ptr->green,fhandle);
			putc(ptr->red,fhandle);
			ptr++;
		}
	}

	fclose(fhandle);
}


int
main(int argc, char **argv)
{
	progname = *argv++;
	if (!*progname) {			/* if for some reason the runtime library didn't get our name... */
		progname = PROGNAME;		/* assume this is our name */
	}
	argc--;
	pad_boundary = 8;
	outfilename = (char *)0;
	new_width = 0;
	new_height = 0;
	clip_x = clip_y = clip_w = clip_h = 0;

	while (*argv && **argv == '-') {
		if (!strcmp(*argv, "-pad")) {
			argv++; argc--;
			if (!*argv) {
				usage();
			}
			if (sscanf(*argv, "%i", &pad_boundary) != 1)
				usage();
		} else if (!strcmp(*argv, "-resize")) {
			argv++; argc--;
			if (!*argv) {
				usage();
			}
			if (sscanf(*argv, "%i,%i", &new_width, &new_height) != 2)
				usage();
		} else if (!strcmp(*argv, "-crop")) {
			argv++; argc--;
			if (!*argv) {
				usage();
			}
			if (sscanf(*argv, "%i,%i,%i,%i", &clip_x, &clip_y, &clip_w, &clip_h) != 4)
				usage();
		} else if (!strcmp(*argv, "-o")) {
			argv++; argc--;
			if (!*argv) {
				usage();
			}
			outfilename = *argv;
		} else {
			usage();		/* illegal option */
		}
		argv++; argc--;
	}
	if (argc != 1) {		/* should be exactly one argument left, the input file name */
		usage();
	}
	infilename = *argv;
	if (!outfilename) {
		outfilename = infilename;	/* write over our input */
	}

	read_file(infilename);
	write_file(outfilename);

	free(srcfile);

	return(0);
}

