;================================================================================================
;
;  TWOMICE2.S
;
;================================================================================================
; Matthias Domin, 20060211: 
; This 68000-based sourcecode (derived from JOYPAD.S provided by Atari)
; creates a test-program for checking rotary devices (digital mice or similar)
;
; Please note one important addition: 
; In this version of JOYPAD.S two BRANCH-objects are preleading the object-list,
; this avoids certain problems with displaying objects.
;
; Matthias Domin, 20060214: 
; Now a Timer-IRQ (1200Hz) is used to read from port 2 where the digital mouse 
; (or RotaryController) is plugged in.
;
;------------------------------------------------------------------------------------------------
;
; The pinout for the adapter for Atari ST digital mice:
;
; DSUB 9 male - Jaguar DSUB-HD15 male
; ===================================
;           1 - 14     ( J8/J12)   
;           2 - 13     ( J9/J13)   
;           3 - 12     (J10/J14)
;           4 - 11     (J11/J15)
;           5 - not connected
;           6 - 10     ( B1/ B3)
;           7 -  7     ( +5V DC)
;           8 -  9     (    Gnd)
;           9 -  6     ( B0/ B2)
; 
;
; And here ist the pinout for an Amiga-mouse-adapter.
; If you have an Amiga-mouse and this adapter you can 
; use the software in mode "1" with the red canvas,
; to run the software in mode "2" with the blue canvas 
; is only necessary if you have an Amiag-mouse and 
; the above described "Atari ST mouse"-adapter.
;
; DSUB 9 male - Jaguar DSUB-HD15 male
; ===================================
;           1 - 11     (J11/J15)
;           2 - 13     ( J9/J13)
;           3 - 12     (J10/J14)
;           4 - 14     ( J8/J12)
;           5 - not connected
;           6 - 10     ( B1/ B3)
;           7 -  7     ( +5V DC)
;           8 -  9     (    Gnd)
;           9 -  6     ( B0/ B2)
;                      
;
; The pinout for the adapter for VCS Driving Controllers,
; this will convert clockwise turns into upwards movements
; and counter-clockwise turns into downwards movements.
;
; DSUB 9 male - Jaguar DSUB-HD15 male
; ==============================
;           1 - 11     (J11/J15)   
;           2 - 12     (J10/J14)   
;           3 - not connected
;           4 - not connected
;           5 - not connected
;           6 - 10     ( B1/ B3)
;           7 - not connected
;           8 -  9     (    Gnd)
;           9 - not connected
;
; You can attach a VCS Driving Controller also to the above 
; described Atari ST mouse-adapter, then a clockwise turn 
; will be converted into a movement to the right and 
; a counter-clockwise turn will become a leftwards movement.
;
;
; "Movement"-directions are of course controlled by the software!
; 
;------------------------------------------------------------------------------------------------
;
; Program userinterface:
;
; Jaguar-Joypad at Port1:
; 1: Switch into mode for Atari ST-usage, canvas is filled red
; 2: Switch into mode for Amiga-usage, canvas is filled blue
; * and #: Leave program, start 8bit-BJL-loader
;
; Mouse (VCS Driving controller(DC), Rotary controller) at Port2:
; Movement: Move cursor-sprite over canvas, a white pixel will be drawn at actual position
; Left mouse-button (button of VCS-DC, Fire A on RotaryJoypad): Clear canvas
;
;================================================================================================

 .include "jaguar.inc"
 .include "jag.h"


;*********** VIDEO ***********************************************
LEVEL2 equ 4*$40



CURSOR_X equ 80     ; position of objects on the screen
CURSOR_Y equ 30*2


CANVAS_WIDTH    equ 320
CANVAS_HEIGHT   equ 200


;================================================================================================

TOTAL_OBJS      equ     2

;================================================================================================

start:
    move.l  #$00070007,G_END    ; don't need to swap for GPU
    move.w  #$35cc,MEMCON2      ; Do the old endian thing
    move.w  #$8000,CHRO_CLK

    lea.l   mypal,a0
    jsr     set_palette  ;load in the palette

    jsr     InitVideo
    jsr     IntInit

    ; center the canvas-object on the screen:
    ; vertical:
    move.w  obj_data+8,d0   ; object-height (use this value for the y-axis doubled!)
    move.w  vermid,d1       ; vertical mid of the screen
    sub.w   d0,d1           ; object-height
	move.w	d1,obj_data+2	; object-ypos (canvas)
    add.w   #CANVAS_HEIGHT/2,d1
    move.w  d1,cury         ; object-ypos (cursor)

    ; horizontal:
    move.w  obj_data+10,d0  ; object-width
    asl.w   #3-1,d0         ; *8/2
    moveq.l #0,d1
    move.w  hormid,d1       ; horizontal mid of the screen
    sub.w   a_hdb,d1        ; - start of scanline output
    divu    #3,d1           ; / pixel_clock
    sub.w   d0,d1           ; object-width
    move.w  d1,obj_data+0   ; object-xpos (canvas)
    add.w   #CANVAS_WIDTH/2,d1
    move.w  d1,curx         ; object-xpos (cursor)

    move.b  #$01,fillcolour ; red for Atari ST-mouse
	bsr		clrscreen


;***************************************************************
; Just do once to ensure audio is audible
    move.w  #$100,JOYSTICK  ; turns off audio mute
;***************************************************************

; Create branch Objects for 2 OP-lists
    lea     olist1_branch,a0    ; adress Branch Objekt
    move.l  #olist1_stop,d2     ; adress Stop Objekt
    bsr     create_branch       ; create Branch Objekt

    lea     olist2_branch,a0    ; adress Branch Objekt
    move.l  #olist2_stop,d2     ; adress Stop Objekt
    bsr     create_branch       ; create Objekt erstellen


    move.w  #TOTAL_OBJS,d6
    lea     obj_data,a1
    jsr     build               ; initialize the object lists
    jsr     copy_olist


    move.l  #olist1_branch,d0   ; Address at which the OP should start to proces the object-list
    swap    d0                  ; word-swap address-value for OP
    move.l  d0,OLP              ; now copy it into OP

    move.w  #$4C7,VMODE         ; Turn on the display (RGB)



    move.w  #CANVAS_WIDTH/2,xpos
    move.w  #CANVAS_HEIGHT/2,ypos

    move.w  #0,mouseX_last  ; sequence 0,2,1,-1 is positive
    move.w  #0,mouseY_last  ; sequence 0,-1,1,2 is negative

    ; prepare for Atari ST-mouse at mouse-adapter
    move.l  bitsXA+0,testbitXA
    move.l  bitsXB+0,testbitXB
    move.l  bitsYA+0,testbitYA
    move.l  bitsYB+0,testbitYB

    ; start the timer by writing a non-zero value to register PIT:
    ; (Don't do this in IntInit, because it will start immediately
    ; the timer, and perhaps not everything is setup at that time!)
    ; 1 second      = 26593900 = $195CA6C  (the frequenzy of the  system)
    ; 1/1200 second =    22162 = $0005692
    move.l  #$0005692,d0
    swap    d0
    move.l  d0,PIT


;----------------------------
mainloop:
    ;stop    #$2000         ; this lets the 68000er sleep and wakes it up again at _any_ interrupt,
                            ; so we can't use it here, becuase of the 1200Hz-Timer-Interrupt!
                            ; Instead: Use the polling for the variable changed by the VBL-Handler:
    jsr     wait_int        ; wait for VBL by polling a variable
    ;jsr     wait_int_and_poll    ; wait for VBL and poll the mouse while doing this!
    move.w  #TOTAL_OBJS,d6  ; number of Object Headers
    lea.l   obj_data,a1     ; beginning of object list
    jsr     build           ; rebuild the object list

    bsr     readpadport1    ; read joypad input from port1


.chkMouseKeys:
    move.l  joycur+4,d0     ; current button-state of mouse at port2!!

    btst.l  #FIRE_A,d0       ; left mouse-key (or VCS driving controller button)
    ;btst.l  #KEY_P,d0       ; right mouse-key
    beq.s   .chkJoypadReset
    bsr     clrscreen


.chkJoypadReset:
    move.l  joycur,d0       ; current button-state of joypad at port1!!
    ; Test for reset
    btst.l  #KEY_ASTERISK,d0
    beq     .chkJoypad1keys
    btst.l  #KEY_HASH,d0
    bne     .leavescreen

.chkJoypad1keys:
    move.l  joyedge,d0       ; changed button-state of joypad at port1!!

.chkJoypad1_key_1:
    btst.l  #KEY_1,d0
    beq.s   .chkJoypad1_key_2
    ; prepare for Atari ST-mouse at mouse-adapter
    move.l  bitsXA+0,testbitXA
    move.l  bitsXB+0,testbitXB
    move.l  bitsYA+0,testbitYA
    move.l  bitsYB+0,testbitYB
    move.b  #$01,fillcolour ; red for Atari ST-mouse
    bsr     clrscreen
    bra     mainloop

.chkJoypad1_key_2:
    btst.l  #KEY_2,d0
    beq.s   .chkJoypad1_key_3
    ; prepare for Amiga-mouse at mouse-adapter
    move.l  bitsXA+4,testbitXA
    move.l  bitsXB+4,testbitXB
    move.l  bitsYA+4,testbitYA
    move.l  bitsYB+4,testbitYB
    move.b  #$04,fillcolour ; blue for Amiga-mouse
    bsr     clrscreen
    bra     mainloop

.chkJoypad1_key_3:

.jmp_mainloop:
    bra     mainloop
;----------------------------


.leavescreen:

    move.l  #0,BJLtype      ; start 8bit-Loader
    bra.s   back
    nop

;************************* Spiel zu Ende **************************
back:

_reset:
    move.w  #$2700,sr
    moveq   #0,d0
    move.l  d0,$f00050  ; PIT
    move.l  d0,$F02114  ; G_CTRL
    move.l  d0,$f02100  ; G_FLAGS
.again:
    move.l  d0,$F1A114  ; D_CTRL (MOD-Player)
    move.l  $f1a114,d1  ; D_FLAGS
    btst    #0,d1
    bne.s   .again      ; wait for DSP being stopped
    move.l  d0,$f1a100  ; D_FLAGS

    move.l  d0,$F10000  ; JPI
    move.w  #$7fff,$f0004e  ; VI
    move.l  #$fF800000,$00F000E0    ; INT1

    move.l  #$FF00FF00,$F00020    ; DSP INT
    move.l  d0,$F1A154      ; SMODE
    move.l  d0,$F1A150      ; SCLK
    move.w  #$1861,$F00000      ; MEMCON1
    
 ;#####################################
 ;#####################################
;######    illegal ;#####################################
 ;#####################################
 ;#####################################

    lea     BJLLoader,a0
    lea     $1fc000,a1
    move.w  #(BJLLoaderEnd-BJLLoader)>>2,d0
.1:
    move.l  (a0)+,(a1)+
    dbra    d0,.1
    move.l  BJLtype,$8
    jmp     $1fc000



;===============================================================================================
clrscreen:
    movem.l d0-d1/a0,-(sp)

    ; clear our screen (CANVAS_HEIGHT*CANVAS_WIDTH 8bit)
    ;move.l  #$01010101,d0   ; four pixels with index 1!   red
    ;move.l  #$0f0f0f0f,d0   ; four pixels with index 15! white
    ;move.l  #$00000000,d0   ; four pixels with index 0!  black
    move.b  fillcolour,d0
    move.b  d0,d1
    lsl.l   #8,d0
    move.b  d1,d0
    lsl.l   #8,d0
    move.b  d1,d0
    lsl.l   #8,d0
    move.b  d1,d0

    lea     canvas_bmp,a0
    move.l  #((CANVAS_WIDTH/4)*CANVAS_HEIGHT)-1,d1
.clrloop:
    move.l  d0,(a0)+
    dbf     d1,.clrloop

    movem.l (sp)+,d0-d1/a0
	rts


;--------------------------------------------------------------------------
; Create 2 BRANCH-Objects
;--------------------------------------------------------------------------
; In: 
;    A0 ; Address for first BRANCH-object
;    D2 ; Address of the STOP-object to which the branches will lead
create_branch:
; First BRANCH-object (branch if YPOS < a_vdb)
    clr.l   d1
    move.l  #(BRANCHOBJ|O_BRGT),d0   ; YPOS < VC

; Create Format Link
    movem.l d2-d3,-(sp)
    andi.l  #$3FFFF8,d2     ; Ensure alignment/valid address
    move.l  d2,d3           ; Make a copy

    swap    d2              ; This section places bits 10-3
    clr.w   d2              ; in bits 31-24. It saves cycles
    lsl.l   #5,d2           ; over using three shifts.
    or.l    d2,d0

    lsr.l   #8,d3           ; Put bits 21-11 in bits 42-32
    lsr.l   #3,d3
    or.l    d3,d1
    movem.l (sp)+,d2-d3     ; Restore regs
; End of "Create Format Link"

    move.w  a_vdb,d3        ; upper visible screen edge
    lsl.w   #3,d3           ; Make it bits 13-3
    or.w    d3,d0

    move.l  d1,(a0)+
    move.l  d0,(a0)+        ; First OBJ is done.

; Second BRANCH-object (branch if YPOS > a_vde)
    andi.l  #$FF000007,d0   ; Mask off CC and YPOS
    ori.l   #O_BRLT,d0      ; $8000 = YPOS < VC
    move.w  a_vde,d3        ; lower visible screen edge
    lsl.w   #3,d3           ; Make it bits 13-3
    or.w    d3,d0

    move.l  d1,(a0)+        ; Second OBJ is done
    move.l  d0,(a0)+
    rts


;-------------------------------------------------------------------------
; This copies the BIN file at location (a0) to its intended destination.
;-------------------------------------------------------------------------
copy_binfile:
    move.l  (a0)+,a1    ; Get starting address (destination)
    move.l  (a0)+,d0    ; Get length of code segment...
    asr     #2,d0       ; Divide by 4 (do longs)
.loop:
    move.l  (a0)+,(a1)+
    dbra    d0,.loop
    rts

;================================================================================================
readpadport1::
;scan for player 1
 move.l #$f0fffffc,d1  ; d1 = Joypad data mask
 moveq.l #-1,d2   ; d2 = Cumulative joypad reading

 move.w #$81fe,JOYSTICK
 move.l JOYSTICK,d0  ; Read joypad, pause button, A button
 or.l d1,d0   ; Mask off unused bits
 ror.l #4,d0
 and.l d0,d2   ; d2 = xxAPxxxx RLDUxxxx xxxxxxxx xxxxxxxx
 move.w #$81fd,JOYSTICK
 move.l JOYSTICK,d0  ; Read *741 keys, B button
 or.l d1,d0   ; Mask off unused bits
 ror.l #8,d0
 and.l d0,d2   ; d2 = xxAPxxBx RLDU741* xxxxxxxx xxxxxxxx
 move.w #$81fb,JOYSTICK
 move.l JOYSTICK,d0  ; Read 2580 keys, C button
 or.l d1,d0   ; Mask off unused bits
 rol.l #6,d0
 rol.l #6,d0
 and.l d0,d2   ; d2 = xxAPxxBx RLDU741* xxCxxxxx 2580xxxx
 move.w #$81f7,JOYSTICK
 move.l JOYSTICK,d0  ; Read 369# keys, Option button
 or.l d1,d0   ; Mask off unused bits
 rol.l #8,d0
 and.l d0,d2   ; d2 = xxAPxxBx RLDU741* xxCxxxOx 2580369# <== inputs active low

 moveq.l #-1,d1
 eor.l d2,d1   ; d1 = xxAPxxBx RLDU741* xxCxxxOx 2580369# <== now inputs active high

 move.l joycur,d0  ; old joycur needed for determining the new joyedge
 move.l d1,joycur  ; Current joypad reading stored into joycur
 eor.l d1,d0
 and.l d1,d0
 move.l d0,joyedge  ;joypad, buttons, keys that were just pressed

 rts

;================================================================================================
readpadport2::
;scan for player 2
 move.l #$0ffffff3,d1  ; d1 = Joypad data mask
 moveq.l #-1,d2   ; d2 = Cumulative joypad reading

 move.w #$817f,JOYSTICK
 move.l JOYSTICK,d0  ; Read joypad, pause button, A button
 or.l d1,d0   ; Mask off unused bits
 rol.b #2,d0   ; note the size of rol
 ror.l #8,d0
 and.l d0,d2   ; d2 = xxAPxxxx RLDUxxxx xxxxxxxx xxxxxxxx
 move.w #$81bf,JOYSTICK
 move.l JOYSTICK,d0  ; Read *741 keys, B button
 or.l d1,d0   ; Mask off unused bits
 rol.b #2,d0   ; note the size of rol
 ror.l #8,d0
 ror.l #4,d0
 and.l d0,d2   ; d2 = xxAPxxBx RLDU741* xxxxxxxx xxxxxxxx
 move.w #$81df,JOYSTICK
 move.l JOYSTICK,d0  ; Read 2580 keys, C button
 or.l d1,d0   ; Mask off unused bits
 rol.b #2,d0   ; note the size of rol
 rol.l #8,d0
 and.l d0,d2   ; d2 = xxAPxxBx RLDU741* xxCxxxxx 2580xxxx
 move.w #$81ef,JOYSTICK
 move.l JOYSTICK,d0  ; Read 369# keys, Option button
 or.l d1,d0   ; Mask off unused bits
 rol.b #2,d0   ; note the size of rol
 rol.l #4,d0
 and.l d0,d2   ; d2 = xxAPxxBx RLDU741* xxCxxxOx 2580369# <== inputs active low

 moveq.l #-1,d1
 eor.l d2,d1   ; d1 = xxAPxxBx RLDU741* xxCxxxOx 2580369# <== now inputs active high

 move.l joycur+4,d0  ; old joycur needed for determining the new joyedge
 move.l d1,joycur+4  ; Current joypad reading stored into joycur
 eor.l d1,d0
 and.l d1,d0
 move.l d0,joyedge+4  ;joypad, buttons, keys that were just pressed

 rts


;***********************************************************************************
; This function checks the movement-input coming from the mouse or rotary controller.
; It converts the input into "normal" joypad-inputdata (like coming from the thumbpad).
; This is done by creating an index into a table holding 16 entries out of 
; the 2 current and the 2 old bits indicating the device-movement.
;
readmouse:  
    movem.l d0-d7/a0-a6,-(sp)
;scan for player 1:
;   moveq.l #-1,d2              ; d2 = Cumulative joypad reading
;
;   move.l  #$f0fffffc,d1   ; d1 = Joypad data mask of Socket A of Port 1
;   move.w  #$81fe,JOYSTICK ; Socket A of Port 1
;   move.l  JOYSTICK,d0 ; Read joypad,pause button,A button
;   or.l    d1,d0       ; Mask off unused bits
;   ror.l   #4,d0
;   and.l   d0,d2       ; d2 = xxAPxxxx RLDUxxxx xxxxxxxx xxxxxxxx
;
;   moveq.l #-1,d1
;   eor.l   d2,d1               ; d1 = xxAPxxBx RLDU741* xxCxxxOx 2580369# <== now inputs active high

;scan for player 2:
    moveq.l #-1,d2              ; d3 = Cumulative joypad reading

    move.l  #$0ffffff3,d1   ; d1 = Joypad data mask of Socket A of Port 2
    move.w  #$817f,JOYSTICK ; Socket A of Port 2
    move.l  JOYSTICK,d0 ; Read joypad,pause button,A button
    or.l    d1,d0       ; Mask off unused bits
    rol.b   #2,d0       ; note the size of rol
    ror.l   #8,d0
    and.l   d0,d2       ; d2 = xxAPxxxx RLDUxxxx xxxxxxxx xxxxxxxx

    moveq.l #-1,d1
    eor.l   d2,d1               ; d1 = xxAPxxBx RLDU741* xxCxxxOx 2580369# <== now inputs active high
    

    ; the mouse/rotary-specific part:
    move.l  d1,d2           ; save input
    and.l   #$30000000,d2   ; we can only use the 2 buttons A and P directly!


    moveq.l #0,d4
    move.w  mouseY_last,d3
    lsl.w   #2,d3

    move.l  testbitYA,d7
    btst.l  d7,d1

    beq.s   m_l             ; Joyport-input pins for buttons R and L are tied to YA and YB of the mouse-adapter
    bset.l  #0,d3

m_l:    
    move.l  testbitYB,d7
    btst.l  d7,d1

    beq.s   m_rl
    bset.l  #1,d3

m_rl:   
    and.w   #$000f,d3
    move.w  d3,d4
    lea     dirtab_vertical,a0
    asl.l   #2,d3       ; *4
    move.l  (a0,d3.w),d3
    bmi.s   .skip
    bset.l  d3,d2
.skip:

m_u:    
    moveq.l #0,d6
    move.w  mouseX_last,d5  ;  Joyport-input pins for buttons U and D are tied to XA and XB of the mouse-adapter
    lsl.w   #2,d5

    move.l  testbitXA,d7
    btst.l  d7,d1

    beq.s   m_d
    bset.l  #0,d5

m_d:    
    move.l  testbitXB,d7
    btst.l  d7,d1

    beq.s   m_ud
    bset.l  #1,d5

m_ud:
    and.w   #$000f,d5
    move.w  d5,d6
    asl.l   #2,d5       ; *4
    lea     dirtab_horizontal,a0
    move.l  (a0,d5.w),d5
    bmi.s   .skip
    bset.l  d5,d2
.skip:



m_end:  
    move.w  d4,mouseY_last
    move.w  d6,mouseX_last

    move.l  d2,d1
    ;move.l  joycur,d0      ; for port1: old joycur needed for determining the new joyedge
    move.l  joycur+4,d0     ; for port2: old joycur needed for determining the new joyedge
    ;move.l  d1,joycur      ; for port1: Current joypad reading stored into joycur
    move.l  d1,joycur+4     ; for port2: Current joypad reading stored into joycur
    eor.l   d1,d0
    and.l   d1,d0
    ;move.l  d0,joyedge     ; for port1: joypad,buttons,keys that were just pressed
    move.l  d0,joyedge+4    ; for port2: joypad,buttons,keys that were just pressed

    movem.l (sp)+,d0-d7/a0-a6
    rts       

;*****************************************************************
;*******************************************************************
; Turn joypad-input into action
; Here: We position the Mouxe-cursor-sprite according to the detected mouse-input
; (but only within the 200*320 rectangle of the canvas-sprite).
; Additonally we draw a pixel at the new mouse-position onto the canvas.
chkmouseaction: 
    movem.l   d0-d7/a0-a4,-(sp)
    move.w  xpos,d1
    move.w  ypos,d2
    ;move.l  joycur,d6       ; Port A
    move.l joycur+4,d6     ; Port B

keyLeft:
    btst.l  #KEY_L,d6
    beq.s   keyRight
    subq.w  #1,d1
    bpl.s   keyRight
    clr.w   d1
keyRight:
    btst.l  #KEY_R,d6
    beq.s   keyUp
    addq.w  #1,d1
    cmp.w   #CANVAS_WIDTH-1,d1
    blt.s   keyUp
    move.w  #CANVAS_WIDTH-1,d1
keyUp:
    btst.l  #KEY_U,d6
    beq.s   keyDown
    subq.w  #1,d2
    bpl.s   keyDown
    clr.w   d2
keyDown:
    btst.l  #KEY_D,d6
    beq.s   chkend
    addq.w  #1,d2
    cmp.w   #CANVAS_HEIGHT-1,d2
    blt.s   chkend
    move.w  #CANVAS_HEIGHT-1,d2
chkend:
    move.w  d1,xpos
    move.w  d2,ypos


    ; show our mouse-cursor at new position:
    move.w  xpos,d0
    add.w   obj_data+0,d0
    move.w  d0,curx
    move.w  ypos,d0
    add.w   d0,d0       ; verdoppeln
    add.w   obj_data+2,d0
    move.w  d0,cury


    ; and now draw a pixel at the new cursor-position:
    moveq.l #0,d0
    moveq.l #0,d1
    lea     canvas_bmp,a0
    move.w  xpos,d0
    move.w  ypos,d1
    mulu    #CANVAS_WIDTH,d1
    add.l   d0,d1
    add.l   d1,a0
    move.b  #$0f,(a0)   ; $0f = draw white pixel


    movem.l             (sp)+,d0-d7/a0-a4
    rts       

;================================================================================================

build::              ;pack the object data into olist_ram
;A1 = need address of first object
;D6 = total number of objects to be processed

;register use
;A0 is the address of the object list used by the object processor
;A2 is the address of the object list being built
;D0 holds the long being built
;D1 is used for temporary calculations
;D2 is the lines off screen,calculated in set_odata,used by set_oheight
;D3 is #12 for shifting d0
;D4 is #10 for shifting d0
;D5 is #11 for shifting d0
;D6 counts the objects being processed
;D7 is the address of the boject used by the object processor divided by 8
    movem.l d0-d7/a0-a3,-(sp)

          moveq.l   #12,d3              ;for doing lsl.l on d0
          moveq.l   #10,d4              ;for doing lsl.l on d0
          moveq.l   #11,d5              ;for doing lsl.l on d0
          move.l    #olist,d7           ;address of packed olist
          lsr.l     #3,d7               ;divide by 8 for object processor
          lea.l     olist_ram,a2        ;address of copy of packed olist
          subi.w    #1,d6               ;start dbra counter at total objects - 1
proc_obj:           ;process an object
          tst.b     O_TYPE(a1)          ;check the delete flag for this object
          bmi       next_obj            ;skip this object if negative bit is set

;PHRASE 1
set_odata:          
;if the object is above top line of the screen,
;the data pointer is set to the first line of the object that is on screen
          move.l    O_DATA(a1),d0       ;get normal data address
          ;move.w    #SCRN_TOP-1,d2
          ;move.w    #$2e-1,d2
;          move.w    a_vdb,d2
;          move.w    O_YPOS(a1),d1
;          ;sub.w    O_YPOS(a1),d2       ;top of screen - ypos = half lines off top of screen
;          sub.w     d1,d2
;          bpl.s     .05                 ;if object is above top of screen,use normal data address
    
.01:      clr.w     d2                  ;0 lines off screen,use normal data address
          bra.s     .10
.05:      lsr.w     #1,d2               ;convert half lines to lines
          move.w    O_DWIDTH(a1),d1     ;get phrases per line of data in the object
          lsl.w     #3,d1               ;multiply phrases * 8 bytes per phrase
          mulu.w    d2,d1               ;lines off screen * bytes per line in the object
          add.l     d1,d0               ;skip the total bytes that are off the top of the screen
.10:      lsr.l     #3,d0               ;divide by 8
          lsl.l     d5,d0               ;shift left 11 bits to make room for link pointer

;set the link pointer to the next object
          addi.l    #4,d7               ;next object is 4 phrases ahead
          move.l    d7,d1               ;get address in olist of next object

          lsr.l     #8,d1               ;the rightmost 8 bits go in the next long,so remove them
          or.l      d1,d0               ;or with data pointer
          move.l    d0,(a2)+            ;save first long in phrase 1,advance the pointer to next long

          move.b    d7,d0               ;get the low 8 bits of the link pointer that was saved earlier
          lsl.l     d4,d0               ;shift left 10 bits to make room for height

set_oheight:        
;d2.w is set to the number of lines that are above the top of the screen
;if part of the object is above the top line of the screen,
;the height is reduced by the number of lines off the top of the screen
          move.w    O_HEIGHT(a1),d1     ;get the height of the object
          cmpi.b    #1,O_TYPE(a1)       ;see if this is a scaled bit map object
          bne.s     .10                 ;branch if object is not scaled
          subi.w    #1,d1               ;an extra line is displayed at the bottom of the object if this is not here
.10:      sub.w     d2,d1               ;subtract lines off screen from scaled height
          bmi.s     .20                 ;if lines off screen > height,make height 0 instead of negative
          or.w      d1,d0               ;or with link pointer in second half of phrase 1
.20:      lsl.l     d5,d0               ;shift left 11 bits to make room for ypos

set_oypos:          
;if ypos is negative,it gets set to 0
;set_odata and set_oheight will adjust the data pointer and height accordingly
          move.w    O_YPOS(a1),d1
          bmi.s     .10                 ;if ypos is negative,object would not be displayed,so set ypos to 0
          andi.w    #$7ff,d1            ;only write 11 bits for ypos
          or.w      d1,d0               ;or with link pointer and height
.10:      lsl.l     #3,d0               ;shift left 3 bits to make room for type

set_otype:          
          or.b      O_TYPE(a1),d0       ;object type
          move.l    d0,(a2)+            ;write second half of phrase 1

;PHRASE 2
set_ofirstpix:      
          move.b    O_FIRSTPIX(a1),d0
          lsl.l     #4,d0               ;shift left 4 bits to make room for release,transparent,rmw,reflect bits

set_flags:          
          or.b      O_FLAGS(a1),d0      ;or with firstpix
          lsl.l     #7,d0               ;shift left 7 bits to make room for index

set_oindex:         
          or.b      O_INDEX(a1),d0      ;or with firstpix and flags
          lsl.l     #6,d0               ;shift left 6 bits to make room for part of iwidth

set_oiwidth:        
          move.w    O_IWIDTH(a1),d1
          lsr.w     #4,d1               ;right 4 bits go in the next long,so remove them
          or.b      d1,d0
          move.l    d0,(a2)+            ;write first long of phrase 2

          move.w    O_IWIDTH(a1),d0
          andi.w    #$f,d0              ;get low 4 bits of iwidth
          lsl.l     d4,d0               ;shift left 10 bits to make room for dwidth

set_odwidth:        
          or.w      O_DWIDTH(a1),d0     ;or with part of iwidth
          lsl.l     #3,d0               ;shift left 3 bits to make room for pitch

set_opitch:         
          or.b      O_PITCH(a1),d0      ;or with dwidth and iwidth
          lsl.l     #3,d0               ;shift left 3 bits to make room for depth

set_odepth:         
          or.b      O_DEPTH(a1),d0      ;or with dwidth,iwidth,pitch
          lsl.l     d3,d0               ;shift left 12 bits to make room for xpos

set_oxpos:          
          move.w    O_XPOS(a1),d1       ;object xpos
          andi.w    #$fff,d1            ;only write 12 bits for xpos
          or.w      d1,d0               ;or with the rest of the stuff
          move.l    d0,(a2)+            ;write second half of phrase 2

;PHRASE 3
;for scaled bit map object only
          move.l    O_SCALE(a1),4(a2)   ;write remainder,vscale,hscale bytes in 2nd long of phrase 3
          adda.l    #16,a2              ;move pointer past the last 2 phrases for current object

next_obj:           ;jump to here if delete flag is on
          adda.l    #OBJSIZE,a1         ;point to next object in pseudo object list
          dbra      d6,proc_obj         ;repeat until all objects have been processed

;write a stop object at the end of the list
          move.l    #0,(a2)+
          move.l    #4,(a2)             ;write object type 4 (stop object)

    movem.l (sp)+,d0-d7/a0-a3

          rts       


;***************************************************************************

copy_olist::
    lea.l   olist_ram,a0  ;beginning of olist_ram
    lea.l   olist,a1  ;beginning of olist
.10:
    move.l  (a0),(a1)+  ;copy olist_ram to olist
    cmpi.l  #4,(a0)+  ;see if we are at the stop object
    bne.b   .10
    rts



;***************************************************************************
set_palette::
;A0 must be the address of the palette data
;palette data is 256 words of colors
    lea.l   CLUT,a1          ;get address of color lookup table
    move.w  #255,d0   ;number of colors to set
.10:
    move.w  (a0)+,(a1)+  ;put color value into color lookup table
    dbra    d0,.10   ;continue through palette table
    rts

;***************************************************************************

wait_int::
    movem.l d7,-(sp)        

    move.b  framecnt,d7
.10:
    cmp.b   intcount,d7
    bgt.b   .10
    clr.b   intcount

    movem.l (sp)+,d7
    rts

;-------------------------------------------------------------------------------
wait_int_and_poll: 
    movem.l d7,-(sp)        

    move.b  framecnt,d7
.10:
    bsr     readmouse   ; poll the mouse while waiting for the next VBL
    bsr     chkmouseaction
    cmp.b   intcount,d7
    bgt.s   .10
    clr.b   intcount

    movem.l (sp)+,d7
    rts       
;-------------------------------------------------------------------------------


IntInit::
    move.l  #Frame,LEVEL2
    ;***move.w  #n_vde+1,VI
    move.w  a_vde,d0
    ori.w   #1,d0    ; must be odd!!!!
    move.w  d0,VI


    ;move.w  #1,INT1    ; VBL only
    move.w  #9,INT1     ; VBL + PIT
    move.w  sr,d0
    and.w   #$f8ff,d0
    move.w  d0,sr
    rts

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
Frame::
    movem.l d0-d6/a0-a6,-(sp)

    move.w  INT1,d0         

    move.w  d0,d1
    lsl.w   #8,d1
    or.w    #$09,d1         ; VBL + PIT
    move.w  d1,INT1         ; mark interrupts as serviced, and re-enable them

    btst    #3,d0           ; bit3 = a timer-IRQ?
    beq.s   no_timer
is_timer:
    bsr     readmouse       
    bsr     chkmouseaction

no_timer:
    btst    #0,d0           ; bit0 = a VBL?
    beq.s   no_blank
is_blank:
    jsr     copy_olist      ; changes a0+a1 !
    add.b   #1,intcount
    
no_blank:
    move.w  #0,INT2
    movem.l (sp)+,d0-d6/a0-a6
    rte

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
; Procedure: InitVideo (same as in vidinit.s)
;            Build values for hdb, hde, vdb, and vde and store them.
;
       
InitVideo:
  move.l d0,-(sp)  ; Save the one register we use
 
  move.w CONFIG,d0  ; Also is joystick register
  andi.w #VIDTYPE,d0  ; 0 = PAL, 1 = NTSC
  beq palvals

  move.w #NTSC_HMID,d2
  move.w #NTSC_WIDTH,d0

  move.w #NTSC_VMID,d6
  move.w #NTSC_HEIGHT,d4

  bra calc_vals
palvals:
  move.w #PAL_HMID,d2
  move.w #PAL_WIDTH,d0

  move.w #PAL_VMID,d6
  move.w #PAL_HEIGHT,d4

calc_vals:
  move.w d0,width
  move.w d4,height
  move.w d2,hormid	; DO, new!
  move.w d6,vermid

  move.w d0,d1
  asr #1,d1   ; Width/2

  sub.w d1,d2   ; Mid - Width/2
  add.w #4,d2   ; (Mid - Width/2)+4

  sub.w #1,d1   ; Width/2 - 1
  ori.w #$400,d1  ; (Width/2 - 1)|$400
  
  move.w d1,a_hde
  move.w d1,HDE

  move.w d2,a_hdb
  move.w d2,HDB1
  move.w d2,HDB2

  move.w d6,d5
  sub.w d4,d5
  move.w d5,a_vdb

  add.w d4,d6
  move.w d6,a_vde

  move.w a_vdb,VDB
  move.w #$FFFF,VDE
   
  move.l #0,BORD1  ; Black border
  move.w #0,BG   ; Init line buffer to black
   
  move.l (sp)+,d0
  rts
  


;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Uninitialized Data!!!
;;;;;;;;;;;;;;;;;;;;;;;;;;

  .BSS

a_hdb:  .ds.w 1
a_hde:  .ds.w 1
a_vdb:  .ds.w 1
a_vde:  .ds.w 1
width:  .ds.w 1
height: .ds.w 1
hormid: .ds.w 1
vermid: .ds.w 1

    EVEN
mouseX_last:    ds.w    1
mouseY_last:    ds.w    1

xpos:       ds.w 1
ypos:       ds.w 1

    even
BJLtype:    ds.l    1


    dphrase
canvas_bmp: ds.b    CANVAS_HEIGHT*CANVAS_WIDTH

 .DATA

;-----------------------------------------
; The object-lists:
; 1 template and 1 which the OP processes
;-----------------------------------------
    .qphrase

      rept 4    ; 4 longs = 2 phrases
      dc.l 0    ; !!!! NEW NEW NEW Scaled objects need 4 phrase alignment NEW NEW NEW !!!!
      endr
    
olist1_branch:
      rept 4
      dc.l 0   ; 2 Branch Objekte   = 2 * 1 Phrase = 2*8 = 16 bytes = 4 longs
      endr

olist::
      ;object list used by object processor
      ;need 8 longs for each object
      ;and 2 longs for a stop object
      rept ((8*TOTAL_OBJS)+2)
      dc.l 0
      endr
    
    olist1_stop:
      dc.l 0   ; 1. part of STOP-object
      dc.l 4   ; 2. part of STOP-object

;-----------------------------------------
    .qphrase
      rept 4    ; 4 longs = 2 phrases
      dc.l 0    ; !!!! NEW NEW NEW Scaled objects need 4 phrase alignment NEW NEW NEW !!!!
      endr
    
olist2_branch:
      rept 4
      dc.l 0   ; 2 Branch Objekte   = 2 * 1 Phrase = 2*8 = 16 bytes = 4 longs
      endr

olist_ram::
      ;packed version of objects in obj_data
      ;need 8 longs for each object
      ;and 2 longs for a stop object
      rept ((8*TOTAL_OBJS)+2)
      dc.l 0
      endr
    
olist2_stop:
      dc.l 0   ; 1. part of STOP-objekt
      dc.l 4   ; 2. part of STOP-objekt


;------------------------------------------
    .phrase
mypal:
    dc.w    $0000   ; black
    dc.w    $8000   ; red
    dc.w    $0020   ; green
    dc.w    $8020   ; yellow
    dc.w    $0400
    dc.w    $8400
    dc.w    $0420
    dc.w    $C630
    dc.w    $8420
    dc.w    $F000
    dc.w    $003C
    dc.w    $F03C
    dc.w    $0780
    dc.w    $F780
    dc.w    $07BC
    dc.w    $F7BC   ; white

    ; now the rest of the 256 CLUT-entries:
    .rept (256-16)
    dc.w    $ffff   ; white
    .endr

;================================================================================================

    dphrase
cursor_bmp:  ; 4bit-pixel (white outline, green filling with our palette)
    dc.l    $ff000000, $000000000
    dc.l    $f2ff0000, $000000000
    dc.l    $f222ff00, $000000000
    dc.l    $f22222ff, $000000000
    dc.l    $f2222222, $ff0000000
    dc.l    $f2222222, $22ff00000
    dc.l    $f2222222, $ff0000000
    dc.l    $f222222f, $000000000
    dc.l    $f222f222, $f00000000
    dc.l    $f2ff0f22, $2f0000000
    dc.l    $ff0000f2, $22f000000
    dc.l    $0000000f, $222f00000
    dc.l    $00000000, $f222f0000
    dc.l    $00000000, $0f222f000
    dc.l    $00000000, $00f22ff00
    dc.l    $00000000, $000ff0000

                                                      

;================================================================================================
;  Object List Headers

        .qphrase
obj_data::
        dc.w    CURSOR_X  ; o_xpos 
        dc.w    CURSOR_Y  ; o_ypos
        dc.l    canvas_bmp   ; o_data
        dc.w    CANVAS_HEIGHT      ; o_height
        dc.w    CANVAS_WIDTH/8     ; o_dwidth
        dc.w    CANVAS_WIDTH/8     ; o_iwidth
        dc.b    0   ; o_flags
        dc.b    0   ; o_firstpix
        dc.b    0   ; o_type
        dc.b    3   ; o_depth      ; 8bit
        dc.b    1   ; o_pitch
        dc.b    0   ; o_index
        dc.w    0   ; o_desc
        dc.l    $00202020  ; scaling info
        dc.l    0   ; o_rom
        dc.l    0   ; o_mode


;***Cursor***********
curx:     DC.w CURSOR_X   ; o_xpos
cury:     DC.w CURSOR_Y   ; o_ypos
curadr:   DC.l cursor_bmp ; o_data
          DC.w 16   ; o_height
          DC.w 16/2/8 ; o_dwidth
          DC.w 16/2/8 ; o_iwidth
          DC.b 4    ; o_flags   ; Transparent!!
          DC.b 0    ; o_firstpix
          DC.b 0    ; o_type    ; Bitmap (not scaled)
          DC.b 2    ; o_depth   ; 4bit/pixel
          DC.b 1    ; o_pitch
          DC.b 0    ; o_index
          DC.w 0    ; o_desc
          DC.w 20
          DC.b $20
          DC.b $20
          DC.l 0   ; o_rom
          DC.l 0   ; o_mode


;---------------------------------------------------------------------
 .phrase
intcount::  dc.b    0
framecnt::  dc.b    1           ;60 frames per second

fillcolour: dc.b    $01

    even
joyedge::       dc.l        0,0     ; shows changes between last and current button-states
joycur::        dc.l        0,0     ; currently depressed joypad-buttons


    even
AxisTestbits:
bitsYA:
    dc.l  KEY_R       ; Atari ST-Mouse: 
    dc.l  KEY_U       ; Amiga-Mouse: 
bitsYB:
    dc.l  KEY_L       ; Atari ST-Mouse
    dc.l  KEY_L       ; Amiga-Mouse

bitsXA:
    dc.l  KEY_D       ; Atari-Mouse
    dc.l  KEY_D       ; Amiga-Mouse
bitsXB:
    dc.l  KEY_U       ; Atari ST-Mouse
    dc.l  KEY_R       ; Amiga-Mouse


; these 4 variables are used in function "readmouse":
testbitXA:  dc.l    KEY_D
testbitXB:  dc.l    KEY_U
testbitYA:  dc.l    KEY_R
testbitYB:  dc.l    KEY_L



    even

; Possible combinations for axis-input:
; XA,XB
; 0,0
; 0,1
; 1,0
; 1,1

; All possible combinations of old and new axis-input:
; New    Old
; 0,0  ; 0,0  --> no movement  
       ; 0,1  --> negative
       ; 1,0  --> positive
       ; 1,1  --> undetermined  

; 0,1  ; 0,0  --> positive
       ; 0,1  --> no movement  
       ; 1,0  --> undetermined
       ; 1,1  --> negative
       
; 1,0  ; 0,0  --> negative
       ; 0,1  --> no movement  
       ; 1,0  --> undetermined
       ; 1,1  --> positive
       
; 1,1  ; 0,0  --> undetermined
       ; 0,1  --> positive
       ; 1,0  --> negative
       ; 1,1  --> no movement

; And now two tables for these old/new-combos for the two axis:
dirtab_horizontal:
    dc.l -1
    dc.l KEY_R
    dc.l KEY_L
    dc.l -1

    dc.l KEY_L
    dc.l -1
    dc.l -1
    dc.l KEY_R

    dc.l KEY_R
    dc.l -1
    dc.l -1
    dc.l KEY_L

    dc.l -1
    dc.l KEY_L
    dc.l KEY_R
    dc.l -1

dirtab_vertical:
    dc.l -1
    dc.l KEY_D
    dc.l KEY_U
    dc.l -1

    dc.l KEY_U
    dc.l -1
    dc.l -1
    dc.l KEY_D

    dc.l KEY_D
    dc.l -1
    dc.l -1
    dc.l KEY_U

    dc.l -1
    dc.l KEY_U
    dc.l KEY_D
    dc.l -1


    .QPHRASE
BJLLoader:  
    ;incbin  "loader8.bin"
    incbin  "loader.bin"  ; with the white text on gray background
BJLLoaderEnd:

 .end

