/*******************************************************************
 *
 *    TITLE:        b2k_game.c
 *
 *    DESCRIPTION:	Game handling code for Breakout 2000
 *
 *    AUTHOR:		Mario Perdue & Richard Degler
 *
 *    HISTORY:    
 *
 *
 *           COPYRIGHT 1994, 1995 MP Graphics Systems         
 *      UNATHORIZED REPRODUCTION, ADAPTATION, DISTRIBUTION,
 *      PERFORMANCE OR DISPLAY OF THIS COMPUTER PROGRAM OR   
 *     THE ASSOCIATED AUDIOVISUAL WORK IS STRICTLY PROHIBITED.
 *                        ALL RIGHTS RESERVED.               
 *******************************************************************/

/** include files **/
#include "\jaguar\include\jaguar.h"
#include "player.h"
#include "b2k_game.h"
#include "brick.h"
#include "ball.h"
#include "paddle.h"
#include "goody.h"
#include "droid.h"
#include "displays.h"
#include "support.h"
#include "jagobj.h"
#include "star.h"
#include "bomb.h"
#include "option.h"
#include "ripple.h"
#include "hiscore.h"


/** external data **/
extern	DEF_PLAYER		Player[2];
extern	DEF_DROID		Droid;

extern	ULONG			ticks;

extern	char 			NUM_SCREENS;

extern  DEF_SCREEN		ScreenDef[];

extern	DEF_RIPPLE		Ripple[MAX_RIPPLES];
extern	DEF_DROPBRICK	DropBrick[MAX_DROPBRICKS];

extern 	BYTE			PlayerOnTop;

extern	int  			BallMaxSpeedZ;
extern	int				BallSpeedLimit;

extern	long 			BlitAddIndex;
extern	long 			BlitUseIndex;

extern	WORD 			ObjectListFlag;

extern	int				BallRadius;

/** internal data **/
int						ScreenCount;
int						Screen;

int						OldBkg;

DEF_OPTION				Option;
ULONG					OldTicks;
ULONG					ElapsedTime;
ULONG					DemoTimer;

extern unsigned short 	height;

char 					TitleExpand;

/** internal functions **/
/*********************************************************************
 *  FUNCTION:		PlayGame
 *
 *  PARAMETERS:		short who
 *
 *  DESCRIPTION:	Breakout game handler
 *
 *  RETURNS:		void
 *
 *********************************************************************/
void PlayGame()
	{
	if(abs(ticks-OldTicks)<40)
		return;

	OldTicks = ticks;

	HandlePlayer(PLAYER_RED);
	HandlePlayer(PLAYER_GREEN);

/*	while(BlitAddIndex != BlitUseIndex);*/

	EraseSprites();
	DrawSprites();
	DisplaySprites();

	AnimateRipples();
	AnimateDropBricks();

	MoveDroid();

	UpdateScore(PLAYER_RED, FALSE);
	UpdateScore(PLAYER_GREEN, FALSE);

	ObjectListFlag = TRUE;
	}


/*********************************************************************
 *  FUNCTION:		HandlePlayer
 *
 *  PARAMETERS:		short who
 *
 *  DESCRIPTION:	Breakout player handler
 *
 *  RETURNS:		void
 *
 *********************************************************************/
void HandlePlayer(short who)
	{
	int i;

	BYTE Reset;

	Player[who].OldKeypad = Player[who].Keypad;

	Player[who].Keypad = ReadKeyPad(who);
	Player[who].Button = ReadButtons(who);
	Player[who].Joypad = ReadJoyStick(who);

	switch(Player[who].Mode)
		{
		case modeIDLE:
			Player[who].Draw = FALSE;
			return;
			break;

		case modeTITLE:
			Player[who].Draw = FALSE;
			ObjectListFlag = TRUE;
			DoTitle();

			if(!DemoTimer)
				{
				DoOptions();
				Player[PLAYER_RED].Timer = ticks;
				Player[PLAYER_GREEN].Timer = ticks;
				}

   			return;
			break;

		case modeSTARTGAME:
			Player[who].Draw = FALSE;
			ObjectListFlag = TRUE;

			InitPlayer(who);
			Player[who].Goody.Active = FALSE;
			Droid.Active = FALSE;
			Player[who].Bomb.Active = FALSE;

			if(Option.GameMode == gameCLASSIC)
				{
				Option.NumPlayers = 1;
				BallRadius = CLASSIC_R;
				}
			else
				BallRadius = BALL_R;

			if((who == PLAYER_GREEN) && (Option.NumPlayers < 2))
				{
				Player[who].Mode = modeIDLE;
				return;
				}
			else
				{
 				OldBkg = -1;

				if(DemoTimer == 0)
					{
					Screen = 0;
					ScreenCount = 0;
					}

				Player[who].Mode = modeSTARTSCREEN;
				Player[who].Paddle.Active = TRUE;
				}
			return;
			break;
		
		case modeSTARTSCREEN:
			if(Option.GameMode == gameCLASSIC)
				{
				/* this is redundant, but I'm putting it here anyway */
				Option.NumPlayers = 1;

				if(ScreenCount >= 2)
					{
					Player[who].Mode = modeENDGAME;
					return;
					}
				}

			Player[PLAYER_RED].Draw = FALSE;
			Player[PLAYER_GREEN].Draw = FALSE;

			ObjectListFlag = TRUE;

			StopMusic();

 			DoTransition();

 			BlankBomb(PLAYER_RED);                                 
 			BlankBomb(PLAYER_GREEN);                               
                                                                   
 			if(Option.NumPlayers == 2)                             
 				{                      
 				BallSpeedLimit = BALL_SPEED_LIMIT_2;
 				                            
 				if(Player[PLAYER_RED].Paddle.Active &&             
 				   Player[PLAYER_GREEN].Paddle.Active)             
 					{                                              
 					PlayerOnTop = 1-Screen%2;                      
 					Option.ScreenWrapOn      = TRUE;               
 					SetPlayfieldBack(ROUND);                       
 					}                                              
 				else if(Player[PLAYER_RED].Paddle.Active)          
 					{                                              
 					PlayerOnTop = PLAYER_GREEN;                    
 					Option.ScreenWrapOn      = FALSE;              
 					SetPlayfieldBack(FLAT);                        
 					}                                              
 				else                                               
 					{                                              
 					PlayerOnTop = PLAYER_RED;                      
 					Option.ScreenWrapOn      = FALSE;              
 					SetPlayfieldBack(FLAT);                        
 					}                                              
 				}                                                  
 			else
 				{                                                   
				BallSpeedLimit = BALL_SPEED_LIMIT_1;
 				PlayerOnTop = 1;
				Option.ScreenWrapOn      = FALSE;              
				SetPlayfieldBack(FLAT);                        
 				}     
                                                                
			DrawPaddleFace(PLAYER_ON_TOP);
			DrawPaddleFrame(PLAYER_ON_TOP);
			DrawPaddleStrobe(PLAYER_ON_TOP);
			DrawPaddleGun(PLAYER_ON_TOP);
			DrawPaddlePower(PLAYER_ON_TOP);
            
			DrawPaddleFace(PLAYER_ON_BOTTOM);
			DrawPaddleFrame(PLAYER_ON_BOTTOM);
			DrawPaddleStrobe(PLAYER_ON_BOTTOM);
			DrawPaddleGun(PLAYER_ON_BOTTOM);
			DrawPaddlePower(PLAYER_ON_BOTTOM);

			SetPaddleHover(PLAYER_RED, 32);
			SetPaddleHover(PLAYER_GREEN, 32);
			MovePaddle(PLAYER_RED);
			MovePaddle(PLAYER_GREEN);
            
            while(BlitAddIndex != BlitUseIndex);
                                       
 			for(i=PLAYER_RED; i<Option.NumPlayers; i++)            
 				{
				if(Option.GameMode == gameCLASSIC)
	 				SetPaddleSize(i, PADDLE_MAX_SIZE-1); 
				else
	 				SetPaddleSize(i, PADDLE_MAX_SIZE-Option.Skill[i]); 

 				if(Player[i].BallCount > 0)
					{                
					Player[i].Timer = ticks;        
					Player[i].Mode = modeSHOOT;                    
					Player[i].Draw = TRUE;
					}
 				else 
					{                                              
 					Player[i].Mode = modeIDLE;                     
					Player[i].Draw = FALSE;
					}
 				}                                                  

			ObjectListFlag = TRUE;
	
 			LoadBricks();

 			StartMusic(((ScreenCount/5)%10)+songPHASE0);                        
			Player[PLAYER_RED].Timer = ticks;                           
			Player[PLAYER_GREEN].Timer = ticks;                           

			RefreshScoreBoard();
			return;
			break;
		  
		case modeRESUMESCREEN:
			Player[who].Draw = TRUE;

			if(Player[who].Paddle.Cracks >= MAX_CRACKS)
				Player[who].Paddle.Cracks = 0;
			SetPaddleSize(who, PADDLE_MAX_SIZE-Option.Skill[who]);
			Player[who].Timer = ticks;
			Player[who].Mode = modeSHOOT;
			break;
				
		case modeENDSCREEN:
			Player[who].Draw = TRUE;

			for (i=0; i<MAX_BALLS; i++)
				BlankBall(who, i);

			for(i=0; i<MAX_PLAYERS; i++)
				BlankBomb(i);

			if((Option.GameMode == gameCLASSIC) && (ScreenCount >= 2))
				{
				Player[who].Mode = modeENDGAME;
				}
			else
				{
				ElapsedTime = ticks;
				Player[who].Mode = modeDOBONUS;
				}
			break;

		case modeDOBONUS:
			Player[who].Draw = TRUE;

			if(!SpritesAreActive())
				{
				if((ticks-ElapsedTime) > 60)
					{
					AddPaddleBonus(who);
					if(Option.NumPlayers >= 2)
						{
						for (i=0; i<MAX_BALLS; i++)
							BlankBall(who^1, i);

						DoBrickPenalty(who^1);
						}

					ScreenCount++;
					Screen = ScreenCount%NUM_SCREENS;

					if(ScreenCount%5 == 0)
						{
						ClearPowerUps(PLAYER_RED);
						ClearPowerUps(PLAYER_GREEN);
						RefreshScoreBoard();
						if(ScreenCount%2 == 1)
							Player[PLAYER_RED].Mode = modePONGSTART;
						else
							Player[PLAYER_RED].Mode = modeTARGETSTART;
						Player[PLAYER_GREEN].Mode = modeIDLE;
						return;
						}

					if(Player[PLAYER_RED].Mode == modeENDGAME)
						Player[PLAYER_RED].Mode = modeIDLE;
					else
						Player[PLAYER_RED].Mode = modeSTARTSCREEN;

					if(Option.NumPlayers > 1)
						{
						if(Player[PLAYER_GREEN].Mode == modeENDGAME)
							Player[PLAYER_GREEN].Mode = modeIDLE;
						else
							Player[PLAYER_GREEN].Mode = modeSTARTSCREEN;
						}
					}
				}
			break;										  

		case modeSHOOT:
			Player[who].Draw = TRUE;

			if(Player[who].BrickCount <= 0)
				{
				if(Player[who].BallCount > 0)
					Player[who].Mode = modeENDSCREEN;
				}

			if(LaunchBalls(who))
				Player[who].Mode = modePLAY;

			if(Option.GameMode == gameCLASSIC)
				{
				Player[PLAYER_RED].Paddle.Catch = Option.Catch;
				if(Option.Breakthru)
					Player[PLAYER_RED].Ball[0].Style = ballBREAKTHRU;
				else
					Player[PLAYER_RED].Ball[0].Style = ballCLASSIC;
				}
			break;
	   
		case modePLAY:
			Player[who].Draw = TRUE;

			if(Player[who].BrickCount <= 0)
				{
				if(DemoTimer)
					{
					if(!SpritesAreActive())
						{
						BlankBall(PLAYER_RED, 0);
						BlankBall(PLAYER_RED, 1);
						BlankBall(PLAYER_GREEN, 0);
						BlankBall(PLAYER_GREEN, 1);
						Player[PLAYER_RED].Mode = modeTITLE;
						Player[PLAYER_GREEN].Mode = modeIDLE;
						return;
						}
					}
				else if(Player[who].BallCount > 0)
					Player[who].Mode = modeENDSCREEN;
				}

			if((!Player[who].Ball[0].Active) && 
			   (!Player[who].Ball[1].Active))
				{
				BlankBall(who, 0);
				if(Player[who].BallCount <= 0)
					{
					BlankBall(who, 1);

					if(DemoTimer)
						{
						if(!SpritesAreActive())
							{
							Player[PLAYER_RED].Mode = modeTITLE;
							Player[PLAYER_GREEN].Mode = modeIDLE;
							return;
							}
						}
					else
						{
						Player[who].Mode = modeENDGAME;
						ExplodeGoody(who);
						UpdateBallCount(who);
						Player[who].Paddle.Cracks = 0;
						Player[who].Paddle.Active = FALSE;
						return;
						}
					}
				else
					{
					Player[who].Mode = modeRESUMESCREEN;
					UpdateBallCount(who);
					}
				}
			break;

		case modeKILLSCREEN:
			Player[who].Draw = TRUE;

			while((Player[who].Brick[Player[who].Tier][Player[who].Col][Player[who].Row].Style == NONE) ||
				  (Player[who].Brick[Player[who].Tier][Player[who].Col][Player[who].Row].Style == GOLD))
				{
				if(++Player[who].Col >= BRICK_COLS)
					{
					Player[who].Col = 0;
					if(++Player[who].Row >= BRICK_ROWS)
						{
						Player[who].Row = 0;
						if(Player[who].Tier > 0)
							Player[who].Tier--;
						else
							{
							Player[who].Mode = modePLAY;
							break;
							}
						}
					}
				}
			if(Player[who].Brick[Player[who].Tier][Player[who].Col][Player[who].Row].Style > SILVER1)
				Player[who].Brick[Player[who].Tier][Player[who].Col][Player[who].Row].Style = SILVER1;

			EraseBrick(who, Player[who].Tier, Player[who].Col, Player[who].Row);
			break;

		case modeREVIVESCREEN:
			Player[who].Draw = TRUE;

			while((Player[who].Brick[0][Player[who].Col][Player[who].Row].Style == 
			  ScreenDef[Screen].Brick[0][Player[who].Row][Player[who].Col]) &&
				(Player[who].Brick[1][Player[who].Col][Player[who].Row].Style == 
			  ScreenDef[Screen].Brick[1][Player[who].Row][Player[who].Col]) &&
				(Player[who].Brick[2][Player[who].Col][Player[who].Row].Style == 
			  ScreenDef[Screen].Brick[2][Player[who].Row][Player[who].Col]))
				{
				if(++Player[who].Col >= BRICK_COLS)
					{
					Player[who].Col = 0;
					if(++Player[who].Row >= BRICK_ROWS)
						{
						Player[who].Mode = modePLAY;
						break;
						}
					}
				}

			RestoreBrick(who, 2, Player[who].Col, Player[who].Row);
			RestoreBrick(who, 1, Player[who].Col, Player[who].Row);
			RestoreBrick(who, 0, Player[who].Col, Player[who].Row);
			Player[who].Score -= 5;

			break;

		case modeENDGAME:
			Player[who].Draw = TRUE;

			if((Player[who^1].Mode != modeIDLE) && 
			  (Player[who^1].Mode != modeENDGAME))
				break;
				
			if(!SpritesAreActive())
				{
				Player[who].Draw = FALSE;
				ShowMessage(msgOVER);
				ElapsedTime = ticks;

				while(ticks < ElapsedTime+7000)
					{
					if(ReadButtons(PLAYER_RED) & BUTTON_OPTION)
						{
						HideMessage();
						Player[PLAYER_RED].Mode = modeHIGHSCORE;
						if(Option.NumPlayers >= 2)
							Player[PLAYER_GREEN].Mode = modeHIGHSCORE;

						ObjectListFlag = TRUE;
	
						DoOptions();
						return;
						}
					}

				Player[PLAYER_RED].Mode = modeHIGHSCORE;
				if(Option.NumPlayers >= 2)
					Player[PLAYER_GREEN].Mode = modeHIGHSCORE;

				ObjectListFlag = TRUE;
	
				HideMessage();
				return;
				}
			break;

		case modeHIGHSCORE:
			Player[who].Draw = TRUE;
			ObjectListFlag = TRUE;

			if(Option.GameMode == game2000)
				{
				Player[PLAYER_RED].Draw = FALSE;
				Player[PLAYER_GREEN].Draw = FALSE;

				EnterHighScore(PLAYER_RED);
				if(Option.NumPlayers >= 2)
					EnterHighScore(PLAYER_GREEN);

				ShowHighScores();

				ElapsedTime = ticks;
				while(ticks < ElapsedTime+10000)
					{
					if(ReadButtons(PLAYER_RED) & BUTTON_ANY)
						break;
					}
				}

			DoOptions();
			return;
			break;

		case modePONGSTART:
			if(DemoTimer)
				{
				Player[who].Mode = modePONGEND;
				return;
				}
			Player[who].Draw = TRUE;
			ObjectListFlag = TRUE;

			ClearBrickBuffer();
			ClearPlayfieldBuffer();
			ClearPlayfield();

			ClearPowerUps(who);
			
			ShowMessage(msgPONG);
			Delay(2000);

			Option.ScreenWrapOn = TRUE;               
			SetPlayfieldBack(ROUND);                       

			Player[PLAYER_RED].Paddle.Cracks = 0;
			Player[PLAYER_RED].BonusBalls = 1;
			UpdateBallCount(PLAYER_RED);
			Player[PLAYER_RED].Mode = modePONGSHOOT;

			Player[PLAYER_GREEN].Paddle.Cracks = 0;
			Player[PLAYER_GREEN].BonusBalls = 1;
			UpdateBallCount(PLAYER_GREEN);
			Player[PLAYER_GREEN].Mode = modePONGSHOOT;

			Player[PLAYER_RED].Paddle.Active = TRUE;
			Player[PLAYER_GREEN].Paddle.Active = TRUE;

			if(Option.NumPlayers == 1)
				{
				PlayerOnTop = PLAYER_GREEN;
				Player[PLAYER_GREEN].AutoPilot = TRUE;
				SetPaddleSize(PLAYER_GREEN, 2); 
				SetPaddleSize(PLAYER_RED, PADDLE_MAX_SIZE-Option.Skill[PLAYER_RED]);
				Option.NumPlayers = 2;
				Player[PLAYER_GREEN].Timer = ticks;
				}
			else if (Player[PLAYER_RED].BallCount <= 0)
				{
				PlayerOnTop = PLAYER_RED;
				Player[PLAYER_RED].AutoPilot = TRUE;
				SetPaddleSize(PLAYER_RED, 2); 
				SetPaddleSize(PLAYER_GREEN, PADDLE_MAX_SIZE-Option.Skill[PLAYER_GREEN]);
				Player[PLAYER_RED].Timer = ticks;
				}
			else if (Player[PLAYER_GREEN].BallCount <= 0)
				{
				PlayerOnTop = PLAYER_GREEN;
				Player[PLAYER_GREEN].AutoPilot = TRUE;
				SetPaddleSize(PLAYER_RED, PADDLE_MAX_SIZE-Option.Skill[PLAYER_RED]);
				SetPaddleSize(PLAYER_GREEN, 2);
				Player[PLAYER_GREEN].Timer = ticks;
				}
			else
				{
				PlayerOnTop = Random(2);
				SetPaddleSize(PLAYER_RED, PADDLE_MAX_SIZE-Option.Skill[PLAYER_RED]);
				SetPaddleSize(PLAYER_GREEN, PADDLE_MAX_SIZE-Option.Skill[PLAYER_GREEN]);
				if(Player[PLAYER_GREEN].AutoPilot)
					Player[PLAYER_GREEN].Timer = ticks;
				}

			SetPaddleHover(PLAYER_RED, 32);
			SetPaddleHover(PLAYER_GREEN, 32);
			MovePaddle(PLAYER_RED);
			MovePaddle(PLAYER_GREEN);

			HideMessage();
			break;

		case modePONGSHOOT:
			Player[who].Draw = TRUE;
			ObjectListFlag = TRUE;

			if(LaunchBalls(who))
				Player[who].Mode = modePONGPLAY;
			break;

		case modePONGPLAY:
			Player[who].Draw = TRUE;

			if(!Player[who].Ball[0].Active)
				{
				BlankBall(who, 0);
				if(Player[who].BonusBalls <= 0)
					Player[who].Mode = modePONGEND;
				else
					Player[who].Mode = modePONGSHOOT;
				}
			break;

		case modePONGEND:
			Player[who].Draw = TRUE;

			if((Player[who^1].Mode != modeIDLE) && 
			  (Player[who^1].Mode != modePONGEND))
				break;
				
			if(!SpritesAreActive())
				{
				if(Option.Skill[PLAYER_GREEN] == skillNONE)
					{
					Player[PLAYER_GREEN].AutoPilot = FALSE;
					Player[PLAYER_GREEN].Paddle.Active = FALSE;
					Option.NumPlayers = 1;
					Option.ScreenWrapOn = FALSE;               
					SetPlayfieldBack(FLAT);                       
					}
				if(Player[PLAYER_RED].BallCount <= 0)
					Player[PLAYER_RED].Paddle.Active = FALSE;
				if(Player[PLAYER_GREEN].BallCount <= 0)
					Player[PLAYER_GREEN].Paddle.Active = FALSE;

				Player[PLAYER_RED].Mode = modeSTARTSCREEN;
				Player[PLAYER_GREEN].Mode = modeIDLE;
				}
			break;

		case modeTARGETSTART:
			if(DemoTimer)
				{
				Player[who].Mode = modeTARGETEND;
				return;
				}
			Player[who].Draw = TRUE;
			ObjectListFlag = TRUE;

			ClearBrickBuffer();
			ClearPlayfieldBuffer();
			ClearPlayfield();
			
			ShowMessage(msgTARGET);
			Delay(2000);

			Option.ScreenWrapOn = FALSE;               
			SetPlayfieldBack(FLAT);                       

			if(Player[PLAYER_RED].BallCount <= 0)
				{
				Player[PLAYER_RED].Mode = modeIDLE;
				Player[PLAYER_RED].Paddle.Active = FALSE;
				}
			else
				{
				Player[PLAYER_RED].Mode = modeTARGETPLAY;
				Player[PLAYER_RED].Paddle.Active = TRUE;
				Player[PLAYER_RED].Paddle.Gun = 5;
				Player[PLAYER_RED].Paddle.GunPower = 0;
				}

			if((Player[PLAYER_GREEN].AutoPilot) || (Player[PLAYER_GREEN].BallCount <= 0))
				{
				Player[PLAYER_GREEN].Mode = modeIDLE;
				Player[PLAYER_GREEN].Paddle.Active = FALSE;
				PlayerOnTop = PLAYER_GREEN;
				}
		    else
				{
				Player[PLAYER_GREEN].Mode = modeTARGETPLAY;
				Player[PLAYER_GREEN].Paddle.Active = TRUE;
				Player[PLAYER_GREEN].Paddle.Gun = 5;
				Player[PLAYER_GREEN].Paddle.GunPower = 0;
				}

			SetPaddleHover(PLAYER_RED, 32);
			SetPaddleHover(PLAYER_GREEN, 32);

			MovePaddle(PLAYER_RED);
			MovePaddle(PLAYER_GREEN);

			SetPaddleSize(PLAYER_RED, PADDLE_MAX_SIZE-Option.Skill[PLAYER_RED]); 
			SetPaddleSize(PLAYER_GREEN, PADDLE_MAX_SIZE-Option.Skill[PLAYER_GREEN]); 

			LoadBricks();

			HideMessage();
			RefreshScoreBoard();
			break;

		case modeTARGETPLAY:
			if(!Player[who].Paddle.Active)
				return;

			Player[who].Draw = TRUE;

			if( ((Player[who].Paddle.Gun == 0) && (!Player[who].Bomb.Active)) ||	
			   (Player[who].BrickCount == 0))
				Player[who].Mode = modeTARGETEND;
			break;

		case modeTARGETEND:
			if(Player[who^1].Mode != modeTARGETPLAY)
				{
				if(!SpritesAreActive())
					{
					Player[PLAYER_RED].Score += (Player[PLAYER_RED].Paddle.Gun*500);
					UpdateScore(PLAYER_RED, TRUE);
					Player[PLAYER_GREEN].Score += (Player[PLAYER_GREEN].Paddle.Gun*500);
					UpdateScore(PLAYER_GREEN, TRUE);
				
					Player[PLAYER_RED].Paddle.Gun = FALSE;
					Player[PLAYER_GREEN].Paddle.Gun = FALSE;
					Player[PLAYER_RED].Mode = modeSTARTSCREEN;
					Player[PLAYER_GREEN].Mode = modeIDLE;

					if((Player[PLAYER_GREEN].AutoPilot) && (Player[PLAYER_GREEN].BallCount > 0))
						Player[PLAYER_GREEN].Paddle.Active = TRUE;
					}
				}
			break;
		}

	MoveSprites(who);
	CheckKeypad(who);
	CheckButtons(who);

	if(DemoTimer)
		{
		Reset = FALSE;

		if(ticks-DemoTimer > 120000)
			Reset = TRUE;

		if(Player[who].Button & BUTTON_ANY)
			Reset = TRUE;

		if(Player[who].Joypad & JOYPAD_ANY)
			Reset = TRUE;

		if(Player[who].Keypad & KEYPAD_ANY)
			Reset = TRUE;


		if(Reset)
			{
			if(!SpritesAreActive())
				{
				Player[PLAYER_RED].Mode = modeTITLE;
				Player[PLAYER_GREEN].Mode = modeIDLE;
				}
			}
		}
	}


/*********************************************************************
 *  FUNCTION:		CheckKeypad
 *
 *  PARAMETERS:		short who
 *
 *  DESCRIPTION:	Handles the keypad
 *
 *  RETURNS:		void
 *
 *********************************************************************/
void CheckKeypad(short who)
	{
	switch(Player[who].Keypad)
		{
		case KEYPAD_NONE:
			return;
			break;

		case KEYPAD_0:
			if(Player[who].OldKeypad != KEYPAD_0)
				{
				if(Option.MusicOn)
					{
				  	Option.MusicOn = FALSE;
					StopMusic();
					}
				else
				  	{
					Option.MusicOn = TRUE;
 					StartMusic(((ScreenCount/5)%10)+songPHASE0);                        
				  	}
				}
			break;

		case KEYPAD_RESET:
			if(!SpritesAreActive())
				{
				Player[PLAYER_RED].Mode = modeTITLE;
				Player[PLAYER_GREEN].Mode = modeIDLE;
				}
			break;
/*
		case KEYPAD_1:
			PlaySound(sndEXTRA, 127, 0, 7);
			SetPaddleSize(who, Random(PADDLE_MAX_SIZE+2));
			while(ReadKeyPad(who) == KEYPAD_1);
			break;
*/
		}

	if( (Player[who].Mode == modeSHOOT) && 
		((Player[who^1].Mode == modeSHOOT) || (Player[who^1].Mode == modeIDLE)) )
		{
		/* These are cheat hooks that can only be used before launching ball */
		switch(Player[who].Keypad)
			{
			case KEYPAD_BALL:
				if(Option.GameMode == game2000)
					{
					PlaySound(sndEXTRA, 127, 0, 7);
					Player[who].BallCount	= 99;
					UpdateBallCount(who);
					while(ReadKeyPad(who) == KEYPAD_BALL);
					}
				break;

			case KEYPAD_SCREEN:
				if(Option.GameMode == game2000)
					{
					PlaySound(sndEXTRA, 127, 0, 7);
			 		ScreenCount++;
			 		Screen = ScreenCount%NUM_SCREENS;

					BlankBall(PLAYER_RED, 0);
					BlankBall(PLAYER_RED, 1);
					BlankBall(PLAYER_GREEN, 0);
					BlankBall(PLAYER_GREEN, 1);

		 			Player[PLAYER_RED].Mode = modeSTARTSCREEN;
					if(Option.NumPlayers > 1)
			 			Player[PLAYER_GREEN].Mode = modeSTARTSCREEN;
		
					while(ReadKeyPad(who) == KEYPAD_SCREEN);
					}
				break;

			case KEYPAD_PHASE:
				if(Option.GameMode == game2000)
					{
					PlaySound(sndEXTRA, 127, 0, 7);
			 		ScreenCount += (5-(ScreenCount%5));
			 		Screen = ScreenCount%NUM_SCREENS;

					BlankBall(PLAYER_RED, 0);
					BlankBall(PLAYER_RED, 1);
					BlankBall(PLAYER_GREEN, 0);
					BlankBall(PLAYER_GREEN, 1);

		 			Player[PLAYER_RED].Mode = modeSTARTSCREEN;
					if(Option.NumPlayers > 1)
			 			Player[PLAYER_GREEN].Mode = modeSTARTSCREEN;
		
					while(ReadKeyPad(who) == KEYPAD_PHASE);
					}
				break;
			
			case KEYPAD_SUPER:
				PlaySound(sndEXTRA, 127, 0, 7);
				DoSuperBall(who);
				while(ReadKeyPad(who) == KEYPAD_SUPER);
				break;

			case KEYPAD_CANNON:
				PlaySound(sndEXTRA, 127, 0, 7);
				Player[who].Paddle.Gun = ON;
				Player[who].Paddle.GunPower = 0;
				DrawPaddleGun(who);
				if(who==PLAYER_RED)
					DrawIcon(iconGUN, GUN_RED_X, GUN_RED_Y, ON);
				else
					DrawIcon(iconGUN, GUN_GREEN_X, GUN_GREEN_Y, ON);
				while(ReadKeyPad(who) == KEYPAD_CANNON);
				break;

			case KEYPAD_ATTRACT:
				PlaySound(sndEXTRA, 127, 0, 7);
				Player[who].Paddle.Attract = MIN(1, Player[who].Paddle.Attract+1);
				if(who==PLAYER_RED)
					DrawIcon(iconATTRACT, ATTRACT_RED_X, ATTRACT_RED_Y, Player[who].Paddle.Attract);
				else
					DrawIcon(iconATTRACT, ATTRACT_GREEN_X, ATTRACT_GREEN_Y, Player[who].Paddle.Attract);
				while(ReadKeyPad(who) == KEYPAD_ATTRACT);
				break;

			case KEYPAD_CATCH:
				PlaySound(sndEXTRA, 127, 0, 7);
				Player[who].Paddle.Cracks=0;
				Player[who].Paddle.Catch = TRUE;
				DrawPaddleFace(who);
				if(who == PLAYER_RED)
					DrawIcon(iconCATCH, CATCH_RED_X, CATCH_RED_Y, ON);
				else
					DrawIcon(iconCATCH, CATCH_GREEN_X, CATCH_GREEN_Y, ON);
				while(ReadKeyPad(who) == KEYPAD_CATCH);
				break;
			}
		}
	}



/*********************************************************************
 *  FUNCTION:		CheckButtons
 *
 *  PARAMETERS:		short who
 *
 *  DESCRIPTION:	Handles the buttons
 *
 *  RETURNS:		void
 *
 *********************************************************************/
void CheckButtons(short who)
	{
	int KeyPad;
	int PauseMode;
	int OldMode;
	int Button;
	int Time;
	int Changed;

	BYTE Hidden;
	BYTE Done;

	Button = BUTTON_NONE;

	switch(Player[who].Button)
		{
		case BUTTON_NONE:
			return;
			break;

		case BUTTON_PAUSE:
			Changed = FALSE;
			Hidden = FALSE;
			Done = FALSE;
			PauseMode = pauseIDLE;
			OldMode = pauseIDLE;
			StopMusic();
			ShowMessage(msgPAUSED);

			/* Wait until the PAUSE button is released */
			Delay(200);
			while(ReadButtons(who) & BUTTON_ANY);

			Time = ticks;

			/* Wait until the PAUSE button is pressed again */
			while(!Done)
				{				
				KeyPad = ReadKeyPad(who);

				OldMode = PauseMode;

				/* Check for reset */
				switch(KeyPad)
					{
					case KEYPAD_HIDE:
						HideMessage();
						HidePopup();
						Hidden = TRUE;
						PauseMode = pauseIDLE;
						break;

					case KEYPAD_RESET:
						Player[PLAYER_RED].Mode = modeTITLE;
						Player[PLAYER_GREEN].Mode = modeIDLE;
						return;
						break;
					}

				Button = ReadButtons(who);

				if(Hidden && (Button & BUTTON_ANY))
					{
					PauseMode = pauseIDLE;
					ShowMessage(msgPAUSED);
					Hidden = FALSE;
					}

				if(Button & BUTTON_PAUSE)
					Done = TRUE;
				else
					Delay(50);

				switch (Button)
				  	{
					case BUTTON_A:
						Changed = TRUE;
						PauseMode = pauseMUSIC_ADJUST;
						if(OldMode != PauseMode)
							{
							Option.MusicOn = TRUE;
 							StartMusic(((ScreenCount/5)%10)+songPHASE0);
							}
						ShowPopup(popupMUSIC);
						DisplayMeterLevel(TRUE, POPUP_X+16, POPUP_Y+22, Option.MusicVolume);
						while(ReadButtons(who) & BUTTON_A);
						break;

					case BUTTON_B:
						Changed = TRUE;
						PauseMode = pauseEFFECTS_ADJUST;
						ShowPopup(popupEFFECTS);
						DisplayMeterLevel(TRUE, POPUP_X+16, POPUP_Y+22, Option.EffectsVolume);
						while(ReadButtons(who) & BUTTON_B);
						break;
				  	}

				if(OldMode == pauseMUSIC_ADJUST)
					{
					if(PauseMode != OldMode)
						StopMusic();
					}

				switch (PauseMode)
					{
					case pauseIDLE:
						break;

					case pauseMUSIC_ADJUST:
						HandleMusicPopup(who);
						break;

					case pauseEFFECTS_ADJUST:
						if(abs(ticks-Time) > 1000)
							{
							Time = ticks;
							PlaySound(sndSHOOT, 127, 0, 7);
							}
						HandleEffectsPopup(who);
						break;
					}
				}

			if(Changed)
				WriteOptions();

			HideMessage();
			HidePopup();
				
			/* Wait until the PAUSE button is released again */
			while(ReadButtons(who) & BUTTON_PAUSE);

			StartMusic(((ScreenCount/5)%10)+songPHASE0);

			Player[PLAYER_RED].Timer = ticks;
			Player[PLAYER_GREEN].Timer = ticks;
			break;
		}
	}


/*********************************************************************
 *  FUNCTION:		MoveSprites
 *
 *  PARAMETERS:		short who
 *
 *  DESCRIPTION:	Moves the game object to new locations, but does
 *					not display them.
 *
 *  RETURNS:		void
 *
 *********************************************************************/
void MoveSprites(short who)
	{
	char ball;

	for (ball=0; ball<MAX_BALLS; ball++)
	 	MoveBall(who, ball);

	CheckBallCollision(PLAYER_RED, 0, PLAYER_RED, 1);


	if(Option.NumPlayers > 1)
		{
		CheckBallCollision(PLAYER_RED, 0, PLAYER_GREEN, 0);
		CheckBallCollision(PLAYER_RED, 0, PLAYER_GREEN, 1);
		CheckBallCollision(PLAYER_RED, 1, PLAYER_GREEN, 0);
		CheckBallCollision(PLAYER_RED, 1, PLAYER_GREEN, 1);
		CheckBallCollision(PLAYER_GREEN, 0, PLAYER_GREEN, 1);
		}

 	MoveGoody(who);
	MovePaddle(who);  /* Paddle move must be done before MoveBomb() */
	MoveBomb(who);
	}


/*********************************************************************
 *  FUNCTION:		EraseSprites
 *
 *  PARAMETERS:		void
 *
 *  DESCRIPTION:	Erases the sprites from the playfield buffer
 *
 *  RETURNS:		void
 *
 *********************************************************************/
void EraseSprites()
	{
	int i;
	
	if(Player[PLAYER_RED].Draw)
		{
		EraseBall(PLAYER_RED, 0);
		EraseBall(PLAYER_RED, 1);
		EraseBomb(PLAYER_RED);
		}

	if(Player[PLAYER_GREEN].Draw)
		{
		EraseBall(PLAYER_GREEN, 0);
		EraseBall(PLAYER_GREEN, 1);
		EraseBomb(PLAYER_GREEN);
		}

	for(i=0; i<MAX_RIPPLES; i++)
		EraseRipple(i);

	for(i=0; i<MAX_DROPBRICKS; i++)
		EraseDropBrick(i);

	EraseDroid();
	if(Player[PLAYER_RED].Draw)
		EraseGoody(PLAYER_RED);
	if(Player[PLAYER_GREEN].Draw)
		EraseGoody(PLAYER_GREEN);
	}


/*********************************************************************
 *  FUNCTION:		DrawSprites
 *
 *  PARAMETERS:		void
 *
 *  DESCRIPTION:	Draws the sprites in the playfield buffer
 *
 *  RETURNS:		void
 *
 *********************************************************************/
void DrawSprites()
	{
	int i;

	if(Player[PLAYER_RED].Draw)
		{
		DrawBall(PLAYER_RED, 0);
		DrawBall(PLAYER_RED, 1);
		DrawBomb(PLAYER_RED);
		}

	if(Player[PLAYER_GREEN].Draw)
		{
		DrawBall(PLAYER_GREEN, 0);
		DrawBall(PLAYER_GREEN, 1);
		DrawBomb(PLAYER_GREEN);
		}

	for(i=0; i<MAX_RIPPLES; i++)
		DrawRipple(i);

	for(i=0; i<MAX_DROPBRICKS; i++)
		DrawDropBrick(i);

	DrawDroid();

	if(Player[PLAYER_RED].Draw)
		DrawGoody(PLAYER_RED);
	if(Player[PLAYER_GREEN].Draw)
		DrawGoody(PLAYER_GREEN);
	}



/*********************************************************************
 *  FUNCTION:		DisplaySprites
 *
 *  PARAMETERS:		void
 *
 *  DESCRIPTION:	Displays the sprites in the playfield
 *
 *  RETURNS:		void
 *
 *********************************************************************/
void DisplaySprites()
	{
	int i;
	
	if(Player[PLAYER_RED].Draw)
		{
		DisplayBall(PLAYER_RED, 0);
		DisplayBall(PLAYER_RED, 1);
		DisplayBomb(PLAYER_RED);
		}

	if(Player[PLAYER_GREEN].Draw)
		{
		DisplayBall(PLAYER_GREEN, 0);
		DisplayBall(PLAYER_GREEN, 1);
		DisplayBomb(PLAYER_GREEN);
		}

	for(i=0; i<MAX_RIPPLES; i++)
		DisplayRipple(i);

	for(i=0; i<MAX_DROPBRICKS; i++)
		DisplayDropBrick(i);

	DisplayDroid();

	if(Player[PLAYER_RED].Draw)
		DisplayGoody(PLAYER_RED);
	if(Player[PLAYER_GREEN].Draw)
		DisplayGoody(PLAYER_GREEN);
	}


/*********************************************************************
 *  FUNCTION:		InitPlayer
 *
 *  PARAMETERS:		short who
 *
 *  DESCRIPTION:	Initializes player defaults
 *
 *  RETURNS:		void
 *
 *********************************************************************/
void InitPlayer(short who)
	{
	int i;

	/* Initialize Misc. Shit */
	if((who == PLAYER_GREEN) && (Option.NumPlayers == 1))
		Player[who].BallCount	= 0;
	else
		Player[who].BallCount	= 5;
	Player[who].BallStyle = ballNORMAL;

	Player[who].Score = 0L;
	Player[who].BrickCount = 0;

	/* initialize the goody */
	Player[who].Goody.Active  = FALSE;
	Player[who].Goody.X       = 0;
	Player[who].Goody.Y       = 0;
	Player[who].Goody.Z       = 1000;
	Player[who].Goody.ScreenX = 0;
	Player[who].Goody.ScreenY = 0;
	Player[who].Goody.Width	  = 0;
	Player[who].Goody.Height  = 0;
	Player[who].Goody.OldX    = 0;
	Player[who].Goody.OldY    = 0;
	Player[who].Goody.OldWidth = 0;
	Player[who].Goody.OldHeight = 0;
	Player[who].Goody.MaxSpeed	= 0;
	Player[who].Goody.SpeedX  = 0;
	Player[who].Goody.SpeedY  = 0;
	Player[who].Goody.SpeedZ  = 0;
	Player[who].Goody.Scale   = 100;
	Player[who].Goody.Step    = 0;
	Player[who].Goody.Style   = 0;
	Player[who].Goody.Phase   = 0;
	/* initialize the droid */
	Droid.Active  	= FALSE;
	Droid.Style     = 0;
	Droid.Loc.X		= 0;
	Droid.Loc.Y		= 0;
	Droid.Loc.Z    	= 1000;
	Droid.Dest.X 	= 0;
	Droid.Dest.Y 	= 0;
	Droid.Dest.Z   	= 1000;
	Droid.Screen.X 	= 0;
	Droid.Screen.Y 	= 0;
	Droid.Old.X    	= 0;
	Droid.Old.Y    	= 0;
	Droid.SpeedX  	= 0;
	Droid.SpeedY  	= 0;
	Droid.SpeedZ  	= 0;
	Droid.Scale   	= 100;
	Droid.Direction	= 0;
	Droid.NewHeading = 0;
	Droid.Width		= 0;
	Droid.Height	= 0;
	Droid.Flags 	= DROID_1_FLAGS;
	Droid.Index		= 0;

	/* initialize the bomb */
	Player[who].Bomb.Active  = FALSE;
	Player[who].Bomb.Style   = bombNORMAL;
	Player[who].Bomb.X       = 0;
	Player[who].Bomb.Z       = 1000;
	Player[who].Bomb.ScreenX = 0;
	Player[who].Bomb.ScreenY = 0;
	Player[who].Bomb.OldX    = 0;
	Player[who].Bomb.OldY    = 0;
	Player[who].Bomb.Width   = BOMB_WIDTH;
	Player[who].Bomb.Height  = BOMB_HEIGHT;
	Player[who].Bomb.OldWidth   = BOMB_WIDTH;
	Player[who].Bomb.OldHeight  = BOMB_HEIGHT;
	Player[who].Bomb.SpeedX  = 0;
	Player[who].Bomb.SpeedY  = 0;
	Player[who].Bomb.SpeedZ  = 0;
	Player[who].Bomb.Scale   = 100;
	Player[who].Bomb.Step    = 0;

	/* initialize the balls */
	for(i=0; i<MAX_BALLS; i++)
		{
		Player[who].Ball[i].Active  	= FALSE;
		Player[who].Ball[i].X       	= 0;
		if(who == PLAYER_ON_TOP)
			{
			Player[who].Ball[i].Y   	= -BALL_Y;
			Player[who].Ball[i].Sector 	= sectorTOP;
			}
		else
			{
			Player[who].Ball[i].Y		= BALL_Y;
			Player[who].Ball[i].Sector 	= sectorBOTTOM;
			}

		Player[who].Ball[i].Z       	= 1000;
		Player[who].Ball[i].ScreenX 	= 0;
		Player[who].Ball[i].ScreenY 	= 0;
		Player[who].Ball[i].Width   	= BALL_WIDTH;
		Player[who].Ball[i].Height  	= BALL_HEIGHT;

		Player[who].Ball[i].OldX    	= 0;
		Player[who].Ball[i].OldY    	= 0;
		Player[who].Ball[i].OldW    	= BALL_WIDTH;
		Player[who].Ball[i].OldH    	= BALL_HEIGHT;

		Player[who].Ball[i].ShadowX		= 0;
		Player[who].Ball[i].ShadowY		= 0;
		Player[who].Ball[i].OldShadowX	= 0;
		Player[who].Ball[i].OldShadowY	= 0;

		Player[who].Ball[i].SpeedX  	= 0;
		Player[who].Ball[i].SpeedY  	= 0;
		Player[who].Ball[i].SpeedZ  	= 0;

		Player[who].Ball[i].Size	   	= 6;
		Player[who].Ball[i].Step    	= 0;
		Player[who].Ball[i].Hits	   	= 0;
		Player[who].Ball[i].Style   	= ballNORMAL;
		Player[who].Ball[i].Caught  	= FALSE;

		SetBallImage(who, i);
		}

	/* initialize the paddle */
	Player[who].Paddle.X            = 0;
	Player[who].Paddle.Z            = PADDLE_Z;
	Player[who].Paddle.ScreenX		= 128;
	Player[who].Paddle.OldX			= 128;
	if(who == PLAYER_RED)
		{
		Player[who].Paddle.ScreenY	= 180;
		Player[who].Paddle.OldY		= 180;
		}
	else
		{
		Player[who].Paddle.ScreenY	= 20;
		Player[who].Paddle.OldY		= 20;
		}
	Player[who].Paddle.SpeedX		= 0;
	Player[who].Paddle.SpeedY		= 0;
	switch(Option.PaddleResponse[who])
		{
		case 0:
			Player[who].Paddle.MaxSpeedX = 384;
			break;

		case 1:
			Player[who].Paddle.MaxSpeedX = 448;
			break;

		case 2:
			Player[who].Paddle.MaxSpeedX = 608;
			break;
		}

	if(Player[who].AutoPilot)
		Player[who].Paddle.MaxSpeedX = 550;

	Player[who].Paddle.IncSpeedY	= 0;
	Player[who].Paddle.Hits			= 0;
	Player[who].Paddle.Attract      = FALSE;
	SetPaddleSize(who, PADDLE_MAX_SIZE-Option.Skill[who]);
	Player[who].Paddle.Active		= FALSE;
	Player[who].Paddle.Small		= FALSE;
	Player[who].Paddle.Catch		= FALSE;
	Player[who].Paddle.Gun			= FALSE;
	Player[who].Paddle.GunPower		= 0;
	Player[who].Paddle.Index		= 0;
	Player[who].Paddle.IndexCnt		= 0;
	Player[who].Paddle.Cracks		= 0;

	for(i=0; i<MAX_RIPPLES ; i++)
		Ripple[i].Active = FALSE;

	for(i=0; i<MAX_DROPBRICKS ; i++)
		DropBrick[i].Active = FALSE;

	/* initialize player specific data */
	if(who == PLAYER_ON_TOP)
		{
	    Player[who].Goody.Y			= -GOODY_Y;
	    Player[who].Bomb.Y			= -BOMB_Y;
		Player[who].Ball[0].Y		= -BALL_Y;
		Player[who].Ball[1].Y		= -BALL_Y;
		Player[who].Paddle.Y		= -PADDLE_MID_Y;
		Player[who].Paddle.MaxY		= -PADDLE_MID_Y;
		Player[who].Paddle.MinY		= -PADDLE_MID_Y;
		}
	else
		{
	    Player[who].Goody.Y			= GOODY_Y;
	    Player[who].Bomb.Y			= BOMB_Y;
		Player[who].Ball[0].Y		= BALL_Y;
		Player[who].Ball[1].Y		= BALL_Y;
		Player[who].Paddle.Y		= PADDLE_MID_Y;
		Player[who].Paddle.MaxY		= PADDLE_MID_Y;
		Player[who].Paddle.MinY		= PADDLE_MID_Y;
		}
	}



/*********************************************************************
 *  FUNCTION:	  SpritesAreActive
 *
 *  PARAMETERS:	  
 *
 *  DESCRIPTION:  
 *
 *  RETURNS:	  
 *
 *********************************************************************/
char SpritesAreActive()
	{
	int i;
	char retval;

	retval = FALSE;

	if(Option.GameMode == gameCLASSIC)
		return(retval);
		
	if(Droid.Active)
		{
		AbortDroid();
		retval = TRUE;
		}

	if(Player[0].Goody.Active)
		BlankGoody(0);

	if(Player[0].Bomb.Active)
		BlankBomb(0);

	if(Option.NumPlayers > 1)
		{
		if(Player[1].Goody.Active)
			BlankGoody(1);

		if(Player[1].Bomb.Active)
			BlankBomb(1);
		}

	for(i=0; i<MAX_RIPPLES; i++)
		{
		EraseRipple(i);
		Ripple[i].Active = FALSE;
		}

	for(i=0; i<MAX_DROPBRICKS; i++)
		{
		if(DropBrick[i].Active)
			retval = TRUE;
/*
		EraseDropBrick(i);
		DisplayDropBrick(i);
		DropBrick[i].Active = FALSE;
*/
		}

	return(retval);
	}



/*********************************************************************
 *  FUNCTION:	  DoTitle
 *
 *  PARAMETERS:	  
 *
 *  DESCRIPTION:  
 *
 *  RETURNS:	  
 *
 *********************************************************************/
void DoTitle()
	{
	ULONG	Time;
	int Buttons;
	char Start;

	if(TitleExpand)
		{
		ReadOptions();
		ShowTitle();
		while(BlitAddIndex != BlitUseIndex);
		StartMusic(songTITLE);
		}
	TitleExpand = TRUE;

	ClearBackground();
	while(BlitAddIndex != BlitUseIndex);

	Time = ticks;

	Start = FALSE;

	while(abs(ticks-Time) < 10000)
		{
		Buttons = ReadButtons(PLAYER_RED);
		 
		/* Check for EEPROM initialize */
		if(Buttons & BUTTON_OPTION)
			{
			if(ReadKeyPad(PLAYER_RED) == KEYPAD_RESET)
				{
				StopMusic();
				ShowMessage(msgCLEAR);
				ResetOptions(TRUE);

			 	ShowTitle();
				while(BlitAddIndex != BlitUseIndex);
				StartMusic(songTITLE);
				Time = ticks;
				}
			}

		if(Buttons & BUTTON_FIRE)
			{
			Start = TRUE;
			Player[PLAYER_RED].Mode = modeSTARTGAME;
			Player[PLAYER_GREEN].Mode = modeSTARTGAME;
			}

		if(Start)
			{
			DemoTimer = 0;
			return;
			}

		Player[PLAYER_RED].Keypad = ReadKeyPad(PLAYER_RED);
		CheckKeypad(PLAYER_RED);
		}

	DemoTimer = 0;
	}


/*********************************************************************
 *  FUNCTION:	  HandleMusicPopup
 *
 *  PARAMETERS:	  
 *
 *  DESCRIPTION:  
 *
 *  RETURNS:	  
 *
 *********************************************************************/
void HandleMusicPopup(short who)
	{
	int JoyStick;

	JoyStick = ReadJoyStick(who);

	if(JoyStick & (JOYPAD_LEFT | JOYPAD_RIGHT))	
		{
		switch(JoyStick)
			{
			case JOYPAD_RIGHT:
				Option.MusicVolume += 1;
				if(Option.MusicVolume > OPT_MAX_VOLUME)
					Option.MusicVolume = OPT_MAX_VOLUME;
				break;

			case JOYPAD_LEFT:
				Option.MusicVolume -= 1;
				if(Option.MusicVolume < 0)
					Option.MusicVolume = 0;
				break;
			}

		SMainVolume(Option.MusicVolume <<8);

		DisplayMeterLevel(TRUE, POPUP_X+16, POPUP_Y+22, Option.MusicVolume);
		}
	}


/*********************************************************************
 *  FUNCTION:	  HandleEffectsPopup
 *
 *  PARAMETERS:	  
 *
 *  DESCRIPTION:  
 *
 *  RETURNS:	  
 *
 *********************************************************************/
void HandleEffectsPopup(short who)
	{
	int JoyStick;

	JoyStick = ReadJoyStick(who);

	if(JoyStick & (JOYPAD_LEFT | JOYPAD_RIGHT))	
		{
		switch(JoyStick)
			{
			case JOYPAD_RIGHT:
				Option.EffectsVolume += 1;
				if(Option.EffectsVolume > OPT_MAX_VOLUME)
					Option.EffectsVolume = OPT_MAX_VOLUME;
				break;

			case JOYPAD_LEFT:
				Option.EffectsVolume -= 1;
				if(Option.EffectsVolume < 0)
					Option.EffectsVolume = 0;
				break;
			}

		SSFXVolume(Option.EffectsVolume <<8);

		DisplayMeterLevel(TRUE, POPUP_X+16, POPUP_Y+22, Option.EffectsVolume);
		}
	}