/*******************************************************************
 *
 *	TITLE:			ball.c
 *
 *	DESCRIPTION:	B2K routines to handle the balls
 *
 *	AUTHOR:			Mario Perdue & Richard Degler
 *
 *	HISTORY:		10.01.94		New Program
 *
 *
 *		   COPYRIGHT 1994, 1995 MP Graphics Systems		 
 *	  UNATHORIZED REPRODUCTION, ADAPTATION, DISTRIBUTION,
 * 	  PERFORMANCE OR DISPLAY OF THIS COMPUTER PROGRAM OR   
 *	 THE ASSOCIATED AUDIOVISUAL WORK IS STRICTLY PROHIBITED.
 *						ALL RIGHTS RESERVED.			   
 *******************************************************************/

/** include files **/
#include "\jaguar\include\jaguar.h"
#include "ball.h"
#include "brick.h"
#include "paddle.h"
#include "jagobj.h"
#include "b2k_game.h"
#include "support.h"
#include "displays.h"
#include "player.h"
#include "option.h"
#include "ripple.h"
#include "droid.h"


/** external data **/
extern  long		IMG_BallNormal[MAX_PLAYERS][8][480];
extern  long		IMG_BallSuper[MAX_PLAYERS][8][480];
extern  long		IMG_BallExplode[MAX_PLAYERS][8][1280];
extern  long		IMG_BallLaunch[8][192];
extern  WORD		IMG_BallShadow[480];
extern	WORD		IMG_BallClassic[144];

extern  WORD		IMG_Playfield[];
extern  WORD		IMG_PlayfieldBuffer[];
extern  WORD		IMG_BrickBuffer[];
extern	WORD 		IMG_Background[];

extern	DEF_RAM_PADDLE	IMG_Paddle[MAX_PLAYERS];

extern  int			ScreenCount;

extern  int			ScreenOrgX;
extern  int			ScreenOrgY;

extern	DEF_PLAYER 	Player[2];
extern	DEF_OPTION	Option;

extern	WORD		BallZTab[35];

extern	WORD		ExplodeOffsetTab[7];
extern	WORD		ExplodeWidthTab[7];

extern	WORD		BallOffsetTab[7];
extern	WORD		BallWidthTab[7];

extern	ULONG		ticks;

extern	BYTE		PlayerOnTop;

extern	DEF_BLIT_DATA	Blit[];
extern	LONG			BlitAddIndex;
extern	LONG			BlitUseIndex;

extern	WORD		PaddleWidthTab[5];

/** public data **/
int					BallSpeedLimit;
int					BallMaxSpeedX;
int					BallMaxSpeedY;
int					BallMaxSpeedZ;
int					BallRadius;


/*********************************************************************
 *  FUNCTION:	   SetBallSpeed
 *
 *  PARAMETERS:	   short who, short which, int SpeedX, int SpeedZ
 *
 *  DESCRIPTION:   Sets the ball speed
 *
 *  RETURNS:	   void
 *
 *********************************************************************/
void SetBallSpeed(short who, short which, int SpeedX, int SpeedY, int SpeedZ)
	{
	Player[who].Ball[which].SpeedX = SpeedX;
	Player[who].Ball[which].SpeedY = SpeedY;
	Player[who].Ball[which].SpeedZ = SpeedZ;
	}


/*********************************************************************
 *  FUNCTION:	   ApplyAttract
 *
 *  PARAMETERS:	   short who, short which
 *
 *  DESCRIPTION:   Applies attract factor to ball movement
 *
 *  RETURNS:	   void
 *
 *********************************************************************/
void ApplyAttract(short who, short which)
	{
	WORD Attract;
	WORD DeltaX, DeltaZ;
	WORD Pwidth;

	BYTE BallSector;
	WORD BallX;
	WORD BallSpeedX;
	WORD BallZ;
	WORD BallSpeedZ;

	WORD PaddleAttract;
	BYTE PaddleSize;
	WORD PaddleX;
	WORD PaddleZ;

	
	BallSector		= Player[who].Ball[which].Sector;
	BallX			= Player[who].Ball[which].X;
	BallZ			= Player[who].Ball[which].Z;
	BallSpeedX		= Player[who].Ball[which].SpeedX;
	BallSpeedZ		= Player[who].Ball[which].SpeedZ;
	
	PaddleAttract	= Player[who].Paddle.Attract;
	PaddleSize		= Player[who].Paddle.Size;
	PaddleX			= Player[who].Paddle.X;
	PaddleZ			= Player[who].Paddle.Z;
	
	if(PaddleAttract == 0)
		return;

	if(BallSpeedZ < 0)
		return;

	if((BallSector == sectorTOP) && (who != PLAYER_ON_TOP))
		return;

	if((BallSector == sectorBOTTOM) && (who != PLAYER_ON_BOTTOM))
		return;

	DeltaZ=abs(PaddleZ-BallZ);

	if(DeltaZ > 2500)
		return;

	DeltaX=PaddleX-BallX;

	if(abs(DeltaX) > 2500)
		return;

	Pwidth = (PaddleSize*PADDLE_STEP_W+PADDLE_MIN_W)/2;

	/* If attract, not repel */
	if(PaddleAttract > 0)
		{
		Attract = MIN(20, abs(DeltaX));
		}
	else
		{
		if(abs(DeltaX) <= Pwidth)
			Attract = -15;
		else
			return;
		}
				
	if(DeltaX>=0)
		BallSpeedX += Attract;
	else
		BallSpeedX -= Attract;

	Player[who].Ball[which].SpeedX = BallSpeedX;
	}


/*********************************************************************
 *  FUNCTION:	   LimitBallSpeed
 *
 *  PARAMETERS:	   short who, short which
 *
 *  DESCRIPTION:   Limits the ball speed
 *
 *  RETURNS:	   void
 *
 *********************************************************************/
void LimitBallSpeed(short who, short which)
	{
	int SpeedX;
	int SpeedY;
	int SpeedZ;

	int Mode;

	WORD BallSpeedX;
	WORD BallSpeedY;
	WORD BallSpeedZ;

	Mode = Player[who].Mode;

	BallSpeedX = Player[who].Ball[which].SpeedX;
	BallSpeedY = Player[who].Ball[which].SpeedY;
	BallSpeedZ = Player[who].Ball[which].SpeedZ;

	if(Mode == modePONGPLAY)
		{
		SpeedX = BallSpeedLimit*5;
		SpeedY = BallSpeedLimit*5;
		SpeedZ = BallSpeedLimit*5;
		}
	else
		{
		SpeedX = BallSpeedLimit;
		SpeedY = BallSpeedLimit;
		SpeedZ = BallSpeedLimit;
		}

	/* Limit Ball X Speed */
	if(BallSpeedX > SpeedX)
		BallSpeedX = SpeedX;
	else if(BallSpeedX < -SpeedX)
		BallSpeedX = -SpeedX;
	
	/* Limit Ball Y Speed */
	if(BallSpeedY > SpeedY)
		BallSpeedY -= 1;
	else if(BallSpeedY < -SpeedY)
		BallSpeedY += 1;
	
	/* Limit Ball Z Speed */
	if(BallSpeedZ > SpeedZ)
		BallSpeedZ = SpeedZ;
	else if(BallSpeedZ < -SpeedZ)
		BallSpeedZ = -SpeedZ;

	if(Player[who].Ball[which].Sector == sectorBACK)
		{
		if(abs(BallSpeedY) < BALL_MIN_SPEED)
			BallSpeedY = BALL_MIN_SPEED;
		}
	else if(abs(BallSpeedZ) < BALL_MIN_SPEED)
		{
		if(BallSpeedZ > 0)
			BallSpeedZ = BALL_MIN_SPEED;
		else
			BallSpeedZ = -BALL_MIN_SPEED;
		}

	Player[who].Ball[which].SpeedX = BallSpeedX;
	Player[who].Ball[which].SpeedY = BallSpeedY;
	Player[who].Ball[which].SpeedZ = BallSpeedZ;
	}



/*********************************************************************
 *  FUNCTION:	   SetBallImage
 *
 *  PARAMETERS:	   short who, short which
 *
 *  DESCRIPTION:   Sets up ball image
 *
 *  RETURNS:	   void
 *
 *********************************************************************/
void SetBallImage(short who, short which)
	{
	BYTE 	BallActive;
	UWORD	BallWidth;
	UWORD	BallHeight;
	WORD	BallSpeedX;
	UWORD	BallSize;
	WORD	BallStep;
	BYTE	BallStyle;
	LONG	*BallImage;

	BYTE	PaddleSize;

	BallActive	= Player[who].Ball[which].Active;
	BallWidth	= Player[who].Ball[which].Width;
	BallHeight	= Player[who].Ball[which].Height;
	BallSpeedX	= Player[who].Ball[which].SpeedX;
	BallSize	= Player[who].Ball[which].Size;
	BallStep	= Player[who].Ball[which].Step;
	BallStyle	= Player[who].Ball[which].Style;

	PaddleSize	= Player[who].Paddle.Size;

	BallStep++;

	if(Option.GameMode == game2000)
		{
		if(BallStep > 7)
			{
			BallStep = 0;
			switch (BallStyle)
				{
				case ballEXPLODE:
					BallActive = FALSE;
					BallStyle = ballNORMAL;
/*					Player[who].BallStyle = BallStyle;*/
					break;

				case ballLAUNCH:
					BallStyle = Player[who].BallStyle;
					break;
				}
			}
		}

	switch(BallStyle)
		{
		case ballSUPER:
			BallImage = IMG_BallSuper[who][BallStep];
			BallWidth =	BallWidthTab[BallSize];
			BallHeight = BallWidth;
			break;

		case ballEXPLODE:
			BallImage = IMG_BallExplode[who][BallStep];

			BallWidth =	ExplodeWidthTab[BallSize];
			BallHeight = BallWidth;
			break;

		case ballLAUNCH:
			BallImage = (long *)IMG_Paddle[who].BallLaunch[BallStep];

			BallWidth =	PaddleWidthTab[PaddleSize];
			BallHeight = BALL_LAUNCH_HEIGHT;
			break;

		case ballCLASSIC:
		case ballBREAKTHRU:
			BallImage = (long *)IMG_BallClassic;

			BallWidth =	12;
			BallHeight = 12;
			break;

		case ballNORMAL:
		default:
			if(BallSpeedX >= 0)
				BallImage = IMG_BallNormal[who][7-BallStep];
			else
				BallImage = IMG_BallNormal[who][BallStep];

			BallWidth =	BallWidthTab[BallSize];
			BallHeight = BallWidth;
			break;
		}

	Player[who].Ball[which].Active	= BallActive; 
	Player[who].Ball[which].Width	= BallWidth; 
	Player[who].Ball[which].Height	= BallHeight; 
	Player[who].Ball[which].Step	= BallStep; 
	Player[who].Ball[which].Style	= BallStyle; 
	Player[who].Ball[which].Image	= BallImage; 
	}						
 
 	
/*********************************************************************
 *  FUNCTION:	   ExplodeBall
 *
 *  PARAMETERS:	   short who, short which
 *
 *  DESCRIPTION:   Plays explode sound and starts explode animation
 *
 *  RETURNS:	   void
 *
 *********************************************************************/
void ExplodeBall(short ballwho, short ballwhich, short paddlewho)
	{
	int volume;

	BYTE BallActive;
	WORD BallX;
	WORD BallZ;
	WORD BallStep;
	BYTE BallStyle;

	BYTE OtherBallActive;
	BYTE OtherBallStyle;


	BallActive	= Player[ballwho].Ball[ballwhich].Active;
	BallX		= Player[ballwho].Ball[ballwhich].X;
	BallZ		= Player[ballwho].Ball[ballwhich].Z;
	BallStep	= Player[ballwho].Ball[ballwhich].Step;
	BallStyle	= Player[ballwho].Ball[ballwhich].Style;

	OtherBallActive	= Player[ballwho].Ball[ballwhich^1].Active;
	OtherBallStyle	= Player[ballwho].Ball[ballwhich^1].Style;

	if(!BallActive)
		return;

	if(BallStyle == ballEXPLODE)
		return;

	if(Player[ballwho].Goody.Active)
		{
		if(!OtherBallActive || (OtherBallStyle == ballEXPLODE))
			ExplodeGoody(ballwho);
		}

	volume = BallZ*127/10000;
	PlaySound(sndEXPLODE, volume, 0, 7);
	
	SetBallSpeed(ballwho, ballwhich, 0, 0, -10);

	if(Option.GameMode == gameCLASSIC)
		{
		BlankBall(ballwho, ballwhich);
		BallStyle = ballCLASSIC;
		BallActive = FALSE;
		}
	else
		{
		BallStyle  = ballEXPLODE;
		BallStep   = -1;
		}

	if(!OtherBallActive || (OtherBallStyle == ballEXPLODE))
		{
		if(Player[ballwho].Mode == modePONGPLAY)
			{
			Player[ballwho].BonusBalls--;
			UpdateBallCount(ballwho);
			}
		else if(paddlewho == ballwho)
			{
			Player[paddlewho].BallCount--;
			UpdateBallCount(paddlewho);
			ClearPowerUps(paddlewho);
			}
		}

	switch(Player[paddlewho].Mode)
		{
		case modePLAY:
		case modeSHOOT:
			if(paddlewho != ballwho)
				Player[paddlewho].Score -= 1000;
			break;
		}

	Player[ballwho].Ball[ballwhich].Active = BallActive;
	Player[ballwho].Ball[ballwhich].Step = BallStep;
	Player[ballwho].Ball[ballwhich].Style = BallStyle;
	}
	

/*********************************************************************
 *  FUNCTION:		BlankBall
 *
 *  PARAMETERS:		short who, short which
 *
 *  DESCRIPTION:	Removes the ball from the screen and set in inactive
 *
 *  RETURNS:		void
 *
 *********************************************************************/
void BlankBall(short who, short which)
	{
	EraseBall(who, which);
	DisplayBall(who,which);
	Player[who].Ball[which].OldX = Player[who].Ball[which].ScreenX;
	Player[who].Ball[which].OldY = Player[who].Ball[which].ScreenY;
	Player[who].Ball[which].OldW = Player[who].Ball[which].Width;
	Player[who].Ball[which].OldH = Player[who].Ball[which].Height;
	EraseBall(who, which);
	DisplayBall(who, which);
	Player[who].Ball[which].Active = FALSE;
/*	Player[who].Ball[which].Style = ballNORMAL;*/
	}

	
/*********************************************************************
 *  FUNCTION:		LaunchBalls
 *
 *  PARAMETERS:		short who
 *
 *  DESCRIPTION:	Monitor the action button and starts the ball launch seq.
 *
 *  RETURNS:		TRUE = Ball launched
 * 					FALSE = Ball not launched
 *
 *********************************************************************/
UWORD LaunchBalls(short who)
	{
	int pan;
	int volume;
	int BallY;
	int DoLaunch;

	if((Player[who].Mode==modeKILLSCREEN)||(Player[who^1].Mode==modeKILLSCREEN))
		return(FALSE);

	switch (Player[who].Mode)
		{
		case modeSHOOT:
			if(Player[who].BallCount <= 0)
				return(TRUE);
			break;

		case modePONGSHOOT:
			if(Player[who].BonusBalls <= 0)
				return(TRUE);
			break;
		}


	if((Player[who].Paddle.Cracks >= MAX_CRACKS) &&
	   (Player[who].Paddle.Step >= 7))
		{
		Player[who].Paddle.Cracks = 0;
		SetPaddleSize(who, PADDLE_MAX_SIZE-Option.Skill[who]); 
		}

	/* if the B button has not been pressed set the active state *
	 * of all balls to FALSE and return						  */
	if(!(Player[who].Button & BUTTON_B))
		{
		Player[who].Ball[0].Active = FALSE;
		Player[who].Ball[0].Active = FALSE;
		DoLaunch = FALSE;
		}
	else
		DoLaunch = TRUE;

	if(Player[who].AutoPilot && ((ticks-Player[who].Timer) > 4000))
		DoLaunch = TRUE;

	if(DoLaunch)
		{
		/* Set a new random seed based on the horizontal counter value */
		Randomize(*HC);
		
		/* Set the maximum speed for the ball */

/*		BallMaxSpeedX = MIN((BALL_MIN_SPEED+MIN(2,Option.Skill[who])*SKILL_SPEED_INCR)+
						    ScreenCount*8, BallSpeedLimit);

*/		BallMaxSpeedX = MIN(BallSpeedLimit, BALL_MIN_SPEED+
							MIN(2, Option.Skill[who])*SKILL_SPEED_INCR+ScreenCount*2);
		BallMaxSpeedY = BallMaxSpeedX*2;
		BallMaxSpeedZ = BallMaxSpeedX*75/100;
		
		/* Initialize the paddle */
		Player[who].Paddle.Small = FALSE;
		Player[who].Paddle.Hits = 0;

		/* Center the ball on the paddle and start the launch sequence */
		if(who == PLAYER_ON_TOP)
			{
			BallY = Player[who].Paddle.Y-BallRadius;
			Player[who].Ball[0].Sector = sectorTOP;
			}
		else
			{
			BallY = Player[who].Paddle.Y+BallRadius;
			Player[who].Ball[0].Sector = sectorBOTTOM;
			}

		/* Set the ball starting location */
		Player[who].Ball[0].X = Player[who].Paddle.X;
		Player[who].Ball[0].Y = BallY;
		Player[who].Ball[0].Z = Player[who].Paddle.Z-BallRadius;

		if(Player[who].Mode == modeSHOOT)
			SetBallSpeed(who, 0, Player[who].Paddle.SpeedX/2, 0, -BallMaxSpeedZ/2);
		else
			SetBallSpeed(who, 0, Player[who].Paddle.SpeedX/2, 0, -BallMaxSpeedZ);
		

		Player[who].Ball[0].Active  = TRUE;
		Player[who].Ball[0].Style   = ballLAUNCH;
		Player[who].Ball[0].Step	= -1;
		Player[who].Ball[0].ScreenX = 0;
		Player[who].Ball[0].ScreenY = 0;
		Player[who].Ball[0].Width   = PaddleWidthTab[Player[who].Paddle.Size];
		Player[who].Ball[0].Height  = BALL_LAUNCH_HEIGHT;
		Player[who].Ball[0].Caught	= FALSE;

		Player[who].Ball[0].Hits 	= 0;
		Player[who].Ball[0].GoldHits = 0;

		Player[who].Ball[1].Active = FALSE;
		Player[who].Ball[1].Hits 	= 0;
		Player[who].Ball[1].GoldHits = 0;
		
		pan = (HALF_RANGE-Player[who].Ball[0].X)*127/TOTAL_RANGE;
		volume = Player[who].Ball[0].Z*127/10000;
		PlaySound(sndSHOOT, volume, 0, 7);

		SetPaddleHover(who, 32);
		}

	return(DoLaunch);
	}
		
/*********************************************************************
 *  FUNCTION:		MoveBall
 *
 *  PARAMETERS:		short who, short which
 *
 *  DESCRIPTION:	Moves the ball and sets up display data
 *
 *  RETURNS:		void
 *
 *********************************************************************/
void MoveBall(short who, short which)
	{
	int Delta;
	SCREENLOC Loc;
	int OffsetX, OffsetY;
	
	if(!Player[who].Ball[which].Active)
		return;

	/* Set old screen location */
	Player[who].Ball[which].OldX = Player[who].Ball[which].ScreenX;
	Player[who].Ball[which].OldY = Player[who].Ball[which].ScreenY;
	Player[who].Ball[which].OldW = Player[who].Ball[which].Width;
	Player[who].Ball[which].OldH = Player[who].Ball[which].Height;

	Player[who].Ball[which].OldShadowX = Player[who].Ball[which].ShadowX;
	Player[who].Ball[which].OldShadowY = Player[who].Ball[which].ShadowY;

	ApplyAttract(who, which);

	/* If the ball is not captured by the paddle */
	if(!Player[who].Ball[which].Caught)
		{
		/* Apply X, Y and Z axis movement */ 
		switch(Player[who].Ball[which].Sector)
			{
			case sectorTOP:
				Player[who].Ball[which].X += Player[who].Ball[which].SpeedX;
				Delta = -BALL_Y-Player[who].Ball[which].Y;
				Player[who].Ball[which].SpeedY = Delta/4;
				Player[who].Ball[which].Y += Player[who].Ball[which].SpeedY;
				Player[who].Ball[which].Z += Player[who].Ball[which].SpeedZ;
				break;
			case sectorBOTTOM:
				Player[who].Ball[which].X += Player[who].Ball[which].SpeedX;
				Delta = BALL_Y-Player[who].Ball[which].Y;
				Player[who].Ball[which].SpeedY = Delta/4;
				Player[who].Ball[which].Y += Player[who].Ball[which].SpeedY;
				Player[who].Ball[which].Z += Player[who].Ball[which].SpeedZ;
				break;

			case sectorBACK:
				Player[who].Ball[which].X += Player[who].Ball[which].SpeedX;
				Player[who].Ball[which].Y += (Player[who].Ball[which].SpeedY*3);
				Player[who].Ball[which].Z = 
						BallZTab[MIN(34,abs(Player[who].Ball[which].Y/100))];
				break;
			}

		/* check for collisions */
 		CheckPaddleCollision(who, which);           
 		CheckBallLimits(who, which);                
            
        if(Player[who].Mode != modePONGPLAY)
        	{        
        	if(Player[who].Ball[which].SpeedZ<0)
        		CheckBrickCollision(who, which, BACK);
        	else  
	 			CheckBrickCollision(who, which, FRONT); 
	 	    
	 	    if(Player[who].Ball[which].SpeedX<-64)
				CheckBrickCollision(who, which, LEFT);
			else if(Player[who].Ball[which].SpeedX>64)
				CheckBrickCollision(who, which, RIGHT);	 	    	                                            
	 		else if(!CheckBrickCollision(who, which, LEFT))  
	 			CheckBrickCollision(who, which, RIGHT);
	 		} 
		}
	/* if the ball is captured by the paddle */
	else
		{
		CheckPaddleCollision(who, which);
		switch(Player[who].Ball[which].Sector)
			{
			case sectorBOTTOM:
				Player[who].Ball[which].X = Player[PLAYER_ON_BOTTOM].Paddle.X;
				Player[who].Ball[which].Y = Player[PLAYER_ON_BOTTOM].Paddle.Y+(BallRadius*2);
				Player[who].Ball[which].Z = Player[PLAYER_ON_BOTTOM].Paddle.Z-BallRadius;
				break;

			case sectorTOP:
				Player[who].Ball[which].X = Player[PLAYER_ON_TOP].Paddle.X;
				Player[who].Ball[which].Y = Player[PLAYER_ON_TOP].Paddle.Y-(BallRadius*2);
				Player[who].Ball[which].Z = Player[PLAYER_ON_TOP].Paddle.Z-BallRadius;
				break;
			}
		}

	LimitBallSpeed(who, which);

	/* Set the ball scale */
	Player[who].Ball[which].Size = MAX(0, MIN(6,
								GetDepthScale(Player[who].Ball[which].Z)/9-5));
	
	SetBallImage(who, which);

	/* Set the new screen location */
	Loc = PlayfieldToScreen(Player[who].Ball[which].X, 
					Player[who].Ball[which].Y, Player[who].Ball[which].Z);

	OffsetX = Player[who].Ball[which].Width/2;
	OffsetY = Player[who].Ball[which].Height/2;

	Player[who].Ball[which].ScreenX = Loc.X-OffsetX;
	Player[who].Ball[which].ShadowX = Player[who].Ball[which].ScreenX;

	if(Player[who].Ball[which].Y >= BallRadius)
		{
	 	Player[who].Ball[which].ScreenY = MIN(Loc.Y-OffsetY, 
					PLAYFIELD_BUFFER_HEIGHT-Player[who].Ball[which].Height-1);

		Loc = PlayfieldToScreen(Player[who].Ball[which].X, 
					Player[who].Ball[which].Y+448, Player[who].Ball[which].Z);
		Player[who].Ball[which].ShadowY = 
					MIN(Loc.Y-3, PLAYFIELD_BUFFER_HEIGHT-BALL_SHADOW_HEIGHT-1);
		}
	else if(Player[who].Ball[which].Y <= -BallRadius)
		{
		Player[who].Ball[which].ScreenY = MAX(Loc.Y+OffsetY,
					Player[who].Ball[which].Height);

		Loc = PlayfieldToScreen(Player[who].Ball[which].X, 
					Player[who].Ball[which].Y-448, Player[who].Ball[which].Z);
		Player[who].Ball[which].ShadowY = MAX(Loc.Y+3, BALL_SHADOW_HEIGHT);
		}
	else
		{
		Player[who].Ball[which].ScreenY = 100;
		}
	}


/*********************************************************************
 *  FUNCTION:		CheckBallLimits
 *
 *  PARAMETERS:		short who, short which
 *
 *  DESCRIPTION:	Ensure that the ball stays within the playfield		
 *
 *  RETURNS:		void
 *
 *********************************************************************/
void CheckBallLimits(short who, short which)
	{
	short paddlewho;
	int pan;
	int volume;

	/* If the ball is moving in either the top or bottom plane we   */
	/* need to check for a Collision with the back of the playfield */
	if(Player[who].Ball[which].Sector != sectorBACK)
		{		
		/* If the Ball has hit the back of the playfield.           */
		if(Player[who].Ball[which].Z < (PLAYFIELD_BACK-BallRadius))
			{
			/* If Screen Wrap mode is on, Swap the Z and Y vectors */
			if(Option.ScreenWrapOn)
				{
				volume = Player[who].Ball[which].Z*127/10000;
				PlaySound(sndLOOP, volume, 0, 5);

				Player[who].Ball[which].Z = PLAYFIELD_BACK+BallRadius;
				if(Player[who].Ball[which].Y > 0)
					Player[who].Ball[which].SpeedY = -SPEED_BACK_WALL;
				else
					Player[who].Ball[which].SpeedY = SPEED_BACK_WALL;
				
				Player[who].Ball[which].OldSpeedZ = Player[who].Ball[which].SpeedZ;
				Player[who].Ball[which].SpeedZ = 0;
				Player[who].Ball[which].Sector = sectorBACK;
				}
			
			/* If Screen Wrap mode is NOT on, the Ball bounces forward. */
			else
				{
				pan = (HALF_RANGE-Player[who].Ball[which].X)*127/TOTAL_RANGE;
				volume = Player[who].Ball[which].Z*127/10000;
				PlaySound(sndWALL, volume, 0, 5);
			
				Player[who].Ball[which].Z = PLAYFIELD_BACK+BallRadius;
				if(Player[who].Mode == modePLAY)
					Player[who].Ball[which].SpeedZ = BallMaxSpeedZ;

				StartRipple(Player[who].Ball[which].X, Player[who].Ball[which].Y,
						Player[who].Ball[which].Z, rippleBACK);

				Player[who].Ball[which].Hits = 0;
				}

			/* In either case, if this is the first time the back wall is */
			/* hit, reduce the size of the paddle and set the Small flag. */
			if (!Player[who].Paddle.Small)
				{
				Player[who].Paddle.Small = TRUE;
				SetPaddleSize(who, Player[who].Paddle.Size-1);
				}
			}
		/* If the Ball has NOT hit the Back of the playfield, check to  */
		/* see if it as hit the front of the playfield.                 */
		else if(Player[who].Ball[which].Z > Player[who].Paddle.Z/*PLAYFIELD_FRONT*/)
			{
			if(Player[who].Ball[which].Sector == sectorBOTTOM)
				paddlewho = PLAYER_ON_BOTTOM;
			else
				paddlewho = PLAYER_ON_TOP;

			Player[who].Ball[which].Z = Player[who].Paddle.Z-32;
			ExplodeBall(who, which, paddlewho);
			}
		}

	/* Otherwise the ball in in the back sector and we need to check for */
	/* a collision with the top or botton                                */
	else
		{
		if(Player[who].Ball[which].Y < PLAYFIELD_TOP)
			{
			Player[who].Ball[which].Y = PLAYFIELD_TOP;
			Player[who].Ball[which].Sector = sectorTOP;
			Player[who].Ball[which].SpeedY = 0;
			Player[who].Ball[which].SpeedZ = abs(Player[who].Ball[which].OldSpeedZ);
			}
		else if (Player[who].Ball[which].Y > PLAYFIELD_BOTTOM)
			{
			Player[who].Ball[which].Y = PLAYFIELD_BOTTOM;
			Player[who].Ball[which].Sector = sectorBOTTOM;
			Player[who].Ball[which].SpeedY = 0;
			Player[who].Ball[which].SpeedZ = abs(Player[who].Ball[which].OldSpeedZ);
			}
		}


	if(Player[who].Ball[which].X < (PLAYFIELD_LEFT+BallRadius))
		{
		/* Ball hits Left side */
		volume = Player[who].Ball[which].Z*127/10000;
		PlaySound(sndWALL, volume, 0, 5);
		Player[who].Ball[which].X = PLAYFIELD_LEFT+BallRadius;

		Player[who].Ball[which].SpeedX = abs(Player[who].Ball[which].SpeedX);
		StartRipple(Player[who].Ball[which].X, Player[who].Ball[which].Y,
						Player[who].Ball[which].Z, rippleLEFT);

		Player[who].Ball[which].Hits = 0;
		}
	else if(Player[who].Ball[which].X > (PLAYFIELD_RIGHT-BallRadius))
		{
		/* Ball hits Right side */
		volume = Player[who].Ball[which].Z*127/10000;
		PlaySound(sndWALL, volume, 0, 5);
		Player[who].Ball[which].X = PLAYFIELD_RIGHT-BallRadius;

		Player[who].Ball[which].SpeedX = -abs(Player[who].Ball[which].SpeedX);
		StartRipple(Player[who].Ball[which].X, Player[who].Ball[which].Y,
						Player[who].Ball[which].Z, rippleRIGHT);

		Player[who].Ball[which].Hits = 0;
		}
	}
	
/*********************************************************************
 *  FUNCTION:		CheckBrickCollision
 *
 *  PARAMETERS:		short who, int SideFlag, short which
 *
 *  DESCRIPTION:	Handle Ball to Brick collisions		
 *
 *  RETURNS:		void
 *
 *********************************************************************/
BYTE CheckBrickCollision(short who, short which, char SideFlag)
	{
	UWORD row, col;
	int X, Z;
	
	int speed;
	int maxspeed;
	int Bounce;
	int Sign;
	BYTE BrickStyle;

	short brickwho;

	int Points;

	WORD BallX;
	WORD BallZ;
	WORD BallSpeedX;
	WORD BallSpeedZ;
	WORD BallHits;
	WORD BallGoldHits;
	UWORD BallStyle;
	BYTE BallSector;
	
	BallX	   		= Player[who].Ball[which].X;
	BallZ	   		= Player[who].Ball[which].Z;
	BallSpeedX 		= Player[who].Ball[which].SpeedX;
	BallSpeedZ 		= Player[who].Ball[which].SpeedZ;
	BallHits		= Player[who].Ball[which].Hits;
	BallGoldHits	= Player[who].Ball[which].GoldHits;
 	BallStyle   	= Player[who].Ball[which].Style;
	BallSector 		= Player[who].Ball[which].Sector;

	if(BallStyle == ballEXPLODE)
		return(FALSE);

	switch(BallSector)
		{
		case sectorBOTTOM:
			brickwho = PLAYER_ON_BOTTOM;
			break;

		case sectorTOP:
			brickwho = PLAYER_ON_TOP;
			break;

		case sectorBACK:
		default:
			return(FALSE);
			break;
		}

	X = BallX + abs(PLAYFIELD_LEFT);
	Z = BallZ + BRICK_ADEPTH-BRICK_ORIGIN_Z;

	switch(SideFlag)
		{
		case BACK:
			row = (Z-BallRadius)/BRICK_ADEPTH;
			break;


		case FRONT:
			row = (Z+BallRadius)/BRICK_ADEPTH;
			break;

		case RIGHT:
		case LEFT:
		default:
			row = Z/BRICK_ADEPTH;
			break;
		}
	
	if(row >= BRICK_ROWS)
		return(FALSE);

	switch(SideFlag)
		{
		case BACK:
		case FRONT:
			col = X/BRICK_AWIDTH;
			speed = abs(BallSpeedZ);
			maxspeed = BallMaxSpeedZ;
			Sign = SIGN(BallSpeedZ);
			break;
		
		case RIGHT:
			col = (X+BallRadius)/BRICK_AWIDTH;
			speed = abs(BallSpeedX);
			maxspeed = BallMaxSpeedX;
			Sign = SIGN(BallSpeedX);
			break;
		
		case LEFT:
		default:
			col = (X-BallRadius)/BRICK_AWIDTH;
			speed = abs(BallSpeedX);
			maxspeed = BallMaxSpeedX;
			Sign = SIGN(BallSpeedX);
			break;  
		}

	if(col >= BRICK_COLS)
		return(FALSE);
	
	BrickStyle = Player[brickwho].Brick[0][col][row].Style;
	if((BallStyle == ballSUPER) && (BrickStyle > SILVER1) && (BrickStyle <= SILVER9))
		Player[brickwho].Brick[0][col][row].Style = SILVER1;

	if(BrickStyle >= DROP)
		return(FALSE);

	/* Set new speed and score */
	if(BrickStyle < GOLD)
		{
		speed = MIN(maxspeed, MAX(speed, BALL_MIN_SPEED+(BrickStyle/2)*32+
							MIN(2,Option.Skill[who])*SKILL_SPEED_INCR));

		if(Option.GameMode == gameCLASSIC)
			{
			switch(BrickStyle)
				{
				default:
				case 0:
				case 1:
					Points = 1;
					break;

				case 2:
				case 3:
					Points = 4;
					break;

				case 4:
				case 5:
					Points = 7;
					break;
				}
			}
		else
			Points = (5+BrickStyle/2*5);

		StartDroid(Random(60*Option.NumPlayers));
			
		BallGoldHits = 0;
		}
	else if(BrickStyle == GOLD)
		{
		if((++BallGoldHits) > 10)
			BallGoldHits = 0;
		Points = 1;
		}
	else
		{
		Points = 5;
		BallGoldHits = 0;
		}
	
	EraseBrick(brickwho, 0, col, row);
	Player[who].Score += Points;

	/* Set the bounce flag */
	if(BrickStyle == GOLD)
		{
		Bounce = TRUE;
		}
	else if((BallStyle == ballSUPER) ||
			(BallStyle == ballBREAKTHRU) ||
			(Player[who].BallStyle == ballSUPER) )
		Bounce = FALSE;
	else
		Bounce = TRUE;

	if(Bounce)
		{
		if((BrickStyle == GOLD) && (BallGoldHits == 0))
			{
			BallSpeedX = Random(BallSpeedLimit*2)-BallSpeedLimit;
			BallSpeedZ = Random(BallSpeedLimit*2)-BallSpeedLimit;
			}
		else
			{
			switch(SideFlag)
				{
				case BACK:
					BallSpeedZ = speed;
					BallZ += (BRICK_ADEPTH-((Z-BallRadius)%BRICK_ADEPTH));
					break;
				
				case FRONT:
					BallSpeedZ = -speed;
					BallZ -= ((Z+BallRadius)%BRICK_ADEPTH);
					break;
				
				case RIGHT:
					BallSpeedX = -speed;
					BallX -= ((X+BallRadius)%BRICK_AWIDTH);
					break;
				
				case LEFT:
					BallSpeedX = speed;
					BallX += (BRICK_AWIDTH-((X-BallRadius)%BRICK_AWIDTH));
					break;
				}
			}
		}

	Player[who].Ball[which].X		= BallX;
	Player[who].Ball[which].Z		= BallZ;
	Player[who].Ball[which].SpeedX	= BallSpeedX; 
	Player[who].Ball[which].SpeedZ	= BallSpeedZ; 
	Player[who].Ball[which].Hits	= BallHits;
	Player[who].Ball[which].GoldHits= BallGoldHits;

	return(TRUE);
	}


/*********************************************************************
 *  FUNCTION:		CheckPaddleCollision
 *
 *  PARAMETERS:		short ballwho, short which
 *
 *  DESCRIPTION:	Handle Ball to Paddle Collisions		
 *
 *  RETURNS:		void
 *
 *********************************************************************/
void CheckPaddleCollision(short ballwho, short which)
	{
	short paddlewho;
	int X, Z;
	int Pwidth;
	int Delta;
	int volume;
	int CrackDeviation;
	int MaxDeviation;
	int CountFactor;
	
	if(Player[ballwho].Ball[which].Style == ballEXPLODE)
		return;

	switch(Player[ballwho].Ball[which].Sector)
		{
		case sectorBOTTOM:
			paddlewho = PLAYER_ON_BOTTOM;
			break;

		case sectorTOP:
		default:
			paddlewho = PLAYER_ON_TOP;
			break;
		}

	if(Player[ballwho].Ball[which].Z < Player[paddlewho].Paddle.Z-BallRadius)
		return;

	if(Player[paddlewho].Paddle.Cracks >= MAX_CRACKS)
		return;

	if(Player[ballwho].Ball[which].Caught)
		{
		if(!(Player[paddlewho].Button & BUTTON_C))
			{
			Player[ballwho].Ball[which].Caught = FALSE;
			if (Player[paddlewho].Paddle.SpeedX == 0)
				SetBallSpeed(ballwho, which,Random(BallMaxSpeedX)-(BallMaxSpeedX/2),
							 0, -BallMaxSpeedZ/2);
			else
				SetBallSpeed(ballwho, which, Player[paddlewho].Paddle.SpeedX, 0, 
							-BallMaxSpeedZ/2);

			/* Move the ball off the paddle */
			Player[ballwho].Ball[which].Z -= BallRadius;
			}
		return;
		}

	X = Player[ballwho].Ball[which].X;
	Z = Player[ballwho].Ball[which].Z;
	
	Pwidth = Player[paddlewho].Paddle.Size*PADDLE_STEP_W+PADDLE_MIN_W;
								
	/* If above the paddle */
	
	if(Player[paddlewho].Paddle.Active)
		{
		/* If Left of Paddle */
		if((X+BallRadius) < (Player[paddlewho].Paddle.X-(Pwidth/2)))
			return;
		
		/* If right of Paddle */
		if((X-BallRadius) > Player[paddlewho].Paddle.X+(Pwidth/2))
			return;
		}
	
	/* Getting here means the ball has hit the paddle */
	volume = Player[ballwho].Ball[0].Z*127/1000;
	if(Player[paddlewho].Paddle.Cracks)
		PlaySound(sndCRACK, volume, 0, 6);
	else
		PlaySound(sndPADDLE, volume, 0, 6);

	switch(Player[paddlewho].Mode)
		{
		case modePONGSHOOT:
		case modePONGPLAY:
		case modePONGEND:
			Player[paddlewho].Score += 100;
			Player[ballwho].Ball[which].SpeedZ += 25;
			break;

		default:
			if(Option.GameMode != gameCLASSIC)
				{
				if(Player[paddlewho].Paddle.Active)
					{
					Player[paddlewho].Score+=1;
					Player[paddlewho].Paddle.Hits++;
					}
				}
			break;
		}

	Player[ballwho].Ball[which].SpeedZ =
								-abs(Player[ballwho].Ball[which].SpeedZ);

	if(Player[paddlewho].Paddle.Active)
		{
		if(Player[paddlewho].Paddle.Hits < 3)
			CountFactor = 2;
		else if (Player[paddlewho].Paddle.Hits < 7)
			CountFactor = 3;
		else if (Player[paddlewho].Paddle.Hits < 11)
			CountFactor = 4;
		else if (Player[paddlewho].Paddle.Hits < 12)
			CountFactor = 5;
		else
			CountFactor = 2;

		Delta = X-Player[paddlewho].Paddle.X;
		Player[ballwho].Ball[which].SpeedX = (BallSpeedLimit)*Delta/(Pwidth/CountFactor);
		
		/* if the paddle is cracked, add a random factor*/
		if(Player[paddlewho].Paddle.Cracks)
			{
			MaxDeviation = BallMaxSpeedX*Player[paddlewho].Paddle.Cracks;
			CrackDeviation = Random(MaxDeviation)-(MaxDeviation/2);

			Player[ballwho].Ball[which].SpeedX += CrackDeviation;
			}

		if(abs(Player[ballwho].Ball[which].SpeedX)<10)
			{
			if(Random(1))
				Player[ballwho].Ball[which].SpeedX = 10;
			else
				Player[ballwho].Ball[which].SpeedX = -10;
			}

		if( (Player[paddlewho].Paddle.X > PLAYFIELD_RIGHT-PADDLE_MARGIN-64-Pwidth/2) ||
			(Player[paddlewho].Paddle.X < PLAYFIELD_LEFT+PADDLE_MARGIN+64+Pwidth/2) )
			{
			if(Option.GameMode != gameCLASSIC)
				Player[ballwho].Ball[which].SpeedX += Random(256)-128;
			}
		}
	else
		{
		StartRipple(Player[ballwho].Ball[which].X, Player[ballwho].Ball[which].Y,
					Player[ballwho].Ball[which].Z, rippleFRONT);
		}
	Player[ballwho].Ball[which].Z = (Player[paddlewho].Paddle.Z-BallRadius);

	SetPaddleHover(paddlewho, 32);

	Player[ballwho].Ball[which].Hits = 0;
	
	if((Player[paddlewho].Button & BUTTON_C) && Player[paddlewho].Paddle.Catch)
		Player[ballwho].Ball[which].Caught = TRUE;
	}


/*********************************************************************
 *  FUNCTION: 	   CheckBallCollision
 *
 *  PARAMETERS:	   short who1
 *                 short which1
 *                 short who2
 *                 short which2
 *
 *  DESCRIPTION:   Checks for a collision between two balls
 *
 *  RETURNS:	   void
 *
 *********************************************************************/
void CheckBallCollision( short who1, short which1, short who2, short which2 )
	{
	int ax, ay, az;
	int sx, sy, sz;
	int volume;

	if(!Player[who1].Ball[which1].Active)
		return;
	if(!Player[who2].Ball[which2].Active)
		return;

	if(Player[who1].Ball[which1].Style == ballEXPLODE)
		return;
	if(Player[who2].Ball[which2].Style == ballEXPLODE)
		return;

	if( abs(Player[who1].Ball[which1].Y-Player[who2].Ball[which2].Y)>BALL_W )
		return;
	if( abs(Player[who1].Ball[which1].X-Player[who2].Ball[which2].X)>BALL_W )
		return;
	if( abs(Player[who1].Ball[which1].Z-Player[who2].Ball[which2].Z)>BALL_W )
		return;

	/* Getting here means the balls have collided */
	volume = Player[who1].Ball[which1].Z*127/10000;
	PlaySound(sndBALL2BALL, volume, 0, 5);

	/* Swap speed vectors */
	sx = Player[who1].Ball[which1].SpeedX;
	sy = Player[who1].Ball[which1].SpeedY;
	sz = Player[who1].Ball[which1].SpeedZ;

	Player[who1].Ball[which1].SpeedX = Player[who2].Ball[which2].SpeedX;
	Player[who1].Ball[which1].SpeedY = Player[who2].Ball[which2].SpeedY;
	Player[who1].Ball[which1].SpeedZ = Player[who2].Ball[which2].SpeedZ;

	Player[who2].Ball[which2].SpeedX = sx;
	Player[who2].Ball[which2].SpeedY = sy;
	Player[who2].Ball[which2].SpeedZ = sz;

	/* Offset balls from collision center */
	ax = (Player[who1].Ball[which1].X+Player[who2].Ball[which2].X)/2;
	ay = (Player[who1].Ball[which1].Y+Player[who2].Ball[which2].Y)/2;
	az = (Player[who1].Ball[which1].Z+Player[who2].Ball[which2].Z)/2;

	if(Player[who1].Ball[which1].SpeedX < Player[who2].Ball[which2].SpeedX)
		{
		Player[who1].Ball[which1].X = ax-BallRadius-1;
		Player[who2].Ball[which2].X = ax+BallRadius+1;
		}
	else
		{
		Player[who1].Ball[which1].X = ax+BallRadius+1;
		Player[who2].Ball[which2].X = ax-BallRadius-1;
		}
	}



/*********************************************************************
 *  FUNCTION:		EraseBall
 *
 *  PARAMETERS:		short who, short which
 *
 *  DGSCRIPTION:	Erase the ball image from the playfield buffer	
 *
 *  RETURNS:		void
 *
 *********************************************************************/
void EraseBall(short who, short which)
	{
	WORD  Height;

	int EndX;
	int Offset;
	int StepWidth;

	int a1_step;

	if(!Player[who].Ball[which].Active)
		return;

	if(!Player[who].Ball[which].OldX || !Player[who].Ball[which].OldY)
		return;

	EndX = Player[who].Ball[which].OldX+Player[who].Ball[which].OldW;
	Offset = EndX & 3;
	if(Offset)
		StepWidth = Player[who].Ball[which].OldW+4-Offset;
	else
		StepWidth = Player[who].Ball[which].OldW;
	
	if((Player[who].Ball[which].OldY >= 100) || (Option.GameMode == gameCLASSIC))
		{
		Height = MIN(Player[who].Ball[which].OldH+8, 
					PLAYFIELD_BUFFER_HEIGHT-Player[who].Ball[which].OldY-1);
		a1_step = (1<<16L) | (-StepWidth & 0x0FFFFL);
		}
	else
		{
		Height = MIN(Player[who].Ball[which].OldH+8, 
							Player[who].Ball[which].OldY);
		a1_step = (-1<<16L) | (-StepWidth & 0x0FFFFL);
		}
		

	Blit[BlitAddIndex].Type		= blitBITMAP;

	Blit[BlitAddIndex].A1_Base	= (long)IMG_PlayfieldBuffer;
	Blit[BlitAddIndex].A1_Flags	= PITCH2 | PIXEL16 | WID320 | XADDPHR | ZOFFS1;
	Blit[BlitAddIndex].A1_Pixel	= (Player[who].Ball[which].OldY<<16) | 
								  (Player[who].Ball[which].OldX & 0xFFFFL);
	Blit[BlitAddIndex].A1_Step	= a1_step;

	Blit[BlitAddIndex].A2_Base	= (long)IMG_BrickBuffer;
	Blit[BlitAddIndex].A2_Flags	= Blit[BlitAddIndex].A1_Flags;
	Blit[BlitAddIndex].A2_Pixel	= Blit[BlitAddIndex].A1_Pixel;
	Blit[BlitAddIndex].A2_Step	= Blit[BlitAddIndex].A1_Step;

	Blit[BlitAddIndex].B_Count	= (Height<<16) | 
								  (Player[who].Ball[which].OldW & 0xFFFFL);
										 
	Blit[BlitAddIndex].B_Patd	= 0x0L;
	Blit[BlitAddIndex].B_Cmd	= LFU_REPLACE | UPDA1 | UPDA2 | SRCEN | 
								  DSTEN | SRCENZ | DSTWRZ;
	
	QueBlit();
	}

/*********************************************************************
 *  FUNCTION:		DrawBall
 *
 *  PARAMETERS:		short who, short which
 *
 *  DESCRIPTION:	Draw ball into the playfield buffer	
 *
 *  RETURNS:		void
 *
 *********************************************************************/
void DrawBall( short who, short which )
	{
	int Width,Height;
	int a1_step;
	int a2_flags;
	int a2_pixel;

	if(!Player[who].Ball[which].Active)
		return;

	Width = Player[who].Ball[which].Width;
	Height = Player[who].Ball[which].Height;

	/* Blit the ball image */
	if((Player[who].Ball[which].ScreenY >= 100) || (Option.GameMode == gameCLASSIC))
		a1_step	= (1<<16L) | (-Width & 0xFFFFL);
	else
		a1_step	= (-1<<16L) | (-Width & 0xFFFFL);
										 
	switch(Player[who].Ball[which].Style)
		{
		case ballEXPLODE:
			a2_flags = PITCH1 | PIXEL16 | WID80 | XADDPIX;
			a2_pixel = ExplodeOffsetTab[Player[who].Ball[which].Size];
			break;

		case ballLAUNCH:
			a2_flags = PITCH1 | PIXEL16 | WID64 | XADDPIX;
			a2_pixel = 0x0L;
			break;
		
		case ballCLASSIC:
		case ballBREAKTHRU:
			a2_flags = PITCH1 | PIXEL16 | WID12 | XADDPIX;
			a2_pixel = 0x0L;
			break;

		case ballNORMAL:
		case ballSUPER:
		default:
			a2_flags = PITCH1 | PIXEL16 | WID80 | XADDPIX;
			a2_pixel = BallOffsetTab[Player[who].Ball[which].Size];
			break;
		}

	Blit[BlitAddIndex].Type		= blitBITMAP;

	Blit[BlitAddIndex].A1_Base	= (long)IMG_PlayfieldBuffer;
	Blit[BlitAddIndex].A1_Flags	= PITCH2 | PIXEL16 | WID320 | XADDPIX | ZOFFS1;
	Blit[BlitAddIndex].A1_Pixel	= (Player[who].Ball[which].ScreenY<<16L) | 
				 				  (Player[who].Ball[which].ScreenX & 0x0FFFFL);
	Blit[BlitAddIndex].A1_Step  = a1_step;

	Blit[BlitAddIndex].A2_Base	= (long)Player[who].Ball[which].Image;
	Blit[BlitAddIndex].A2_Flags	= a2_flags;
	Blit[BlitAddIndex].A2_Pixel	= a2_pixel;
	Blit[BlitAddIndex].A2_Step	= (1<<16) | (-Width & 0xFFFFL);

	Blit[BlitAddIndex].B_Count	= (Height<<16) | Width;
										 
	Blit[BlitAddIndex].B_Patd	= 0x0L;
	Blit[BlitAddIndex].B_Srcz1	= Player[who].Ball[which].Z;
	Blit[BlitAddIndex].B_Cmd	= LFU_REPLACE | UPDA1 | UPDA2 | SRCEN | DSTEN | 
								  DCOMPEN | DSTENZ | DSTWRZ | ZMODELT;
	
	QueBlit();


	if(Width > 12 )
		return;

	if((Player[who].Ball[which].ScreenY>80) && (Player[who].Ball[which].ScreenY<120))
		return;

	/* Now blit the ball shadow */
	/* First Blit the Background */
	Blit[BlitAddIndex].Type		= blitBITMAP;

	Blit[BlitAddIndex].A1_Base	= (long)IMG_PlayfieldBuffer;
	Blit[BlitAddIndex].A1_Flags	= PLAYFIELD_BUFFER_FLAGS;
	Blit[BlitAddIndex].A1_Pixel	= (Player[who].Ball[which].ShadowY<<16L) | 
								  (Player[who].Ball[which].ShadowX & 0xFFFFL);
	Blit[BlitAddIndex].A1_Step	= a1_step;
										 
	Blit[BlitAddIndex].A2_Base	= (long)IMG_Background;
	Blit[BlitAddIndex].A2_Flags	= BACKGROUND_FLAGS;
	Blit[BlitAddIndex].A2_Pixel	= ((Player[who].Ball[which].ShadowY+20)<<16L) | 
								  (Player[who].Ball[which].ShadowX & 0xFFFFL);
	Blit[BlitAddIndex].A2_Step	= Blit[BlitAddIndex].A1_Step;

	Blit[BlitAddIndex].B_Count	= (BALL_SHADOW_HEIGHT<<16L) | (Width & 0xFFFFL);
										 
	Blit[BlitAddIndex].B_Patd	= 0x0L;
	Blit[BlitAddIndex].B_Srcz1	= 0x0L;
	Blit[BlitAddIndex].B_Cmd	= LFU_REPLACE | UPDA1 | UPDA2 | SRCEN | 
								  DSTEN | DSTWRZ | DSTENZ | ZMODELT;
	
	QueBlit();

	/* Then Blit the Shadow */
	Blit[BlitAddIndex].Type		= blitBITMAP;

	Blit[BlitAddIndex].A1_Base	= (long)IMG_PlayfieldBuffer;
	Blit[BlitAddIndex].A1_Flags	= PLAYFIELD_BUFFER_FLAGS;
	Blit[BlitAddIndex].A1_Pixel	= (Player[who].Ball[which].ShadowY<<16L) | 
								  (Player[who].Ball[which].ShadowX & 0xFFFFL);
	Blit[BlitAddIndex].A1_Step	= a1_step;
										 
	Blit[BlitAddIndex].A2_Base	= (long)IMG_BallShadow;
	Blit[BlitAddIndex].A2_Flags	= BALL_SHADOW_FLAGS;
	Blit[BlitAddIndex].A2_Pixel	= 0xFFFFL & BallOffsetTab[Player[who].Ball[which].Size];
	Blit[BlitAddIndex].A2_Step	= (1<<16) | (-Width & 0xFFFFL);

	Blit[BlitAddIndex].B_Count	= (BALL_SHADOW_HEIGHT<<16L) | (Width & 0xFFFFL);
										 
	Blit[BlitAddIndex].B_Patd	= 0x0L;
	Blit[BlitAddIndex].B_Srcz1	= 0x10L;
	Blit[BlitAddIndex].B_Cmd	= LFU_REPLACE | UPDA1 | UPDA2 | SRCEN | 
								  DSTEN | DCOMPEN | DSTENZ | ZMODELT | 
								  DSTWRZ | ADDDSEL;
	
	QueBlit();
	}

/*********************************************************************
 *  FUNCTION:		DisplayBall
 *
 *  PARAMETERS:		short who, short which
 *
 *  DESCRIPTION:	Show ball image in the playfield	
 *
 *  RETURNS:		void
 *
 *********************************************************************/
void DisplayBall( short who, short which )
	{
	SCREENLOC Loc;
	int DeltaX, DeltaY;
	int Width, Height;

	ULONG step;

	int EndX;
	int Offset;
	int StepWidth;

	if(!Player[who].Ball[which].Active)
		return;

	if(!Player[who].Ball[which].OldX)
		Player[who].Ball[which].OldX = Player[who].Ball[which].ScreenX;

	if(!Player[who].Ball[which].OldY)
		Player[who].Ball[which].OldY = Player[who].Ball[which].ScreenY;

	DeltaX = abs(Player[who].Ball[which].ScreenX-Player[who].Ball[which].OldX);
	DeltaY = abs(Player[who].Ball[which].ScreenY-Player[who].Ball[which].OldY);

	Loc.X = MIN(Player[who].Ball[which].ScreenX, Player[who].Ball[which].OldX);

	Width = MAX(Player[who].Ball[which].Width,Player[who].Ball[which].OldW)+DeltaX;
	Width = MIN(Width, PLAYFIELD_BUFFER_WIDTH-Loc.X-1);

	Height = MAX(Player[who].Ball[which].Height,
										Player[who].Ball[which].OldH)+DeltaY+8;

	EndX = Loc.X+Width;
	Offset = EndX & 3;
	if(Offset)
		StepWidth = Width+4-Offset;
	else
		StepWidth = Width;
	
	if((Player[who].Ball[which].ScreenY >= 100) || (Option.GameMode == gameCLASSIC))
		{
		if((Player[who].Ball[which].OldY >= 100) || (Option.GameMode == gameCLASSIC))
			{
			step = (1<<16L) | (-StepWidth & 0x0FFFFL);
			Loc.Y = MIN(Player[who].Ball[which].ScreenY, 
						Player[who].Ball[which].OldY);
			}
		else
			{
			step = (1<<16L) | (-StepWidth & 0x0FFFFL);
			Loc.Y = Player[who].Ball[which].OldY-Player[who].Ball[which].OldH;
			Height += Player[who].Ball[which].OldH;
			}

		Height = MIN(Height, PLAYFIELD_BUFFER_HEIGHT-Loc.Y-1);
		}
	else
		{
		if((Player[who].Ball[which].OldY >= 100) || (Option.GameMode == gameCLASSIC))
			{
			step = (-1<<16L) | (-StepWidth & 0x0FFFFL);
			Loc.Y = Player[who].Ball[which].OldY+Player[who].Ball[which].OldH;
			Height += Player[who].Ball[which].OldH;
			}
		else
			{
			step = (-1<<16L) | (-StepWidth & 0x0FFFFL);
			Loc.Y = MAX(Player[who].Ball[which].ScreenY, 
					Player[who].Ball[which].OldY);
			}

		Height = MIN(Height, Loc.Y);
		}

	Blit[BlitAddIndex].Type		= blitBITMAP;

	Blit[BlitAddIndex].A1_Base	= (long)IMG_Playfield;
	Blit[BlitAddIndex].A1_Flags	= PITCH1 | PIXEL16 | WID320 | XADDPHR;
	Blit[BlitAddIndex].A1_Pixel	= ((Loc.Y+20)<<16) | (Loc.X & 0xFFFFL);
	Blit[BlitAddIndex].A1_Step	= step;
										 
	Blit[BlitAddIndex].A2_Base	= (long)IMG_PlayfieldBuffer;
	Blit[BlitAddIndex].A2_Flags	= PITCH2 | PIXEL16 | WID320 | XADDPHR | ZOFFS1;
	Blit[BlitAddIndex].A2_Pixel	= (Loc.Y<<16) | (Loc.X & 0xFFFFL);
	Blit[BlitAddIndex].A2_Step	= Blit[BlitAddIndex].A1_Step;

	Blit[BlitAddIndex].B_Count	= (Height<<16) | (Width & 0xFFFFL);
										 
	Blit[BlitAddIndex].B_Patd	= 0x0L;
	Blit[BlitAddIndex].B_Cmd	= LFU_REPLACE | UPDA1 | UPDA2 | SRCEN;
	
	QueBlit();
	}
