/*******************************************************************
 *
 *	TITLE:			Bomb.c
 *
 *	DESCRIPTION:	Handlers for the bomb
 *
 *	AUTHOR:			Mario Perdue  Richard Degler
 *
 *	HISTORY:		
 *
 *
 *		   COPYRIGHT 1994, 1995 MP Graphics Systems		 
 *	  UNATHORIZED REPRODUCTION, ADAPTATION, DISTRIBUTION,
 *	  PERFORMANCE OR DISPLAY OF THIS COMPUTER PROGRAM OR   
 *	 THE ASSOCIATED AUDIOVISUAL WORK IS STRICTLY PROHIBITED.
 *						ALL RIGHTS RESERVED.			   
 *******************************************************************/

/** include files      **/
#include "\jaguar\include\jaguar.h"
#include "bomb.h"
#include "brick.h"
#include "support.h"
#include "paddle.h"
#include "player.h"
#include "option.h"
#include "ripple.h"
#include "b2k_game.h"
#include "displays.h"

/** default settings   **/

/** external data      **/
extern	WORD			IMG_Bomb[8][144];			/* Bomb images array */
extern	WORD			IMG_BombExplode[8][2432];	/* Bomb images array */
extern	DEF_RAM_PADDLE	IMG_Paddle[MAX_PLAYERS];

extern	WORD			IMG_Playfield[];
extern	WORD  			IMG_PlayfieldBuffer[];
extern	WORD  			IMG_BrickBuffer[];

extern	DEF_PLAYER		Player[2];

extern 	int 			ScreenOrgX;
extern 	int				ScreenOrgY;

extern	DEF_OPTION		Option;

extern BYTE				PlayerOnTop;	

extern	DEF_BLIT_DATA	Blit[];
extern	LONG			BlitAddIndex;
extern	LONG			BlitUseIndex;

extern	WORD			PaddleWidthTab[5];


/** internal functions **/
/*********************************************************************
 *  FUNCTION:		LimitBombSpeed
 *
 *  PARAMETERS:		short who
 *
 *  DESCRIPTION:	Limits the bomb speed
 *
 *  RETURNS:		void
 *
 *********************************************************************/
void LimitBombSpeed( short who )
	{
	/* Limit Bomb Speed */
	if(Player[who].Bomb.SpeedX >= BOMB_MAX_SPEED)
		Player[who].Bomb.SpeedX = BOMB_MAX_SPEED;
	
	if(Player[who].Bomb.SpeedX <= -BOMB_MAX_SPEED)
		Player[who].Bomb.SpeedX = -BOMB_MAX_SPEED;

	if(Player[who].Bomb.SpeedZ >= BOMB_MAX_SPEED)
		Player[who].Bomb.SpeedZ = BOMB_MAX_SPEED;
	
	if(Player[who].Bomb.SpeedZ <= -BOMB_MAX_SPEED)
		Player[who].Bomb.SpeedZ = -BOMB_MAX_SPEED;
	}


/*********************************************************************
 *  FUNCTION:		SetBombSpeed
 *
 *  PARAMETERS:		short who
 *					int SpeedX
 *					int SpeedY
 *					int SpeedZ
 *
 *  DESCRIPTION:	Set the bomb speed
 *
 *  RETURNS:		void
 *
 *********************************************************************/
void  SetBombSpeed( short who, int SpeedX, int SpeedY, int SpeedZ )
	{
	Player[who].Bomb.SpeedX = SpeedX;
	Player[who].Bomb.SpeedY = SpeedY;
	Player[who].Bomb.SpeedZ = SpeedZ;
	}


/*********************************************************************
 *  FUNCTION:		AnimateBomb
 *
 *  PARAMETERS:		short who
 *
 *  DESCRIPTION:	changes the bomb image
 *
 *  RETURNS:		void
 *
 *********************************************************************/
void AnimateBomb( short who )
	{
	/* Step to next Bomb image */
	if(++Player[who].Bomb.Step > 7)
		{
		Player[who].Bomb.Step = 0;
			
		switch(Player[who].Bomb.Style)
			{
			case bombLAUNCH:
				Player[who].Bomb.Style = bombNORMAL;
				Player[who].Bomb.Width = BOMB_WIDTH;
				Player[who].Bomb.Height = BOMB_HEIGHT;
				if(who == PLAYER_ON_TOP)
					Player[who].Bomb.Y += BOMB_OFFSET_Y;
				else
					Player[who].Bomb.Y -= BOMB_OFFSET_Y;
				break;

			case bombEXPLODE:
				Player[who].Bomb.OldX = Player[who].Bomb.ScreenX;
				Player[who].Bomb.OldY = Player[who].Bomb.ScreenY;
				Player[who].Bomb.Active = FALSE;
				break;
			}
		}
	}


/*********************************************************************
 *  FUNCTION: 	   LaunchBomb
 *
 *  PARAMETERS:	   short who
 *
 *  DESCRIPTION:   Starts the bomb
 *
 *  RETURNS:	   void
 *
 *********************************************************************/
void LaunchBomb(short who)
	{
	WORD xspeed, yspeed, zspeed;
	int OffsetX;
	SCREENLOC Loc;

	int volume;

	if((Player[who].Mode != modePLAY) && (Player[who].Mode != modeTARGETPLAY))
		{
		Player[who].Paddle.GunPower = 0;
		return;
		}

	if(Player[who].Bomb.Active)
		return;

	Player[who].Bomb.X = Player[who].Paddle.X;
	Player[who].Bomb.Y = Player[who].Paddle.Y;
	Player[who].Bomb.Z = Player[who].Paddle.Z;

	xspeed = Player[who].Paddle.SpeedX/2;
	zspeed = -BOMB_MAX_SPEED;

	if(who == PLAYER_ON_TOP)
		{
		yspeed = Player[who].Paddle.GunPower*14;
		Player[who].Bomb.IncrY = -10;
		}
	else
		{
		yspeed = -(Player[who].Paddle.GunPower*14);
		Player[who].Bomb.IncrY = 10;
		}

	SetPaddleHover(who, 32);

	SetBombSpeed(who, xspeed, yspeed, zspeed);

	Player[who].Bomb.Active = TRUE;
	Player[who].Bomb.Style  = bombLAUNCH;
	Player[who].Bomb.Width  = PaddleWidthTab[Player[who].Paddle.Size];
	Player[who].Bomb.Height = BOMB_LAUNCH_HEIGHT;
	Player[who].Bomb.Step   = -1;

	Player[who].Bomb.X = Player[who].Paddle.X;
	Player[who].Bomb.Y = Player[who].Paddle.Y;
	Player[who].Bomb.Z = Player[who].Paddle.Z;
	Player[who].Bomb.Scale = 100;

	OffsetX = Player[who].Bomb.Width/2;

	Loc = PlayfieldToScreen(Player[who].Bomb.X, Player[who].Bomb.Y, 
							Player[who].Bomb.Z);
	Loc.X -= OffsetX;

	Player[who].Bomb.ScreenX = Loc.X;
	Player[who].Bomb.ScreenY = Loc.Y;

	volume = Player[who].Bomb.Z*127/10000;
	PlaySound(sndBOMB_LAUNCH, volume,0, 5);

	if(Player[who].Mode == modeTARGETPLAY)
		Player[who].Paddle.Gun--;
	}


/*********************************************************************
 *  FUNCTION: 	   MoveBomb
 *
 *  PARAMETERS:	   short who
 *
 *  DESCRIPTION:   Moves the bomb
 *
 *  RETURNS:	   void
 *
 *********************************************************************/
void MoveBomb(short who)
	{
	int OffsetX;
	int OffsetY;
	SCREENLOC Loc;
	
	if(!Player[who].Bomb.Active)
		return;

	Player[who].Bomb.OldX = Player[who].Bomb.ScreenX;
	Player[who].Bomb.OldY = Player[who].Bomb.ScreenY;
	Player[who].Bomb.OldWidth = Player[who].Bomb.Width;
	Player[who].Bomb.OldHeight = Player[who].Bomb.Height;

	AnimateBomb(who);

	switch (Player[who].Bomb.Style)
		{
		case bombNORMAL:
			LimitBombSpeed(who);

			/* Apply the Bomb movement */
			Player[who].Bomb.X += Player[who].Bomb.SpeedX;
			Player[who].Bomb.Y += Player[who].Bomb.SpeedY;
			Player[who].Bomb.Z += Player[who].Bomb.SpeedZ;

			Player[who].Bomb.SpeedY += Player[who].Bomb.IncrY;
			
			Player[who].Bomb.Scale = GetDepthScale(Player[who].Bomb.Z);

			CheckBombCollision(who);
			break;

		case bombLAUNCH:
			Player[who].Bomb.X = Player[who].Paddle.X;
			Player[who].Bomb.Y = Player[who].Paddle.Y;
			Player[who].Bomb.Z = Player[who].Paddle.Z;
			Player[who].Bomb.Scale = 100;
			break;

		case bombEXPLODE:
			ExplodeBrick(who);
			break;
		}

	OffsetX = (Player[who].Bomb.Width*Player[who].Bomb.Scale)/200;
	OffsetY = (Player[who].Bomb.Height*Player[who].Bomb.Z/TOTAL_RANGE)/2;

	Loc = PlayfieldToScreen(Player[who].Bomb.X, Player[who].Bomb.Y, 
							Player[who].Bomb.Z);

	Player[who].Bomb.ScreenX = Loc.X-OffsetX;
	if(who == PLAYER_ON_TOP)
		Player[who].Bomb.ScreenY = Loc.Y+OffsetY;
	else
		Player[who].Bomb.ScreenY = Loc.Y-OffsetY;
	}
	

/*********************************************************************
 *  FUNCTION: 	   CheckBombCollision
 *
 *  PARAMETERS:	   short who
 *
 *  DESCRIPTION:   Checks to see if bomb has hit brick wall or floor
 *
 *  RETURNS:	   void
 *
 *********************************************************************/
void CheckBombCollision(short who)
	{
	int volume;

	BYTE Explode;
 	UWORD tier, row, col;

	Explode = FALSE;

	if (!Player[who].Bomb.Active)
		return;

	volume = Player[who].Bomb.Z*127/10000;

	if(Player[who].Bomb.X < PLAYFIELD_LEFT)
		{
		/* Bomb hits Left side */
		PlaySound(sndBOMB2WALL, volume, 0, 5);

		Player[who].Bomb.X = PLAYFIELD_LEFT;
		Player[who].Bomb.SpeedX = -Player[who].Bomb.SpeedX; 

		StartRipple(Player[who].Bomb.X, Player[who].Bomb.Y, Player[who].Bomb.Z, 
								rippleLEFT);
		}
	else if(Player[who].Bomb.X > PLAYFIELD_RIGHT-BOMB_W)  
		{
		/* Bomb hits Right side */
		PlaySound(sndBOMB2WALL, volume, 0, 5);

		Player[who].Bomb.X = PLAYFIELD_RIGHT-BOMB_W;
		Player[who].Bomb.SpeedX = -Player[who].Bomb.SpeedX; 

		StartRipple(Player[who].Bomb.X, Player[who].Bomb.Y, Player[who].Bomb.Z, 
								rippleRIGHT);
		}
	else if(Player[who].Bomb.Z < PLAYFIELD_BACK)	  
		{
		/* Bomb hits Back */
		PlaySound(sndBOMB2WALL, volume, 0, 5);
			
		Player[who].Bomb.Z = PLAYFIELD_BACK+(BOMB_W/2);
		Player[who].Bomb.SpeedZ = -Player[who].Bomb.SpeedZ; 

		StartRipple(Player[who].Bomb.X, Player[who].Bomb.Y, Player[who].Bomb.Z, 
								rippleBACK);
		}
	else if(abs(Player[who].Bomb.Y) > BOMB_Y)
		{
		/* Bomb Hits the floor */
		Explode = TRUE;
		}
	else
		{
		tier = (BOMB_Y-abs(Player[who].Bomb.Y)+BRICK_AHEIGHT)/BRICK_AHEIGHT;

		if(tier < BRICK_TIERS)
			{
			row = (Player[who].Bomb.Z+BRICK_ADEPTH-BRICK_ORIGIN_Z-BOMB_H/2)/BRICK_ADEPTH;
			if(row < BRICK_ROWS)
				{
				col = (Player[who].Bomb.X+abs(PLAYFIELD_LEFT))/BRICK_AWIDTH;

				if(Player[who].Brick[tier][col][row].Style != NONE)
					Explode = TRUE;
				}
			}
		}
	
	if(Explode)
		ExplodeBomb(who);
	}


/*********************************************************************
 *  FUNCTION: 	   ExplodeBomb
 *
 *  PARAMETERS:	   short who
 *
 *  DESCRIPTION:   Explodes the bomb
 *
 *  RETURNS:	   void
 *
 *********************************************************************/
void ExplodeBomb( short who )
	{
	int volume;

	volume = Player[who].Bomb.Z*127/10000;
	PlaySound(sndEXPLODE, volume, 0, 7);


	Player[who].Bomb.Style = bombEXPLODE;
	Player[who].Bomb.Width = BOMB_EXPLODE_WIDTH;
	Player[who].Bomb.Height = BOMB_EXPLODE_HEIGHT;

	Player[who].Bomb.Step = 0;
	}


/*********************************************************************
 *  FUNCTION: 	   ExplodeBrick
 *
 *  PARAMETERS:	   short who
 *
 *  DESCRIPTION:   Explodes the bricks
 *
 *  RETURNS:	   void
 *
 *********************************************************************/
void ExplodeBrick(short who)
	{
	BYTE tier, row, col;
	int i, j;
	BYTE Test1, Test2;

	tier = (BOMB_Y-abs(Player[who].Bomb.Y)+BRICK_AHEIGHT)/BRICK_AHEIGHT;
	row = (Player[who].Bomb.Z+BRICK_ADEPTH-BRICK_ORIGIN_Z-BOMB_H/2)/BRICK_ADEPTH;
	col = (Player[who].Bomb.X+abs(PLAYFIELD_LEFT))/BRICK_AWIDTH;

	Test1 = FALSE;
	Test2 = FALSE;

	switch(Player[who].Bomb.Step)
		{
		case 1:
			for(j=2; j>=tier-1; j--)
				{
				if(ClearBrick(who, j, col, row, DECREMENT))
					{
					StartDropBrick(who, j, col-1, row);
					StartDropBrick(who, j, col+1, row);
					Test1 = TRUE;
					for(i=row; i>=MAX(row-2-j, 0); i--)
						{
						DrawBrick(who, j, col, i, FALSE);

						if(col <= CENTER_BRICK)
							DrawBrick(who, j, col-1, i, FALSE);

						if(col>=CENTER_BRICK)
							DrawBrick(who, j, col+1, i, FALSE);
						}
					DrawBomb(who);
					DisplayBricks(who, j, col, row, 1);
					}
				}
			break;

		case 6:
			tier++;
		case 4:
			tier++;
		case 2:
			Test1 = ClearBrick(who, tier, col, row-1, DECREMENT);
			Test2 = ClearBrick(who, tier, col, row+1, DECREMENT);

			if(Test1 || Test2)
				{
				if(Test1)
					{
					StartDropBrick(who, tier, col-1, row-1);
					StartDropBrick(who, tier, col+1, row-1);
					}
				
				if(Test2)
					{
					StartDropBrick(who, tier, col-1, row+1);
					StartDropBrick(who, tier, col+1, row+1);
					}

	   			for(j=tier; j>=0; j--)
					{
					for(i=row+1; i>=MAX(row-3-j, 0); i--)
						{
						DrawBrick(who, j, col, i, FALSE);

						if(col <= CENTER_BRICK)
							DrawBrick(who, j, col-1, i, FALSE);

						if(col>=CENTER_BRICK)
							DrawBrick(who, j, col+1, i, FALSE);
						}
					}
				DrawBomb(who);
				DisplayBricks(who, tier, col, row-1, 1);
				DisplayBricks(who, tier, col, row+1, 1);
				}
			break;

		case 7:
			tier++;
		case 5:
			tier++;
		case 3:
			Test1 = ClearBrick(who, tier, col-1, row, DECREMENT);
			Test2 = ClearBrick(who, tier, col+1, row, DECREMENT);

			if(Test1)
				StartDropBrick(who, tier, col-2, row);

			if(Test2)
				StartDropBrick(who, tier, col+2, row);
			
			if(Test1 || Test2)
				{
				for(j=tier; j>=0; j--)
					{
					for(i=row+1; i>=MAX(row-3-j, 0); i--)
						{
						DrawBrick(who, j, col-1, i, FALSE);
						DrawBrick(who, j, col+1, i, FALSE);

						if((col-1) <= CENTER_BRICK)
							DrawBrick(who, j, col-2, i, FALSE);

						if((col+1) >= CENTER_BRICK)
							DrawBrick(who, j, col+2, i, FALSE);
						}
					}
				DrawBomb(who);
				DisplayBricks(who, tier, col-1, row, 3);
				}
			break;
		}

	switch(Player[who].Mode)
		{
		case modeTARGETSTART:
		case modeTARGETPLAY:
			if(Test1)
				Player[who].Score += 100;

			if(Test2)
				Player[who].Score += 100;
			break;

		default:
			if(Test1)
				Player[who].Score += 5;

			if(Test2)
				Player[who].Score += 5;
			break;
		}

	UpdateScore(who, TRUE);
	}


/*********************************************************************
 *  FUNCTION:		BlankBomb
 *
 *  PARAMETERS:		short who
 *
 *  DESCRIPTION:	Removes the bomb from the screen and set in inactive
 *
 *  RETURNS:		void
 *
 *********************************************************************/
void BlankBomb(short who)
	{
	Player[who].Bomb.OldX = Player[who].Bomb.ScreenX;
	Player[who].Bomb.OldY = Player[who].Bomb.ScreenY;
	EraseBomb(who);
	DisplayBomb(who);
	Player[who].Bomb.Active = FALSE;
	}

	
/*********************************************************************
 *  FUNCTION: 	   EraseBomb
 *				   
 *  PARAMETERS:	   short who
 *
 *  DESCRIPTION:   Removes bomb from playfield and buffer
 *
 *  RETURNS:	   void
 *
 *********************************************************************/
void EraseBomb(short who)
	{
	int EndX;
	int Offset;
	int StepWidth;
	int a1_step;

	if(!Player[who].Bomb.Active)
		return;

	EndX = Player[who].Bomb.OldX+Player[who].Bomb.OldWidth;
	Offset = EndX & 3;
	if(Offset)
		StepWidth = Player[who].Bomb.OldWidth+4-Offset;
	else
		StepWidth = Player[who].Bomb.OldWidth;
	
	if(Player[who].Bomb.OldY >= 100)
		a1_step	= (1<<16) | (-StepWidth & 0xFFFFL);
	else
		a1_step	= (-1<<16) | (-StepWidth & 0xFFFFL);

	Blit[BlitAddIndex].Type		= blitBITMAP;

	Blit[BlitAddIndex].A1_Base	= (long)IMG_PlayfieldBuffer;
	Blit[BlitAddIndex].A1_Flags	= PITCH2 | PIXEL16 | WID320 | XADDPHR | 
										  ZOFFS1;
	Blit[BlitAddIndex].A1_Pixel	= (Player[who].Bomb.OldY<<16) | 
										  (Player[who].Bomb.OldX & 0xFFFFL);
	Blit[BlitAddIndex].A1_Step	= a1_step;
										 
	Blit[BlitAddIndex].A2_Base	= (long)IMG_BrickBuffer;
	Blit[BlitAddIndex].A2_Flags	= Blit[BlitAddIndex].A1_Flags;
	Blit[BlitAddIndex].A2_Pixel	= Blit[BlitAddIndex].A1_Pixel;
	Blit[BlitAddIndex].A2_Step	= Blit[BlitAddIndex].A1_Step;

	Blit[BlitAddIndex].B_Count	= (Player[who].Bomb.OldHeight<<16) | 
								  (Player[who].Bomb.OldWidth & 0xFFFFL);
										 
	Blit[BlitAddIndex].B_Patd	= 0x0L;
	Blit[BlitAddIndex].B_Cmd	= LFU_REPLACE | UPDA1 | UPDA2 | SRCEN | 
								  DSTEN | SRCENZ | DSTENZ | DSTWRZ;
	
	QueBlit();
	}

/*********************************************************************
 *  FUNCTION: 	   DrawBomb
 *
 *  PARAMETERS:	   short who
 *
 *  DESCRIPTION:   Draws the bomb to the buffer
 *
 *  RETURNS:	   void
 *
 *********************************************************************/
void DrawBomb(short who)
	{
	long ScaledWidth;
	long ScaledHeight;
	LONG Increment;
	int Fraction;

	long a1_base;
	int a1_flags;
	int a2_step;

	if(!Player[who].Bomb.Active)
		return;

	ScaledWidth = (Player[who].Bomb.Width*Player[who].Bomb.Scale/100) & 0x0FFFFL;
	ScaledHeight = (Player[who].Bomb.Height*Player[who].Bomb.Scale/100) & 0x0FFFFL;

 	Increment = 0x10000*Player[who].Bomb.Width/ScaledWidth;
	Fraction = Increment & 0x0FFFF;
 	Increment /= 0x010000;

	switch (Player[who].Bomb.Style)
		{
		case bombLAUNCH:
		default:
			a1_base = (long)IMG_Paddle[who].BombLaunch[Player[who].Bomb.Step];
			a1_flags = BOMB_LAUNCH_FLAGS;
			break;

		case bombNORMAL:
			a1_base = (long)IMG_Bomb[Player[who].Bomb.Step];
			a1_flags = BOMB_FLAGS;
			break;

		case bombEXPLODE:
			a1_base = (long)IMG_BombExplode[Player[who].Bomb.Step];
			a1_flags = BOMB_EXPLODE_FLAGS;
			break;
		}

	if(Player[who].Bomb.ScreenY >= 100)
		a2_step	= (1<<16) | (-ScaledWidth & 0xFFFFL);
	else
		a2_step	= (-1<<16) | (-ScaledWidth & 0xFFFFL);

	Blit[BlitAddIndex].Type		= blitSCALED;

	Blit[BlitAddIndex].A1_Base	= a1_base;
	Blit[BlitAddIndex].A1_Flags = a1_flags;
	Blit[BlitAddIndex].A1_Clip	= (Player[who].Bomb.Height<<16) | 
								  (Player[who].Bomb.Width & 0xFFFFL);;
	Blit[BlitAddIndex].A1_Pixel	= 0x0L;
	Blit[BlitAddIndex].A1_Fpixel	= 0x0L;
	Blit[BlitAddIndex].A1_Step	= (Increment<<16) | 
								  (-Player[who].Bomb.Width & 0xFFFFL);
	Blit[BlitAddIndex].A1_Fstep	= Fraction<<16;
	Blit[BlitAddIndex].A1_Inc	= Increment & 0xFFFFL;
	Blit[BlitAddIndex].A1_Finc	= Fraction & 0xFFFFL;
										 
	Blit[BlitAddIndex].A2_Base	= (long)IMG_PlayfieldBuffer;
	Blit[BlitAddIndex].A2_Flags	= PLAYFIELD_BUFFER_FLAGS;
	Blit[BlitAddIndex].A2_Pixel	= (Player[who].Bomb.ScreenY<<16) | 
								  (Player[who].Bomb.ScreenX&0xFFFFL);
	Blit[BlitAddIndex].A2_Step	= a2_step;

	Blit[BlitAddIndex].B_Count	= (ScaledHeight<<16) | ScaledWidth;
										 
	Blit[BlitAddIndex].B_Patd	= 0x0L;
	Blit[BlitAddIndex].B_Srcz1	= Player[who].Bomb.Z;
	Blit[BlitAddIndex].B_Cmd	= LFU_REPLACE | UPDA1 | UPDA1F | UPDA2 | 
								  DSTA2 | SRCEN | DSTEN | DCOMPEN | 
								  DSTENZ | DSTWRZ | ZMODELT | ZMODEEQ | 
								  CLIP_A1;
	
	QueBlit();
	}

/*********************************************************************
 *  FUNCTION: 	   DisplayBomb
 *
 *  PARAMETERS:	   short who
 *
 *  DESCRIPTION:   Draws bomb from buffer to playfield
 *
 *  RETURNS:	   void
 *
 *********************************************************************/
void DisplayBomb(short who)
	{
	SCREENLOC Loc;
	int DeltaX, DeltaY;
	int Width, Height;

	int EndX;
	int Offset;
	int StepWidth;

	int a1_step;

	if(!Player[who].Bomb.Active)
		return;

	DeltaX = abs(Player[who].Bomb.ScreenX-Player[who].Bomb.OldX);
	DeltaY = abs(Player[who].Bomb.ScreenY-Player[who].Bomb.OldY);

	Width = MAX(Player[who].Bomb.Width,Player[who].Bomb.OldWidth)+DeltaX;
	Height = MAX(Player[who].Bomb.Height,Player[who].Bomb.OldHeight)+DeltaY;

	Loc.X = MIN(Player[who].Bomb.ScreenX, Player[who].Bomb.OldX);

	EndX = Loc.X+Width;
	Offset = EndX & 3;
	if(Offset)
		StepWidth = Width+4-Offset;
	else
		StepWidth = Width;


	if(Player[who].Bomb.ScreenY >= 100)
		{
		a1_step = (1<<16) | (-StepWidth & 0xFFFFL);
		Loc.Y = MIN(Player[who].Bomb.ScreenY, Player[who].Bomb.OldY);
		}
	else
		{
		a1_step = (-1<<16) | (-StepWidth & 0xFFFFL);
		Loc.Y = MAX(Player[who].Bomb.ScreenY, Player[who].Bomb.OldY);
		}

	Blit[BlitAddIndex].Type		= blitBITMAP;

	Blit[BlitAddIndex].A1_Base	= (long)IMG_Playfield;
	Blit[BlitAddIndex].A1_Flags	= PITCH1 | PIXEL16 | WID320 | XADDPHR;
	Blit[BlitAddIndex].A1_Pixel	= ((Loc.Y+20)<<16) | (Loc.X & 0xFFFFL);
	Blit[BlitAddIndex].A1_Step 	= a1_step;
										 
	Blit[BlitAddIndex].A2_Base	= (long)IMG_PlayfieldBuffer;
	Blit[BlitAddIndex].A2_Flags	= PITCH2 | PIXEL16 | WID320 | XADDPHR | 
										  ZOFFS1;
	Blit[BlitAddIndex].A2_Pixel	= (Loc.Y<<16) | (Loc.X & 0xFFFFL);
	Blit[BlitAddIndex].A2_Step	= Blit[BlitAddIndex].A1_Step;

	Blit[BlitAddIndex].B_Count	= (Height<<16) | (Width & 0xFFFFL);
										 
	Blit[BlitAddIndex].B_Patd	= 0x0L;
	Blit[BlitAddIndex].B_Cmd	= LFU_REPLACE | UPDA1 | UPDA2 | SRCEN;
	
	QueBlit();
	}
