/*******************************************************************
 *
 *    TITLE:        brick.c
 *
 *    DESCRIPTION:	Routines for handling bricks
 *
 *    AUTHOR:		Mario Perdue & Richard Degler
 *
 *    HISTORY:    
 *
 *
 *           COPYRIGHT 1994, 1995 MP Graphics Systems         
 *      UNATHORIZED REPRODUCTION, ADAPTATION, DISTRIBUTION,
 *      PERFORMANCE OR DISPLAY OF THIS COMPUTER PROGRAM OR   
 *     THE ASSOCIATED AUDIOVISUAL WORK IS STRICTLY PROHIBITED.
 *                        ALL RIGHTS RESERVED.               
 *******************************************************************/

/** include files **/
#include "\jaguar\include\jaguar.h"
#include "brick.h"
#include "jagobj.h"
#include "b2k_game.h"
#include "support.h"
#include "player.h"
#include "ball.h"
#include "option.h"
#include "displays.h"


/** external data **/
extern	WORD			IMG_Playfield[];
extern	WORD			IMG_PlayfieldBuffer[];

extern	WORD			IMG_BrickBuffer[];
extern	WORD			IMG_BrickEraseBuffer[];

extern  WORD			IMG_Brick[BRICK_SIZES][BRICK_COLORS][BRICK_IMAGES][BRICK_HEIGHT][BRICK_WIDTH];
extern  WORD			IMG_BrickClassic[6][20][10];

extern	WORD			IMG_DropBrick[4][1440];

extern int 				ScreenOrgX;
extern int 				ScreenOrgY;

extern char 			MaxScreens;
extern int 				Screen;

extern  DEF_SCREEN		ScreenDef[];
extern  DEF_PLAYER 	 	Player[2];

extern	BYTE			ClassicBrick[10][13];
extern	BYTE			TargetBrick[5][10][13];


extern WORD	BrickScreenXTab[BRICK_ROWS][BRICK_COLS];
extern WORD	BrickScreenYTab[BRICK_TIERS][BRICK_ROWS];

extern WORD	BrickWidthTab[BRICK_ROWS];
extern WORD	BrickHeightTab[BRICK_ROWS];

extern WORD BrickIndexTab[MAX_PLAYERS][BRICK_COLS];
extern WORD BrickSizeTab[BRICK_ROWS];

extern DEF_OPTION		Option;

extern WORD				DropWidthTab[5];
extern WORD				DropOffsetTab[5];

extern BYTE				PlayerOnTop;	

extern	DEF_BLIT_DATA	Blit[];
extern	LONG			BlitAddIndex;
extern	LONG			BlitUseIndex;

DEF_DROPBRICK			DropBrick[MAX_DROPBRICKS];
DEF_DROPBRICK_BUFFER	DropBrickBuffer;

extern long				ticks;

extern WORD	ObjectListFlag;

/** internal functions **/
/*********************************************************************
 *  FUNCTION:		InitBrickArray
 *
 *  PARAMETERS:		void
 *
 *  DESCRIPTION:	Setup the brick array from rom
 *
 *  RETURNS:		short who
 *
 *********************************************************************/
void InitBrickArray(short who)
	{
	int tier, col, row;
	

	for(tier=0; tier<BRICK_TIERS; tier++)
		{	
		for(row=0; row<BRICK_ROWS; row++)
			{
			for(col=0; col<BRICK_COLS; col++)
				{
				Player[who].Brick[tier][col][row].Style = NONE;
				Player[who].Brick[tier][col][row].Index = BrickIndexTab[who][col];

				switch(Option.GameMode)
					{
					case game2000:
						Player[who].Brick[tier][col][row].Width = BrickWidthTab[row];
						Player[who].Brick[tier][col][row].Height = BrickHeightTab[row];

						Player[who].Brick[tier][col][row].Size = BrickSizeTab[row];
						Player[who].Brick[tier][col][row].Screen.X = 
															BrickScreenXTab[row][col]-4;
						if(who == PLAYER_ON_TOP)
							{
							Player[who].Brick[tier][col][row].Screen.Y = 
									PLAYFIELD_BUFFER_HEIGHT-BrickScreenYTab[tier][row];
							}
						else
							{
							Player[who].Brick[tier][col][row].Screen.Y = 
											BrickScreenYTab[tier][row];
							}
						break;

					case gameCLASSIC:
						Player[who].Brick[tier][col][row].Width = 20;
						Player[who].Brick[tier][col][row].Height = 10;

						Player[who].Brick[tier][col][row].Size = 0;
						Player[who].Brick[tier][col][row].Screen.X = col*20+30;
						Player[who].Brick[tier][col][row].Screen.Y = row*10+49;
						break;
					}
				Player[who].Brick[tier][col][row].Depth = 
										BRICK_ORIGIN_Z+BRICK_ADEPTH*row+2*tier;
				if(col<CENTER_BRICK)
					Player[who].Brick[tier][col][row].Depth += col;
				else
					Player[who].Brick[tier][col][row].Depth += (BRICK_COLS-col);
				}
			}
		}
	}


/*********************************************************************
 *  FUNCTION:		LoadBricks
 *
 *  PARAMETERS:		void
 *
 *  DESCRIPTION:	Loads both players brick arrays from rom
 *
 *  RETURNS:		void
 *
 *********************************************************************/
void LoadBricks()
	{
	int tier, col, row;
	short who;
	
	ClearBrickBuffer();

	ClearPlayfieldBuffer();
	ClearPlayfield();
	
	for(who=0; who<Option.NumPlayers; who++)
		{
		Player[who].Ball[1].Style = ballNORMAL;

		InitBrickArray(who);

		Player[who].BrickCount = 0;
		UpdateBrickCount(who);

		for(tier=0; tier<BRICK_TIERS; tier++)
			{	
			for(row=0; row<BRICK_ROWS; row++)
				{
				for(col=0; col<BRICK_COLS; col++)
					{
					if(Player[who].Paddle.Active)
						{
						if(Option.GameMode == gameCLASSIC)
							{
							if(tier == 0)
								{
								Player[who].Brick[tier][col][row].Style = 
									ClassicBrick[row][col];
								}
							else
								{
								Player[who].Brick[tier][col][row].Style = NONE;
								}
							}
						else if((Player[who].Mode == modeTARGETPLAY) && (tier == 0))
							Player[who].Brick[tier][col][row].Style = 
									TargetBrick[MAX(0,Screen/10-1)][row][col];
						else if(Player[who].Mode == modeSHOOT)
							Player[who].Brick[tier][col][row].Style = 
									ScreenDef[Screen].Brick[tier][row][col];
						}
					
					if( (Player[who].Brick[tier][col][row].Style > SILVER9) &&
						(Player[who].Brick[tier][col][row].Style != NONE) )
						Player[who].Brick[tier][col][row].Style = SILVER9;

					if(Player[who].Brick[tier][col][row].Style != NONE && 
					  Player[who].Brick[tier][col][row].Style != GOLD)
						Player[who].BrickCount++;
					}
				}
			}
		}
	PlaceBrickFrames();
	PlaceBricks();                                          
	}


/*********************************************************************
 *  FUNCTION:		PlaceBrickFrames
 *
 *  PARAMETERS:		void
 *
 *  DESCRIPTION:	Puts the bricks in the buffers and on the playfield
 *
 *  RETURNS:		void
 *
 *********************************************************************/
void PlaceBrickFrames()
	{
	int tier, col, row;
	int volume, pitch;
	short who;
	int Time;

	if(Option.GameMode == game2000)
		{
		for(row=0; row<BRICK_ROWS; row++)
			{
			Time = ticks;

			for(tier=0; tier<BRICK_TIERS; tier++)
				{
				for(col=0; col<BRICK_COLS; col++)
					{
					for(who=PLAYER_RED; who<MIN(Option.NumPlayers,2); who++)
						{
						if (Player[who].Brick[tier][col][row].Style != NONE)
							{
							DrawBrick(who, tier, col, row, TRUE);
							DisplayBricks(who, tier, col, row, 1);
							}
						}
					}

				if(Option.NumPlayers > 1)
					{
					Player[PLAYER_GREEN].Joypad = ReadJoyStick(PLAYER_GREEN);
					MovePaddle(PLAYER_GREEN);
					}
							
				Player[PLAYER_RED].Joypad = ReadJoyStick(PLAYER_RED);
				MovePaddle(PLAYER_RED);
				ObjectListFlag = TRUE;
				}

			pitch = 10;
			volume = row*127/10;
			PlaySound(sndBRICK1, volume, pitch, 6);
			
			while(abs(ticks-Time) < 15);
			}

		UpdateBrickCount(PLAYER_RED);

		if(Option.NumPlayers > 1)
			UpdateBrickCount(PLAYER_GREEN);
		}
	}

				
/*********************************************************************
 *  FUNCTION:		PlaceBricks
 *
 *  PARAMETERS:		void
 *
 *  DESCRIPTION:	Puts the bricks in the buffers and on the playfield
 *
 *  RETURNS:		void
 *
 *********************************************************************/
void PlaceBricks()
	{
	int tier, col, row;
	int volume;
	int pitch;
	int i;
	int index;
	int array[BRICK_TIERS*BRICK_COLS*BRICK_ROWS];
	short who;
	int Time;

	switch(Option.GameMode)
		{
		case game2000:
			for(i=0; i<BRICK_TIERS*BRICK_COLS*BRICK_ROWS; i++)
				array[i] = i;

			for(i=BRICK_TIERS*BRICK_COLS*BRICK_ROWS; i>0; i--)
				{
				index = Random(i);

				tier = array[index]/(BRICK_COLS*BRICK_ROWS);
				col  = (array[index]/BRICK_ROWS)%BRICK_COLS;
				row  = array[index]%BRICK_ROWS;

				array[index] = array[i-1];

				for(who=PLAYER_RED; who<MIN(Option.NumPlayers,2); who++)
					{
					if(Player[who].Brick[tier][col][row].Style != NONE)
						{
						Time = ticks;

						if(who == PLAYER_ON_BOTTOM)
							{
							pitch = Player[who].Brick[tier][col][row].Style;
							volume = row*127/10;
							if(i%2)
								PlaySound(sndBRICK1, volume, pitch, 6);
							}

						DrawBrick(who, tier, col, row, FALSE);
						DisplayBricks(who, tier, col, row, 1);
				
						while(abs(ticks-Time) < 25);
						}
					}

				if(i%5 == 0)
					{
					if(Option.NumPlayers > 1)
						{
						Player[PLAYER_GREEN].Joypad = ReadJoyStick(PLAYER_GREEN);
						MovePaddle(PLAYER_GREEN);
						}
								
					Player[PLAYER_RED].Joypad = ReadJoyStick(PLAYER_RED);
					MovePaddle(PLAYER_RED);
					ObjectListFlag = TRUE;
					}
				}
			break;

		case gameCLASSIC:
			for(row=0; row<BRICK_ROWS; row++)
				{
				for(col=0; col<BRICK_COLS; col++)
					DrawBrick(PLAYER_RED, 0, col, row, FALSE);

				DisplayBricks(PLAYER_RED, 0, 0, row, 13);
				}
			break;
		}

	}


	
	
/*********************************************************************
 *  FUNCTION:		RestoreBrick
 *
 *  PARAMETERS:		short who, int tier, int col, int row
 *
 *  DESCRIPTION:	Restores the specified brick
 *
 *  RETURNS:		void
 *
 *********************************************************************/
 void RestoreBrick(short who, int tier, int col, int row)
	{
	int pitch;
	int volume;

	/* Abort if invalid row, column or tier */
	if((col >= BRICK_COLS) || (row >= BRICK_ROWS) || (tier >= BRICK_TIERS) ||
		(col < 0) || (row < 0) || (tier < 0))
		return;

	if((Player[who].Brick[tier][col][row].Style == NONE) ||
		(Player[who].Brick[tier][col][row].Style == GOLD))
		{
		if((ScreenDef[Screen].Brick[tier][row][col] != GOLD) &&
		   (ScreenDef[Screen].Brick[tier][row][col] != NONE))
		   	{	
			Player[who].BrickCount++;
			UpdateBrickCount(who);
			}
		}

	if((ScreenDef[Screen].Brick[tier][row][col] == NONE) &&
		(Player[who].Brick[tier][col][row].Style != NONE))
		{
		ClearBrick(who, tier, col, row, DECREMENT);
		}
	else
		{
		Player[who].Brick[tier][col][row].Style = 
					ScreenDef[Screen].Brick[tier][row][col];

		DrawBrick(who, tier, col, row, FALSE);

		if(tier == 0)
			{
			pitch = Player[who].Brick[tier][col][row].Style;
			volume = row*127/10;
			PlaySound(sndBRICK1, volume, pitch, 6);
			}
		}

	DisplayBricks(who, tier, col, row, 1);
	}


/*********************************************************************
 *  FUNCTION:		EraseBrick
 *
 *  PARAMETERS:		short who, int tier, int col, int row
 *
 *  DESCRIPTION:	Erases the specified brick
 *
 *  RETURNS:		void
 *
 *********************************************************************/
 void EraseBrick(short who, int tier, int col, int row)
	{
	int i;
	int pitch;
	int volume;
	int t;

	if(Player[who].Brick[tier][col][row].Style == NONE)
		return;
				
	/* Abort if invalid row or column */
	if((col >= BRICK_COLS) || (row >= BRICK_ROWS) || (tier >= BRICK_TIERS) ||
		(col < 0) || (row < 0) || (tier < 0))
		return;

	pitch = Player[who].Brick[tier][col][row].Style;
	volume = row*127/10;
	
	if(Player[who].Brick[tier][col][row].Style == GOLD)
		{
		PlaySound(sndBRICK2, volume, pitch, 6);
		return;
		}
	else if(Player[who].Brick[tier][col][row].Style > SILVER1)
		{
		PlaySound(sndBRICK2, volume, pitch, 6);
		Player[who].Brick[tier][col][row].Style--;
	  	return;
		}
		
	PlaySound(sndBRICK1, volume, pitch, 6);

	ClearBrick(who, tier, col, row, DECREMENT);

	StartDropBrick(who, tier+1, col, row);

	for(t=tier; t>=0; t--)
		{
		for(i=row; i>=MAX(row-4, 0); i--)
			{
			DrawBrick(who, t, col, i, FALSE);

			if(col <= CENTER_BRICK)
				DrawBrick(who, t, col-1, i, FALSE);

			if(col>=CENTER_BRICK)
				DrawBrick(who, t, col+1, i, FALSE);
			}
		}

	DisplayBricks(who, tier, col, row, 1);
	}



/*********************************************************************
 *  FUNCTION:		BlankBrick
 *
 *  PARAMETERS:		short who, int tier, int col, int row
 *
 *  DESCRIPTION:	Erases the specified brick
 *
 *  RETURNS:		void
 *
 *********************************************************************/
 void BlankBrick(short who, int tier, int col, int row)
	{
	int i;
	int t;

	if(Player[who].Brick[tier][col][row].Style == NONE)
		return;
				
	/* Abort if invalid row or column */
	if((col >= BRICK_COLS) || (row >= BRICK_ROWS) || (tier >= BRICK_TIERS) ||
		(col < 0) || (row < 0) || (tier < 0))
		return;

	ClearBrick(who, tier, col, row, NO_DECREMENT);

	for(t=tier; t>=0; t--)
		{
		for(i=row; i>=MAX(row-4, 0); i--)
			{
			DrawBrick(who, t, col, i, FALSE);

			if(col <= CENTER_BRICK)
				DrawBrick(who, t, col-1, i, FALSE);

			if(col>=CENTER_BRICK)
				DrawBrick(who, t, col+1, i, FALSE);
			}
		}

	DisplayBricks(who, tier, col, row, 1);
	}



/*********************************************************************
 *  FUNCTION:		ClearBrick
 *
 *  PARAMETERS:		short who, int tier, int col, int row
 *
 *  DESCRIPTION:	Clears the specified brick from the brick buffer
 *
 *  RETURNS:		void
 *
 *********************************************************************/
BYTE ClearBrick(short who, int tier, int col, int row, char DecFlag)
	{
	int a1_step;
	int a2_step;

	if(Player[who].Brick[tier][col][row].Style == NONE)
		return(FALSE);

	/* Abort if invalid row or column */
	if((col >= BRICK_COLS) || (row >= BRICK_ROWS) || (tier >= BRICK_TIERS) ||
		(col < 0) || (row < 0) || (tier < 0))
		return(FALSE);

	if(DecFlag)
		{
		if((Player[who].Brick[tier][col][row].Style != GOLD) || (tier > 0) )
			{
			Player[who].BrickCount--;
			UpdateBrickCount(who);
			}
		}

	Player[who].Brick[tier][col][row].Style = NONE;

	/* Clear the Brick Erase Buffer */
	Blit[BlitAddIndex].Type		= blitPATTERN;

	Blit[BlitAddIndex].A1_Base	= (long)&IMG_BrickEraseBuffer;
	Blit[BlitAddIndex].A1_Flags	= PITCH2 | PIXEL16 | WID20 | XADDPHR | ZOFFS1;
	Blit[BlitAddIndex].A1_Pixel	= 0x0L;
	Blit[BlitAddIndex].A1_Step	= (1<<16L)|(-BRICK_WIDTH & 0x0FFFFL);
										 
	Blit[BlitAddIndex].B_Count	= (BRICK_HEIGHT<<16L) | 
										  (BRICK_WIDTH & 0x0FFFFL);
										 
	Blit[BlitAddIndex].B_Patd	= 0x0L;
	Blit[BlitAddIndex].B_Srcz1	= 0x0L;
	Blit[BlitAddIndex].B_Cmd	= PATDSEL | UPDA1 | DSTWRZ;
	
	QueBlit();

	if(who == PLAYER_ON_TOP)
		a2_step	= (-1<<16L)|(-Player[who].Brick[tier][col][row].Width & 0x0FFFFL);
	else
		a2_step	= (1<<16L)|(-Player[who].Brick[tier][col][row].Width & 0x0FFFFL);

	/* Blit the brick area to the Brick Erase Buffer */
	Blit[BlitAddIndex].Type		= blitBITMAP;

	Blit[BlitAddIndex].A1_Base	= (long)&IMG_BrickEraseBuffer;
	Blit[BlitAddIndex].A1_Flags	= BRICK_ERASE_BUFFER_FLAGS;
	Blit[BlitAddIndex].A1_Pixel	= 0x0L;
	Blit[BlitAddIndex].A1_Step	= (1<<16L)|(-Player[who].Brick[tier][col][row].Width&0x0FFFFL);
										 
	Blit[BlitAddIndex].A2_Base	= (long)&IMG_BrickBuffer;
	Blit[BlitAddIndex].A2_Flags	= BRICK_BUFFER_FLAGS;
	Blit[BlitAddIndex].A2_Pixel	= (Player[who].Brick[tier][col][row].Screen.Y<<16L) |
								  (Player[who].Brick[tier][col][row].Screen.X & 0xFFFFL);
	Blit[BlitAddIndex].A2_Step	= a2_step;

	Blit[BlitAddIndex].B_Count	= (Player[who].Brick[tier][col][row].Height<<16L) | 
								  (Player[who].Brick[tier][col][row].Width & 0x0FFFFL);
										 
	Blit[BlitAddIndex].B_Patd	= 0x0L;
	Blit[BlitAddIndex].B_Dstz	= Player[who].Brick[tier][col][row].Depth;
	Blit[BlitAddIndex].B_Cmd	= LFU_REPLACE | UPDA1 | UPDA2 | SRCEN | 
								  DSTEN | SRCENZ | DSTWRZ | ZMODEEQ;
	
	QueBlit();


	/* Blit the Brick Erase Buffer back to the brick area */
	if(who == PLAYER_ON_TOP)
		a1_step	= (-1<<16L)|(-Player[who].Brick[tier][col][row].Width & 0x0FFFFL);
	else
		a1_step	= (1<<16L)|(-Player[who].Brick[tier][col][row].Width & 0x0FFFFL);

	Blit[BlitAddIndex].Type		= blitBITMAP;

	Blit[BlitAddIndex].A1_Base	= (long)&IMG_BrickBuffer;
	Blit[BlitAddIndex].A1_Flags	= BRICK_BUFFER_FLAGS;
	Blit[BlitAddIndex].A1_Pixel	= (Player[who].Brick[tier][col][row].Screen.Y<<16L) |
								  (Player[who].Brick[tier][col][row].Screen.X & 0xFFFFL);
	Blit[BlitAddIndex].A1_Step	= a1_step;
										 
	Blit[BlitAddIndex].A2_Base	= (long)&IMG_BrickEraseBuffer;
	Blit[BlitAddIndex].A2_Flags	= BRICK_ERASE_BUFFER_FLAGS;
	Blit[BlitAddIndex].A2_Pixel	= 0x0L;
	Blit[BlitAddIndex].A2_Step	= (1<<16L) | 
								  (-Player[who].Brick[tier][col][row].Width & 0x0FFFFL);

	Blit[BlitAddIndex].B_Count	= (Player[who].Brick[tier][col][row].Height<<16L) | 
								  (Player[who].Brick[tier][col][row].Width & 0x0FFFFL);
										 
	Blit[BlitAddIndex].B_Patd	= 0x0L;
	Blit[BlitAddIndex].B_Cmd	= LFU_REPLACE | UPDA1 | UPDA2 | SRCEN | 
								  DSTEN | SRCENZ | DSTWRZ;
	
	QueBlit();

	return(TRUE);
	}

/*********************************************************************
 *  FUNCTION:		DrawBrick
 *
 *  PARAMETERS:		short who, int tier, int col, int row, char frame
 *
 *  DESCRIPTION:	Draws the specified brick in the brick buffer
 *
 *  RETURNS:		void
 *
 *********************************************************************/
void DrawBrick(short who, int tier, int col, int row, char frame)
	{
	int BrickStyle;
	int BrickSize;
	int BrickIndex;

	int a1_step;

	/* Abort if invalid row or column */
	if((col >= BRICK_COLS) || (row >= BRICK_ROWS) || (tier >= BRICK_TIERS) ||
		(col < 0) || (row < 0) || (tier < 0))
		return;

	if(Player[who].Brick[tier][col][row].Style > SILVER9)
		return;

	if(frame) 
		BrickStyle = FRAME;
	else
		BrickStyle = MIN(Player[who].Brick[tier][col][row].Style, SILVER1);

	BrickIndex   = Player[who].Brick[tier][col][row].Index;
	BrickSize    = Player[who].Brick[tier][col][row].Size;

	if(who == PLAYER_ON_TOP)
		a1_step	= (-1<<16L) | (-BRICK_WIDTH & 0x0FFFFL);
	else
		a1_step	= (1<<16L) | (-BRICK_WIDTH & 0x0FFFFL);

	Blit[BlitAddIndex].Type		= blitBITMAP;

	Blit[BlitAddIndex].A1_Base	= (long)&IMG_BrickBuffer;
	Blit[BlitAddIndex].A1_Flags	= BRICK_BUFFER_FLAGS;
	Blit[BlitAddIndex].A1_Pixel	= (Player[who].Brick[tier][col][row].Screen.Y<<16L) |
		  						  (Player[who].Brick[tier][col][row].Screen.X & 0x0FFFFL);
	Blit[BlitAddIndex].A1_Step	= a1_step;
									
	if(Option.GameMode == gameCLASSIC)
		Blit[BlitAddIndex].A2_Base	= (long)&IMG_BrickClassic[BrickStyle];
	else	 
		Blit[BlitAddIndex].A2_Base	= (long)&IMG_Brick[BrickSize][BrickStyle][BrickIndex];
	Blit[BlitAddIndex].A2_Flags	= BRICK_IMAGE_FLAGS;
	Blit[BlitAddIndex].A2_Pixel	= 0x0L;
	Blit[BlitAddIndex].A2_Step	= (1<<16L) | (-BRICK_WIDTH & 0x0FFFFL);

	Blit[BlitAddIndex].B_Count	= (BRICK_HEIGHT<<16L) | (BRICK_WIDTH & 0x0FFFFL);
										 
	Blit[BlitAddIndex].B_Patd	= 0x0L;
	Blit[BlitAddIndex].B_Srcz1	= Player[who].Brick[tier][col][row].Depth;
	Blit[BlitAddIndex].B_Cmd	= LFU_REPLACE | UPDA1 | UPDA2 | SRCEN | 
								  DSTEN | DCOMPEN | DSTENZ | DSTWRZ | 
								  ZMODELT;
	
	QueBlit();
	}

/*********************************************************************
 *  FUNCTION:		DisplayBricks
 *
 *  PARAMETERS:		short who, int tier, int col, int row, int cnt
 *
 *  DESCRIPTION:	Displays bricks from the buffer
 *
 *  RETURNS:		void
 *
 *********************************************************************/
void DisplayBricks(short who, int tier, int col, int row, int cnt)
	{
	ULONG step;
	ULONG count;
	ULONG Width;
	
	int EndX;
	int Offset;

	/* Abort if invalid row or column */
	if((col >= BRICK_COLS) || (row >= BRICK_ROWS) || (tier >= BRICK_TIERS) ||
		(col < 0) || (row < 0) || (tier < 0))
		return;

	if(col+cnt > BRICK_COLS)
		cnt = BRICK_COLS-col;

	Width = Player[who].Brick[tier][col][row].Width*cnt;

	count  = (Player[who].Brick[tier][col][row].Height<<16L)|(Width & 0xFFFFL);

	EndX = Player[who].Brick[tier][col][row].Screen.X+Width;

	Offset = EndX & 3;
	if(Offset)
		Width += (4-Offset);
	
	if(who == PLAYER_ON_TOP)
		step	= (-1<<16L) | (-Width & 0xFFFFL);
	else
		step	= (1<<16L) | (-Width & 0xFFFFL);

	/* Blit from Brick buffer to Playfield buffer */
	Blit[BlitAddIndex].Type		= blitBITMAP;

	Blit[BlitAddIndex].A1_Base	= (long)IMG_PlayfieldBuffer;
	Blit[BlitAddIndex].A1_Flags	= PITCH2 | PIXEL16 | WID320 | XADDPHR | 
										  ZOFFS1;
	Blit[BlitAddIndex].A1_Pixel	= (Player[who].Brick[tier][col][row].Screen.Y<<16L) |
								  (Player[who].Brick[tier][col][row].Screen.X & 0xFFFFL);
	Blit[BlitAddIndex].A1_Step	= step;
										 
	Blit[BlitAddIndex].A2_Base	= (long)IMG_BrickBuffer;
	Blit[BlitAddIndex].A2_Flags	= Blit[BlitAddIndex].A1_Flags;
	Blit[BlitAddIndex].A2_Pixel	= Blit[BlitAddIndex].A1_Pixel;
	Blit[BlitAddIndex].A2_Step	= Blit[BlitAddIndex].A1_Step;

	Blit[BlitAddIndex].B_Count	= count;
										 
	Blit[BlitAddIndex].B_Patd	= 0x0L;
	Blit[BlitAddIndex].B_Cmd	= LFU_REPLACE | UPDA1 | UPDA2 | SRCEN | 
								  SRCENZ | DSTWRZ;
	
	QueBlit();


	/* Blit from Playfield Buffer to Playfield */
	Blit[BlitAddIndex].Type		= blitBITMAP;

	Blit[BlitAddIndex].A1_Base	= (long)IMG_Playfield;
	Blit[BlitAddIndex].A1_Flags	= PITCH1 | PIXEL16 | WID320 | XADDPHR;
	Blit[BlitAddIndex].A1_Pixel	= ((Player[who].Brick[tier][col][row].Screen.Y+20)<<16L) |
								  (Player[who].Brick[tier][col][row].Screen.X & 0xFFFFL);
	Blit[BlitAddIndex].A1_Step	= step;
										 
	Blit[BlitAddIndex].A2_Base	= (long)IMG_PlayfieldBuffer;
	Blit[BlitAddIndex].A2_Flags	= PITCH2 | PIXEL16 | WID320 | XADDPHR | ZOFFS1;
	Blit[BlitAddIndex].A2_Pixel	= (Player[who].Brick[tier][col][row].Screen.Y<<16L) |
							      (Player[who].Brick[tier][col][row].Screen.X & 0xFFFFL);
	Blit[BlitAddIndex].A2_Step	= Blit[BlitAddIndex].A1_Step;

	Blit[BlitAddIndex].B_Count	= count;
										 
	Blit[BlitAddIndex].B_Patd	= 0x0L;
	Blit[BlitAddIndex].B_Cmd	= LFU_REPLACE | UPDA1 | UPDA2 | SRCEN;
	
	QueBlit();
	}


/*********************************************************************
 *  FUNCTION:		ClearBrickBuffer
 *
 *  PARAMETERS:		void
 *
 *  DESCRIPTION:	Clears the entire brick buffer
 *
 *  RETURNS:		void
 *
 *********************************************************************/
void ClearBrickBuffer(void)
	{
	Blit[BlitAddIndex].Type		= blitPATTERN;

	Blit[BlitAddIndex].A1_Base	= (long)IMG_BrickBuffer;
	Blit[BlitAddIndex].A1_Flags	= BRICK_BUFFER_FLAGS;
	Blit[BlitAddIndex].A1_Pixel	= 0x0L;
	Blit[BlitAddIndex].A1_Step	= (1<<16L) | (-BRICK_BUFFER_WIDTH & 0x0FFFFL);
										 
	Blit[BlitAddIndex].B_Count	= (BRICK_BUFFER_HEIGHT<<16L) | 
								  (BRICK_BUFFER_WIDTH & 0x0FFFFL);
										 
	Blit[BlitAddIndex].B_Patd	= 0x0L;
	Blit[BlitAddIndex].B_Srcz1	= 0x0L;
	Blit[BlitAddIndex].B_Cmd	= PATDSEL | UPDA1 | DSTWRZ;
	
	QueBlit();
	}


/*********************************************************************
 *  FUNCTION: 	   DoBrickPenalty
 *
 *  PARAMETERS:	   short who
 *
 *  DESCRIPTION:   
 *
 *  RETURNS:	   void
 *
 *********************************************************************/
void DoBrickPenalty( short who )
	{
	int tier, col, row;

	for(tier=BRICK_TIERS-1; tier>=0; tier--)
		{
		for (row=BRICK_ROWS-1; row>=0; row--)
			{
			for (col=0; col<BRICK_COLS; col++)
			    {
				if(Player[who].Brick[tier][col][row].Style != NONE)
					{
					Player[who].Brick[tier][col][row].Style = MIN(SILVER1,
								Player[who].Brick[tier][col][row].Style);
					EraseBrick(who, tier, col, row);
					Player[who].Score -= 5;
					UpdateScore(who, TRUE);

					Delay(25);
					}
			    }
			}
		}
	}




/*********************************************************************
 *  FUNCTION: 	   StartDropBrick
 *
 *  PARAMETERS:	   short who, char tier, char col, char row
 *
 *  DESCRIPTION:   
 *
 *  RETURNS:	   void
 *
 *********************************************************************/
void StartDropBrick( short who, short tier, short col, short row )
	{
	if((tier <1) || (tier >= BRICK_TIERS))
		return;

	if((row < 0) || (row >= BRICK_ROWS))
		return;

	if((col < 0) || (col >= BRICK_COLS))
		return;

	if(Player[who].Brick[tier][col][row].Style >= DROP)
		return;

	if(Player[who].Brick[tier-1][col][row].Style <= SILVER9)
		return;


	DropBrickBuffer.Data[DropBrickBuffer.AddIndex].Who = who;
	DropBrickBuffer.Data[DropBrickBuffer.AddIndex].Tier = tier;
	DropBrickBuffer.Data[DropBrickBuffer.AddIndex].Col = col;
	DropBrickBuffer.Data[DropBrickBuffer.AddIndex].Row = row;
	DropBrickBuffer.Data[DropBrickBuffer.AddIndex].OldStyle = 
								Player[who].Brick[tier][col][row].Style;
	DropBrickBuffer.Data[DropBrickBuffer.AddIndex].Depth = 
								Player[who].Brick[tier][col][row].Depth;
	DropBrickBuffer.Data[DropBrickBuffer.AddIndex].Size = row/2;
	DropBrickBuffer.Data[DropBrickBuffer.AddIndex].Screen.X = 
								Player[who].Brick[tier][col][row].Screen.X;
	DropBrickBuffer.Data[DropBrickBuffer.AddIndex].Screen.Y = 
								Player[who].Brick[tier][col][row].Screen.Y;
	DropBrickBuffer.Data[DropBrickBuffer.AddIndex].Step = 0;
	DropBrickBuffer.Data[DropBrickBuffer.AddIndex].Active = TRUE;

	if(++DropBrickBuffer.AddIndex > 127)
		DropBrickBuffer.AddIndex = 0;
	}


/*********************************************************************
 *  FUNCTION: 	   AnimateDropBricks
 *
 *  PARAMETERS:	   void
 *
 *  DESCRIPTION:   
 *
 *  RETURNS:	   void
 *
 *********************************************************************/
void AnimateDropBricks()
	{
	int i;
	int w, t, r, c;

	for(i=0; i<MAX_DROPBRICKS; i++)
		{
		if(DropBrick[i].Active)
			{
			w = DropBrick[i].Who;
			t = DropBrick[i].Tier;
			r = DropBrick[i].Row;
			c = DropBrick[i].Col;

			if(Player[w].Brick[t-1][c][r].Style != DROP)
				{
				EraseDropBrick(i);
				DisplayDropBrick(i);
				DropBrick[i].Active = FALSE;
				}

			else if(++DropBrick[i].Step >= DROP_STEPS)
				{
				EraseDropBrick(i);
				DisplayDropBrick(i);
				DropBrick[i].Active = FALSE;

				Player[w].Brick[t-1][c][r].Style = DropBrick[i].OldStyle;

				DrawBrick(w, t-1, c, r, FALSE);
				DisplayBricks(w, t-1, c, r, 1);

				if(t == 1)
					{
					StartDropBrick(w, 2, c, r);
					StartDropBrick(w, t, c-1, r);
					StartDropBrick(w, t, c+1, r);
					}
				else 
					{
					StartDropBrick(w, t, c-1, r);
					StartDropBrick(w, t, c+1, r);

					if(Player[w].Brick[0][c][r].Style == NONE)
						StartDropBrick(w, 1, c, r);
					}
				}
			}
		else if(DropBrickBuffer.AddIndex != DropBrickBuffer.UseIndex)
			{
			DropBrick[i] = DropBrickBuffer.Data[DropBrickBuffer.UseIndex];

			BlankBrick(DropBrick[i].Who, DropBrick[i].Tier, DropBrick[i].Col, 
						DropBrick[i].Row);

			/* override the NONE setting for the bottom brick */
			Player[DropBrick[i].Who].Brick[DropBrick[i].Tier-1][DropBrick[i].Col][DropBrick[i].Row].Style = DROP;
		
			if(++DropBrickBuffer.UseIndex > 127)
				DropBrickBuffer.UseIndex = 0;
			}
		}
	}

/*********************************************************************
 *  FUNCTION: 	   EraseDropBrick
 *
 *  PARAMETERS:	   short which
 *
 *  DESCRIPTION:   
 *
 *  RETURNS:	   void
 *
 *********************************************************************/
void EraseDropBrick(short which)
	{
	ULONG step;
	int EndX;
	int Offset;
	int StepWidth;

	if(!DropBrick[which].Active)
		return;

	EndX = DropBrick[which].Screen.X+DropWidthTab[DropBrick[which].Size];
	Offset = EndX & 3;
	if(Offset)
		StepWidth = DropWidthTab[DropBrick[which].Size]+4-Offset;
	else
		StepWidth = DropWidthTab[DropBrick[which].Size];

	if(DropBrick[which].Screen.Y >= 100)
		step = (1<<16) | (-StepWidth & 0xFFFFL);
	else
		step = (-1<<16) | (-StepWidth & 0xFFFFL);
		
	Blit[BlitAddIndex].Type		= blitBITMAP;

	Blit[BlitAddIndex].A1_Base	= (long)IMG_PlayfieldBuffer;
	Blit[BlitAddIndex].A1_Flags	= PITCH2 | PIXEL16 | WID320 | XADDPHR | 
										  ZOFFS1;
	Blit[BlitAddIndex].A1_Pixel	= (DropBrick[which].Screen.Y<<16) | 
										  (DropBrick[which].Screen.X & 0xFFFFL);
	Blit[BlitAddIndex].A1_Step	= step;
										 
	Blit[BlitAddIndex].A2_Base	= (long)IMG_BrickBuffer;
	Blit[BlitAddIndex].A2_Flags	= Blit[BlitAddIndex].A1_Flags;
	Blit[BlitAddIndex].A2_Pixel	= Blit[BlitAddIndex].A1_Pixel;
	Blit[BlitAddIndex].A2_Step	= Blit[BlitAddIndex].A1_Step;

	Blit[BlitAddIndex].B_Count	= (18<<16) | 
								(DropWidthTab[DropBrick[which].Size] & 0xFFFFL);
										 
	Blit[BlitAddIndex].B_Patd	= 0x0L;
	Blit[BlitAddIndex].B_Cmd	= LFU_REPLACE | UPDA1 | UPDA2 | SRCEN | 
								  DSTEN | SRCENZ | DSTENZ | DSTWRZ;
	
	QueBlit();
	}


/*********************************************************************
 *  FUNCTION: 	   DrawDropBrick
 *
 *  PARAMETERS:	   short which
 *
 *  DESCRIPTION:   
 *
 *  RETURNS:	   void
 *
 *********************************************************************/
void DrawDropBrick(short which)
	{
	int a1_step;

	if(!DropBrick[which].Active)
		return;

	if(DropBrick[which].Screen.Y >= 100)
		a1_step	= (1<<16L) | (-DropWidthTab[DropBrick[which].Size] & 0x0FFFFL);
	else
		a1_step	= (-1<<16L) | (-DropWidthTab[DropBrick[which].Size] & 0x0FFFFL);
	
	Blit[BlitAddIndex].Type		= blitBITMAP;

	Blit[BlitAddIndex].A1_Base	= (long)&IMG_PlayfieldBuffer;
	Blit[BlitAddIndex].A1_Flags	= PLAYFIELD_BUFFER_FLAGS;
	Blit[BlitAddIndex].A1_Pixel	= (DropBrick[which].Screen.Y<<16L) | 
										(DropBrick[which].Screen.X & 0x0FFFFL);
	Blit[BlitAddIndex].A1_Step	= a1_step;
										 
	Blit[BlitAddIndex].A2_Base	= (long)&IMG_DropBrick[DropBrick[which].Step];
	Blit[BlitAddIndex].A2_Flags	= DROPBRICK_FLAGS;
	Blit[BlitAddIndex].A2_Pixel	= DropOffsetTab[DropBrick[which].Size] & 
										  0xFFFFL;
	Blit[BlitAddIndex].A2_Step	= (1<<16L) | 
							(-DropWidthTab[DropBrick[which].Size] & 0x0FFFFL);

	Blit[BlitAddIndex].B_Count	= (18<<16L) | 
							(DropWidthTab[DropBrick[which].Size] & 0x0FFFFL);
										 
	Blit[BlitAddIndex].B_Patd	= 0x0L;
	Blit[BlitAddIndex].B_Srcz1	= DropBrick[which].Depth;
	Blit[BlitAddIndex].B_Cmd		= LFU_REPLACE | UPDA1 | UPDA2 | SRCEN |
										  DSTEN | DCOMPEN | DSTENZ | ZMODELT | 
										  DSTWRZ;
	
	QueBlit();
	}

/*********************************************************************
 *  FUNCTION: 	   DisplayDropBrick
 *
 *  PARAMETERS:	   short which
 *
 *  DESCRIPTION:   
 *
 *  RETURNS:	   void
 *
 *********************************************************************/
void DisplayDropBrick(short which)
	{
	ULONG step;

	int EndX;
	int Offset;
	int StepWidth;

	if(!DropBrick[which].Active)
		return;

	EndX = DropBrick[which].Screen.X+DropWidthTab[DropBrick[which].Size];
	Offset = EndX & 3;
	if(Offset)
		StepWidth = DropWidthTab[DropBrick[which].Size]+4-Offset;
	else
		StepWidth = DropWidthTab[DropBrick[which].Size];
	
	if(DropBrick[which].Screen.Y >= 100)
		step = (1<<16L) | (-StepWidth & 0x0FFFFL);
	else	
		step = (-1<<16L) | (-StepWidth & 0x0FFFFL);

	Blit[BlitAddIndex].Type		= blitBITMAP;

	Blit[BlitAddIndex].A1_Base	= (long)&IMG_Playfield;
	Blit[BlitAddIndex].A1_Flags	= PITCH1 | PIXEL16 | WID320 | XADDPHR;
	Blit[BlitAddIndex].A1_Pixel	= ((DropBrick[which].Screen.Y+20)<<16L) | 
								(DropBrick[which].Screen.X & 0x0FFFFL);
	Blit[BlitAddIndex].A1_Step	= step;
										 
	Blit[BlitAddIndex].A2_Base	= (long)&IMG_PlayfieldBuffer;
	Blit[BlitAddIndex].A2_Flags	= PITCH2 | PIXEL16 | WID320 | XADDPHR | 
										  ZOFFS1;
	Blit[BlitAddIndex].A2_Pixel	= (DropBrick[which].Screen.Y<<16L) | 
									(DropBrick[which].Screen.X & 0x0FFFFL);
	Blit[BlitAddIndex].A2_Step	= step;

	Blit[BlitAddIndex].B_Count	= (18<<16L) | 
							(DropWidthTab[DropBrick[which].Size] & 0x0FFFFL);
										 
	Blit[BlitAddIndex].B_Patd	= 0x0L;
	Blit[BlitAddIndex].B_Cmd		= LFU_REPLACE | UPDA1 | UPDA2 | SRCEN;
	
	QueBlit();
 	}
