/*******************************************************************
 *
 *    TITLE:        goody.c
 *
 *    DESCRIPTION:	Routines for handling goodies
 *
 *    AUTHOR:		Mario Perdue & Richard Degler
 *
 *    HISTORY:    
 *
 *
 *           COPYRIGHT 1994, 1995 MP Graphics Systems         
 *      UNATHORIZED REPRODUCTION, ADAPTATION, DISTRIBUTION,
 *      PERFORMANCE OR DISPLAY OF THIS COMPUTER PROGRAM OR   
 *     THE ASSOCIATED AUDIOVISUAL WORK IS STRICTLY PROHIBITED.
 *                        ALL RIGHTS RESERVED.               
 *******************************************************************/

/** include files **/
#include "\jaguar\include\jaguar.h"
#include "goody.h"
#include "brick.h"
#include "support.h"
#include "paddle.h"
#include "ball.h"
#include "displays.h"
#include "player.h"
#include "b2k_game.h"
#include "option.h"
#include "ripple.h"
#include "droid.h"


/** external data **/
extern	WORD			IMG_Goody[16][8][220];
extern	WORD			IMG_GoodyLaunch[8][1056];
extern	WORD			IMG_GoodyExplode[8][1056];
extern	WORD			IMG_Stinger[9][21][24];

extern	WORD			IMG_Playfield[];
extern	WORD  			IMG_PlayfieldBuffer[];
extern	WORD  			IMG_BrickBuffer[];

extern	WORD			ROM_Shatter[2][5][144];

extern DEF_RAM_PADDLE	IMG_Paddle[MAX_PLAYERS];

extern	DEF_PLAYER		Player[2];

extern 	int 			ScreenOrgX;
extern 	int				ScreenOrgY;

extern	int				BallSpeedLimit;
extern 	int				BallMaxSpeedX;
extern 	int				BallMaxSpeedY;
extern 	int				BallMaxSpeedZ;

extern	DEF_OPTION		Option;

extern	BYTE		 	PlayerOnTop;	
extern	DEF_BLIT_DATA	Blit[];
extern	LONG			BlitAddIndex;
extern	LONG			BlitUseIndex;

extern WORD				PaddleWidthTab[5];

/** internal functions **/
/*********************************************************************
 *  FUNCTION:		SetGoodyLocation
 *
 *  PARAMETERS:		short who
 *					int X
 *					int Y
 *					int Z
 *
 *  DESCRIPTION:	Place the goody at the specified 3D location
 *
 *  RETURNS:		void
 *
 *********************************************************************/
void  SetGoodyLocation( short who, int X, int Y, int Z )
	{
	SCREENLOC Loc;

	Player[who].Goody.X = X;
	Player[who].Goody.Y = Y;
	Player[who].Goody.Z = Z;

	Loc = PlayfieldToScreen(Player[who].Goody.X, Player[who].Goody.Y, 
							Player[who].Goody.Z);

	Player[who].Goody.ScreenX = Loc.X;
	Player[who].Goody.ScreenY = Loc.Y;

	Player[who].Goody.OldX = Loc.X;
	Player[who].Goody.OldY = Loc.Y;

	Player[who].Goody.Scale = GetDepthScale(Player[who].Goody.Z);
	}
	
/*********************************************************************
 *  FUNCTION:		SetGoodySpeed
 *
 *  PARAMETERS:		short who
 *					int SpeedX
 *					int SpeedY
 *					int SpeedZ
 *
 *  DESCRIPTION:	Set the goody speed
 *
 *  RETURNS:		void
 *
 *********************************************************************/
void  SetGoodySpeed( short who, int SpeedX, int SpeedY, int SpeedZ )
	{
	Player[who].Goody.SpeedX = SpeedX;
	Player[who].Goody.SpeedY = SpeedY;
	Player[who].Goody.SpeedZ = SpeedZ;
	}
	
/*********************************************************************
 *  FUNCTION:		LaunchGoody
 *
 *  PARAMETERS:		short who
 *					int X
 *					int Y
 *					int Z
 *
 *  DESCRIPTION:	Drop a random goody at the specified location
 *
 *  RETURNS:		void
 *
 *********************************************************************/
void  LaunchGoody( short who, short which, int X, int Y, int Z )
	{
	int xspeed;
	int yspeed;
	int zspeed;
	int volume;


	if(Player[who].Goody.Active)
		return;

	if(Player[who].Mode == modeKILLSCREEN)
		return;

	if(Player[who].Mode == modeREVIVESCREEN)
		return;

	if(!Player[who].Ball[0].Active && !Player[who].Ball[1].Active)
		return;

	if(Player[who].Ball[0].Style == ballEXPLODE)
		return;

	Player[who].Goody.Style = which;

	switch(Player[who].Goody.Style)
		{
		case goodyBALL_KILL:
			if((Player[who].Ball[0].Active+Player[who].Ball[1].Active) < 2)
				Player[who].Goody.Style = goodyBALL_FREE;
			break;

		case goodyBALL_SPLIT:
			/* Disallow Split ball in two player games. Drop Free Ball instead */
			if(Option.NumPlayers == 2)
				Player[who].Goody.Style = goodyBALL_FREE;
			else if((Player[who].Ball[0].Active+Player[who].Ball[1].Active) == 2)
				Player[who].Goody.Style = goodyBALL_KILL;
			break;

		case goodyBALL_FAST:
			if(BallMaxSpeedX >= BallSpeedLimit)
				Player[who].Goody.Style = goodyBALL_SLOW;
			break;

		case goodyBALL_SLOW:
			if(BallMaxSpeedZ <= BALL_MIN_SPEED)
				Player[who].Goody.Style = goodyBALL_FAST;
			break;

		case goodyATTRACT_UP:
			if(Player[who].Paddle.Attract == 1)
				Player[who].Goody.Style = goodyATTRACT_DOWN;
			break;

		case goodyATTRACT_DOWN:
			if(Player[who].Paddle.Attract == -1)
				Player[who].Goody.Style = goodyATTRACT_UP;
			break;

		case goodyCATCH_OFF:
			if(Player[who].Paddle.Catch == FALSE)
				Player[who].Goody.Style = goodyCATCH_ON;
			break;

		case goodyGUN_ON:
			if(Player[who].Paddle.Gun == TRUE)
				Player[who].Goody.Style = goodyGUN_OFF;
			break;

		case goodyGUN_OFF:
			if(Player[who].Paddle.Gun == FALSE)
				Player[who].Goody.Style = goodyGUN_ON;
			break;
		}

	Player[who].Goody.MaxSpeed = GOODY_MAX_SPEED;

	Player[who].Goody.Active = TRUE;
	
	xspeed = Random(Player[who].Goody.MaxSpeed*2)-Player[who].Goody.MaxSpeed;
	zspeed = Player[who].Goody.MaxSpeed;

	yspeed = (3200-abs(Y))/((TOTAL_RANGE-Z)/zspeed);

	if(who == PLAYER_ON_TOP)
		yspeed = -yspeed;

	SetGoodySpeed(who, xspeed, yspeed, zspeed);
	SetGoodyLocation(who, X, Y, Z);

	Player[who].Goody.Width = GOODY_LAUNCH_WIDTH;
	Player[who].Goody.Height = GOODY_LAUNCH_HEIGHT;
	
	Player[who].Goody.OldWidth = GOODY_LAUNCH_WIDTH;
	Player[who].Goody.OldHeight = GOODY_LAUNCH_HEIGHT;

	Player[who].Goody.Phase = phaseLAUNCH;

	volume = Player[who].Goody.Z*127/10000;
	PlaySound(sndGOODY_LAUNCH, volume, 0, 7);
	}

/*********************************************************************
 *  FUNCTION:		MoveGoody
 *
 *  PARAMETERS:		short who
 *
 *  DESCRIPTION:	Move the goody to the next screen location
 *
 *  RETURNS:		void
 *
 *********************************************************************/
void  MoveGoody(short who)
	{
	SCREENLOC Loc;
	int OffsetX, OffsetY;
	
	if(!Player[who].Goody.Active)
		return;

	Player[who].Goody.OldX = Player[who].Goody.ScreenX;
	Player[who].Goody.OldY = Player[who].Goody.ScreenY;
	Player[who].Goody.OldWidth = Player[who].Goody.Width;
	Player[who].Goody.OldHeight = Player[who].Goody.Height;

	/* Limit Goody Speed */
	if(Player[who].Goody.SpeedX >= Player[who].Goody.MaxSpeed)
		Player[who].Goody.SpeedX = Player[who].Goody.MaxSpeed;
	
	if(Player[who].Goody.SpeedX <= -Player[who].Goody.MaxSpeed)
		Player[who].Goody.SpeedX = -Player[who].Goody.MaxSpeed;

	if(Player[who].Goody.SpeedZ >= Player[who].Goody.MaxSpeed)
		Player[who].Goody.SpeedZ = Player[who].Goody.MaxSpeed;
	
	if(Player[who].Goody.SpeedZ <= -Player[who].Goody.MaxSpeed)
		Player[who].Goody.SpeedZ = -Player[who].Goody.MaxSpeed;

	/* Apply the goody movement */
	Player[who].Goody.X += Player[who].Goody.SpeedX;
	Player[who].Goody.Y += Player[who].Goody.SpeedY;
	Player[who].Goody.Z += Player[who].Goody.SpeedZ;

	Player[who].Goody.Scale = GetDepthScale(Player[who].Goody.Z);

	CheckGoodyCaught(who);
	CheckGoodyLimits(who);

	/* Step to next goody image */
	switch(Player[who].Goody.Phase)
		{
		case phaseLAUNCH:
			if(++Player[who].Goody.Step > 7)
				{
				Player[who].Goody.Step = 0;
				Player[who].Goody.Phase = phaseNORMAL;
				if(Player[who].Goody.Style == goodySTINGER)
					{
					Player[who].Goody.Width = STINGER_WIDTH;
					Player[who].Goody.Height = STINGER_HEIGHT;
					}
				else
					{
					Player[who].Goody.Width = GOODY_WIDTH;
					Player[who].Goody.Height = GOODY_HEIGHT;
					}
				}
			break;

		case phaseNORMAL:
			if(++Player[who].Goody.Step > 7)
				Player[who].Goody.Step = 0;
			break;

		case phaseEXPLODE:
			if(++Player[who].Goody.Step > 7)
				{
				EraseGoody(who);
				DisplayGoody(who);
				Player[who].Goody.Active = FALSE;
				}
			break;
		}

	OffsetX = -(Player[who].Goody.Width/2)*Player[who].Goody.Scale/100;
	if(who == PLAYER_ON_TOP)
		OffsetY = (Player[who].Goody.Height/2)*Player[who].Goody.Scale/100;
	else
		OffsetY = -(Player[who].Goody.Height/2)*Player[who].Goody.Scale/100;
	
	Loc = PlayfieldToScreen(Player[who].Goody.X, Player[who].Goody.Y, 
							Player[who].Goody.Z);
	Player[who].Goody.ScreenX = Loc.X+OffsetX;
	Player[who].Goody.ScreenY = MIN(MAX(Loc.Y+OffsetY, Player[who].Goody.Height), 
								PLAYFIELD_BUFFER_HEIGHT-Player[who].Goody.Height-1);
	}

/*********************************************************************
 *  FUNCTION:		CheckGoodyLimits
 *
 *  PARAMETERS:		short who
 *
 *  DESCRIPTION:	Keep the goody in the playfield
 *
 *  RETURNS:		void
 *
 *********************************************************************/
void  CheckGoodyLimits(short who)
	{
	int volume;

	if (!Player[who].Goody.Active)
		return;

	volume = Player[who].Goody.Z*127/1000;

	if(Player[who].Goody.X < PLAYFIELD_LEFT)
		{
		/* Goody hits Left side */
		PlaySound(sndWALL, volume, 0, 5);
		Player[who].Goody.X = PLAYFIELD_LEFT;
		Player[who].Goody.SpeedX = abs(Player[who].Goody.SpeedX);

		StartRipple(Player[who].Goody.X, Player[who].Goody.Y, 
					Player[who].Goody.Z, rippleLEFT);
		}
	else if(Player[who].Goody.X > PLAYFIELD_RIGHT-GOODY_W)  
		{
		/* Goody hits Right side */
		PlaySound(sndWALL, volume, 0, 5);
		Player[who].Goody.X = PLAYFIELD_RIGHT-GOODY_W;
		Player[who].Goody.SpeedX = -abs(Player[who].Goody.SpeedX);

		StartRipple(Player[who].Goody.X, Player[who].Goody.Y, 
					Player[who].Goody.Z, rippleRIGHT);
 		}
		
	if(Player[who].Goody.Z < PLAYFIELD_BACK)	  
		{
		/* Goody hits Back */
		Player[who].Goody.Z = PLAYFIELD_BACK;
		Player[who].Goody.SpeedZ = abs(Player[who].Goody.SpeedZ);
		}
	else if(Player[who].Goody.Z > PLAYFIELD_FRONT)  
		{
		/* Goody hits Front */
		if(Player[who].Goody.Style == goodySTINGER)
			{
			EraseGoody(who);
			DisplayGoody(who);
			Player[who].Goody.Active = FALSE;
			StartRipple(Player[who].Goody.X, Player[who].Goody.Y, 
						Player[who].Goody.Z, rippleBACK);
			}
		else
			ExplodeGoody(who);
		}
	}
	
/*********************************************************************
 *  FUNCTION:		CheckGoodyCaught
 *
 *  PARAMETERS:		short who
 *
 *  DESCRIPTION:	Check for goody paddle collision
 *
 *  RETURNS:		void
 *
 *********************************************************************/
void  CheckGoodyCaught(short who)
	{
	int Pwidth;
	int volume;
	int Gwidth;
	
	if(Player[who].Goody.Phase == phaseEXPLODE)
		return;

	if(Player[who].Goody.Z < Player[who].Paddle.Z)
		/* If above the paddle */
		return;
	
	Pwidth = (Player[who].Paddle.Size*PADDLE_STEP_W+PADDLE_MIN_W)/2;
	if(Player[who].Goody.Style == goodySTINGER)
		Gwidth = 32;
	else
		Gwidth = GOODY_W/2;
	
	if((Player[who].Goody.X+Gwidth) < (Player[who].Paddle.X-Pwidth))
		/* If Left of Paddle */
		return;
	
	if((Player[who].Goody.X-Gwidth) > (Player[who].Paddle.X+Pwidth))
		/* If right of Paddle */
		return;
	
	/* Getting here means the goody has hit the paddle */
	volume = Player[who].Goody.Z*127/10000;

	if(Player[who].Goody.Style == goodySTINGER)
		{
		PlaySound(sndSTINGER_HIT, volume, 0, 7);
		}
	else
		{
		PlaySound(sndEXTRA, volume, 0, 7);
		Player[who].Score += 1000;
		}

	ActivateGoody(who);

	EraseGoody(who);
	DisplayGoody(who);
	Player[who].Goody.Active  = FALSE;
	}


/*********************************************************************
 *  FUNCTION:	   ExplodeGoody
 *
 *  PARAMETERS:	   short who, short which
 *
 *  DESCRIPTION:   Plays explode sound and starts explode animation
 *
 *  RETURNS:	   void
 *
 *********************************************************************/
void ExplodeGoody(short who)
	{
	int volume;

	if(!Player[who].Goody.Active)
		return;

	if(Player[who].Goody.Phase == phaseEXPLODE)
		return;

	if(Player[who].Goody.Style == goodySTINGER)
		return;

	volume = Player[who].Goody.Z*127/10000;
	PlaySound(sndMISS, volume, 0, 7);
	
	SetGoodySpeed(who, 0, 0, 0);

	Player[who].Goody.Phase  = phaseEXPLODE;
	Player[who].Goody.Step   = -1;
	Player[who].Goody.Width	 = GOODY_EXPLODE_WIDTH;
	Player[who].Goody.Height = GOODY_EXPLODE_HEIGHT;
	}


/*********************************************************************
 *  FUNCTION:		BlankGoody
 *
 *  PARAMETERS:		short who
 *
 *  DESCRIPTION:	Removes the goody from the screen and set in inactive
 *
 *  RETURNS:		void
 *
 *********************************************************************/
void BlankGoody(short who)
	{
	Player[who].Goody.OldX = Player[who].Goody.ScreenX;
	Player[who].Goody.OldY = Player[who].Goody.ScreenY;
	Player[who].Goody.OldWidth = Player[who].Goody.Width;
	Player[who].Goody.OldHeight = Player[who].Goody.Height;
	EraseGoody(who);
	DisplayGoody(who);
	Player[who].Goody.Active = FALSE;
	}

	
/*********************************************************************
 *  FUNCTION:		EraseGoody
 *
 *  PARAMETERS:		short who
 *
 *  DESCRIPTION:	Remove a goody from the buffer
 *
 *  RETURNS:		void
 *
 *********************************************************************/
void EraseGoody(short who)
	{
	int EndX;
	int Offset;
	int StepWidth;
	int a1_step;

	if(!Player[who].Goody.Active)
		return;

	EndX = Player[who].Goody.OldX+Player[who].Goody.OldWidth;
	Offset = EndX & 3;
	if(Offset)
		StepWidth = Player[who].Goody.OldWidth+4-Offset;
	else
		StepWidth = Player[who].Goody.OldWidth;

	if(who == PLAYER_ON_TOP)
	   	a1_step  = 0xFFFF0000 | (-StepWidth & 0xFFFFL);
	else
	   	a1_step  = 0x00010000 | (-StepWidth & 0xFFFFL);

	Blit[BlitAddIndex].Type = blitBITMAP;

	Blit[BlitAddIndex].A1_Base  = (long)IMG_PlayfieldBuffer;
	Blit[BlitAddIndex].A1_Flags = PITCH2 | PIXEL16 | WID320 | XADDPHR |
								  ZOFFS1;
	Blit[BlitAddIndex].A1_Pixel = (Player[who].Goody.OldY<<16)|Player[who].Goody.OldX;
	Blit[BlitAddIndex].A1_Step  = a1_step;

	Blit[BlitAddIndex].A2_Base  = (long)IMG_BrickBuffer;
	Blit[BlitAddIndex].A2_Flags = Blit[BlitAddIndex].A1_Flags;
	Blit[BlitAddIndex].A2_Pixel = Blit[BlitAddIndex].A1_Pixel;
	Blit[BlitAddIndex].A2_Step  = Blit[BlitAddIndex].A1_Step;

	Blit[BlitAddIndex].B_Count  = (Player[who].Goody.OldHeight<<16)|Player[who].Goody.OldWidth;
	Blit[BlitAddIndex].B_Patd   = 0x0L;

	Blit[BlitAddIndex].B_Cmd    = LFU_REPLACE | UPDA1 | UPDA2 | SRCEN | 
								  DSTEN | SRCENZ | DSTENZ | DSTWRZ;
	
	QueBlit();
	}


/*********************************************************************
 *  FUNCTION:		DrawGoody
 *
 *  PARAMETERS:		short who
 *
 *  DESCRIPTION:	Draw a goody in the buffer
 *
 *  RETURNS:		void
 *
 *********************************************************************/
void DrawGoody(short who)
	{
	int ScaledWidth;
	int ScaledHeight;
	LONG Increment;
	int Fraction;

	long a1_base;
	int a1_flags;
	int a2_step;

	if(!Player[who].Goody.Active)
		return;

	ScaledWidth = Player[who].Goody.Width*Player[who].Goody.Scale/100;
	ScaledHeight = Player[who].Goody.Height*Player[who].Goody.Scale/100;

 	Increment = 0x10000*Player[who].Goody.Width/ScaledWidth;
	Fraction = Increment & 0x0FFFF;
 	Increment /= 0x010000;

	switch(Player[who].Goody.Phase)
		{
		case phaseLAUNCH:
			if(Player[who].Goody.Style == goodySTINGER)
				{
				a1_base		= (long)IMG_Stinger[Player[who].Goody.Step];
				a1_flags	= STINGER_FLAGS;
				}
			else
				{
				a1_base		= (long)IMG_GoodyLaunch[Player[who].Goody.Step];
				a1_flags	= GOODY_LAUNCH_FLAGS;
				}
			break;

		case phaseNORMAL:
		default:
			if(Player[who].Goody.Style == goodySTINGER)
				{
				a1_base		= (long)IMG_Stinger[8];
				a1_flags	= STINGER_FLAGS;
				}
			else
				{
				a1_base		= (long)IMG_Goody[Player[who].Goody.Style][Player[who].Goody.Step];
				a1_flags	= GOODY_FLAGS;
				}
			break;

		case phaseEXPLODE:
			a1_base		= (long)IMG_GoodyExplode[Player[who].Goody.Step];
			a1_flags	= GOODY_EXPLODE_FLAGS;
			break;
		}

	if(who == PLAYER_ON_TOP)
		a2_step	= 0xFFFF0000 | (-ScaledWidth & 0xFFFFL);
	else
		a2_step	= 0x00010000 | (-ScaledWidth & 0xFFFFL);

	Blit[BlitAddIndex].Type		= blitSCALED;

	Blit[BlitAddIndex].A1_Base	= a1_base;
	Blit[BlitAddIndex].A1_Flags	= a1_flags;
	Blit[BlitAddIndex].A1_Pixel	= 0x0L;
	Blit[BlitAddIndex].A1_Clip	= (Player[who].Goody.Height<<16)|Player[who].Goody.Width;

	Blit[BlitAddIndex].A1_Step	= (Increment<<16)|(-Player[who].Goody.Width & 0xFFFFL);
	Blit[BlitAddIndex].A1_Fpixel	= 0x0L;
	Blit[BlitAddIndex].A1_Fstep	= Fraction<<16;
	Blit[BlitAddIndex].A1_Inc	= Increment;
	Blit[BlitAddIndex].A1_Finc	= Fraction;
										 
	Blit[BlitAddIndex].A2_Base	= (long)IMG_PlayfieldBuffer;
	Blit[BlitAddIndex].A2_Flags	= PLAYFIELD_BUFFER_FLAGS;
	Blit[BlitAddIndex].A2_Pixel	= (Player[who].Goody.ScreenY<<16)|Player[who].Goody.ScreenX;
	Blit[BlitAddIndex].A2_Step	= a2_step;

	Blit[BlitAddIndex].B_Count	= (ScaledHeight<<16) | ScaledWidth;
										 
	Blit[BlitAddIndex].B_Patd	= 0x0L;
	Blit[BlitAddIndex].B_Srcz1	= Player[who].Goody.Z;

	Blit[BlitAddIndex].B_Cmd		= LFU_REPLACE | UPDA1 | UPDA1F | UPDA2 | 
										  DSTA2 | SRCEN | DSTEN | DCOMPEN | 
										  DSTENZ | DSTWRZ | ZMODELT | ZMODEEQ | 
										  CLIP_A1;
	
	QueBlit();
	}

/*********************************************************************
 *  FUNCTION:		DisplayGoody
 *
 *  PARAMETERS:		short who
 *
 *  DESCRIPTION:	Display a goody from the buffer
 *
 *  RETURNS:		void
 *
 *********************************************************************/
void DisplayGoody(short who)
	{
	SCREENLOC Loc;
	int DeltaX, DeltaY;
	int Width, Height;

	ULONG step;

	int EndX;
	int Offset;
	int StepWidth;

	if(!Player[who].Goody.Active)
		return;

	DeltaX = abs(Player[who].Goody.ScreenX-Player[who].Goody.OldX);
	DeltaY = abs(Player[who].Goody.ScreenY-Player[who].Goody.OldY);

	Loc.X = MIN(Player[who].Goody.ScreenX, Player[who].Goody.OldX);

	Width = MAX(Player[who].Goody.Width, Player[who].Goody.OldWidth)+DeltaX;
	Width = MIN(Width, PLAYFIELD_BUFFER_WIDTH-Loc.X-1);

	Height = MAX(Player[who].Goody.Height, Player[who].Goody.OldHeight)+DeltaY;

	EndX = Loc.X+Width;
	Offset = EndX & 3;
	if(Offset)
		StepWidth = Width+4-Offset;
	else
		StepWidth = Width;
	
	if(who == PLAYER_ON_TOP)
		{
		step  = 0xFFFF0000 | (-StepWidth & 0xFFFFL);
		Loc.Y = MAX(Player[who].Goody.ScreenY, Player[who].Goody.OldY);
		Height = MIN(Height, Loc.Y);
		}
	else
		{
		step  = 0x00010000 | (-StepWidth & 0xFFFFL);
		Loc.Y = MIN(Player[who].Goody.ScreenY, Player[who].Goody.OldY);
		Height = MIN(Height, PLAYFIELD_BUFFER_HEIGHT-Loc.Y-1);
		}

	Blit[BlitAddIndex].Type		= blitBITMAP;

	Blit[BlitAddIndex].A1_Base	= (long)IMG_Playfield;
	Blit[BlitAddIndex].A1_Flags	= PITCH1 | PIXEL16 | WID320 | XADDPHR;
	Blit[BlitAddIndex].A1_Pixel	= ((Loc.Y+20)<<16) | Loc.X;
	Blit[BlitAddIndex].A1_Step	= step;
										 
	Blit[BlitAddIndex].A2_Base	= (long)IMG_PlayfieldBuffer;
	Blit[BlitAddIndex].A2_Flags	= PITCH2 | PIXEL16 | WID320 | XADDPHR | ZOFFS1;
	Blit[BlitAddIndex].A2_Pixel	= (Loc.Y<<16) | Loc.X;
	Blit[BlitAddIndex].A2_Step	= step;

	Blit[BlitAddIndex].B_Count	= (Height<<16) | Width;
										 
	Blit[BlitAddIndex].B_Patd	= 0x0L;
	Blit[BlitAddIndex].B_Cmd	= LFU_REPLACE | UPDA1 | UPDA2 | SRCEN;
	
	QueBlit();
	}


/*********************************************************************
 *  FUNCTION:		ActivateGoody
 *
 *  PARAMETERS:		short who
 *
 *  DESCRIPTION:	Start goody effect
 *
 *  RETURNS:		void
 *
 *********************************************************************/
void ActivateGoody(short who)
	{
	switch(Player[who].Goody.Style)
		{
		case goodyBALL_KILL:
			DoKillBall(who);
			break;
			
		case goodyBALL_FAST:
			DoFastBall(who);
			break;
			
		case goodyBALL_SLOW:
			DoSlowBall(who); 
			break;
			
		case goodyBALL_FREE:
			DoFreeBall(who);
			break;
			
		case goodyBALL_SPLIT:
			DoSplitBall(who);
			break;
			
		case goodyBALL_SUPER:
			DoSuperBall(who);
			break;
			
		case goodyPADDLE_NARROW:
			SetPaddleSize(who, Player[who].Paddle.Size-1);
			break;
			
		case goodyPADDLE_WIDE:
			SetPaddleSize(who, Player[who].Paddle.Size+1);
		   break;
			
		case goodyCATCH_ON:
			if(!Player[who].Paddle.Cracks)
				{
				Player[who].Paddle.Catch = TRUE;
				DrawPaddleFace(who);
				if(who == PLAYER_RED)
					DrawIcon(iconCATCH, CATCH_RED_X, CATCH_RED_Y, ON);
				else
					DrawIcon(iconCATCH, CATCH_GREEN_X, CATCH_GREEN_Y, ON);
				}
			break;
			
		case goodyCATCH_OFF:
			Player[who].Ball[0].Caught = FALSE;
			Player[who].Ball[1].Caught = FALSE;
			Player[who].Paddle.Catch = FALSE;
			DrawPaddleFace(who);
			if(who == PLAYER_RED)
				DrawIcon(iconCATCH, CATCH_RED_X, CATCH_RED_Y, OFF);
			else
				DrawIcon(iconCATCH, CATCH_GREEN_X, CATCH_GREEN_Y, OFF);
			break;
			
		case goodyATTRACT_DOWN:
			Player[who].Paddle.Attract = MAX(-1, Player[who].Paddle.Attract-1);
			if(who == PLAYER_RED)
				DrawIcon(iconREPEL, ATTRACT_RED_X, ATTRACT_RED_Y, Player[who].Paddle.Attract);
			else
				DrawIcon(iconREPEL, ATTRACT_GREEN_X, ATTRACT_GREEN_Y, Player[who].Paddle.Attract);
			break;
			
		case goodyATTRACT_UP:
			Player[who].Paddle.Attract = MIN(1, Player[who].Paddle.Attract+1);
			if(who==PLAYER_RED)
				DrawIcon(iconATTRACT, ATTRACT_RED_X, ATTRACT_RED_Y, Player[who].Paddle.Attract);
			else
				DrawIcon(iconATTRACT, ATTRACT_GREEN_X, ATTRACT_GREEN_Y, Player[who].Paddle.Attract);
			break;
			
		case goodySCREEN_KILL:
			DoKillScreen(who);
			break;

		case goodySCREEN_REVIVE:
			DoReviveScreen(who);
			break;

		case goodyGUN_ON:
			Player[who].Paddle.Gun = TRUE;
			Player[who].Paddle.GunPower = 0;
			DrawPaddleGun(who);
			if(who==PLAYER_RED)
				DrawIcon(iconGUN, GUN_RED_X, GUN_RED_Y, ON);
			else
				DrawIcon(iconGUN, GUN_GREEN_X, GUN_GREEN_Y, ON);
			break;
			
		case goodyGUN_OFF:
			Player[who].Paddle.Gun = FALSE;
			DrawPaddleFrame(who);
			if(who==PLAYER_RED)
				DrawIcon(iconGUN, GUN_RED_X, GUN_RED_Y, OFF);
			else
				DrawIcon(iconGUN, GUN_GREEN_X, GUN_GREEN_Y, OFF);
			break;

		case goodySTINGER:
			DoStinger(who);
			break;			
		}
	}

/*********************************************************************
 *  FUNCTION:		DoKillBall
 *
 *  PARAMETERS:		short who
 *
 *  DESCRIPTION:	Kill Ball Goody
 *
 *  RETURNS:		void
 *
 *********************************************************************/
void DoKillBall(short who)
	{
	if(Player[who].Ball[1].Active)
		ExplodeBall(who, 1, who);
	else if(Player[who].Ball[0].Active)
		ExplodeBall(who, 0, who);
	}

/*********************************************************************
 *  FUNCTION:		DoFastBall
 *
 *  PARAMETERS:		short who
 *
 *  DESCRIPTION:	FastBallGoody
 *
 *  RETURNS:		void
 *
 *********************************************************************/
void DoFastBall(short who)
	{
	int i;

	BallMaxSpeedX = MIN(BallMaxSpeedX+SKILL_SPEED_INCR, BallSpeedLimit);
	BallMaxSpeedY = BallMaxSpeedX*2;
	BallMaxSpeedZ = BallMaxSpeedX*75/100;
	for(i=0; i<MAX_BALLS; i++)
		{
		if(Player[who].Ball[i].SpeedX>0)
			Player[who].Ball[i].SpeedX += SKILL_SPEED_INCR;
		else
			Player[who].Ball[i].SpeedY -= SKILL_SPEED_INCR;
		
		switch (Player[who].Ball[i].Sector)
			{
			case sectorTOP:
			case sectorBOTTOM:
				if(Player[who].Ball[i].SpeedZ>0)
					Player[who].Ball[i].SpeedZ += SKILL_SPEED_INCR;
				else
					Player[who].Ball[i].SpeedZ -= SKILL_SPEED_INCR;
				Player[who].Ball[i].SpeedY = 0;
				break;

			case sectorBACK:
				if(Player[who].Ball[i].SpeedY>0)
					Player[who].Ball[i].SpeedY += SKILL_SPEED_INCR;
				else
					Player[who].Ball[i].SpeedY -= SKILL_SPEED_INCR;
				Player[who].Ball[i].SpeedZ = 0;
				break;
			}
		}
	}

/*********************************************************************
 *  FUNCTION:		DoSlowBall
 *
 *  PARAMETERS:		short who
 *
 *  DESCRIPTION:	Slow Ball Goody
 *
 *  RETURNS:		void
 *
 *********************************************************************/
void DoSlowBall(short who)
	{
	int i;

	for(i=0; i<MAX_BALLS; i++)
		{
		BallMaxSpeedX = MAX(abs(BallMaxSpeedX-SKILL_SPEED_INCR), 0);
		BallMaxSpeedY = MAX(abs(BallMaxSpeedY-SKILL_SPEED_INCR), BALL_MIN_SPEED);
		BallMaxSpeedZ = MAX(abs(BallMaxSpeedZ-SKILL_SPEED_INCR), BALL_MIN_SPEED);

		if(Player[who].Ball[i].SpeedX>0)
			Player[who].Ball[i].SpeedX -= SKILL_SPEED_INCR;
		else
			Player[who].Ball[i].SpeedX += SKILL_SPEED_INCR;
		
		switch (Player[who].Ball[i].Sector)
			{
			case sectorTOP:
			case sectorBOTTOM:
				if(Player[who].Ball[i].SpeedZ>0)
					Player[who].Ball[i].SpeedZ -= SKILL_SPEED_INCR;
				else
					Player[who].Ball[i].SpeedZ += SKILL_SPEED_INCR;
				Player[who].Ball[i].SpeedY = 0;
				break;

			case sectorBACK:
				if(Player[who].Ball[i].SpeedY>0)
					Player[who].Ball[i].SpeedY -= SKILL_SPEED_INCR;
				else
					Player[who].Ball[i].SpeedY += SKILL_SPEED_INCR;
				Player[who].Ball[i].SpeedZ = 0;
				break;
			}
	  	}
	}

/*********************************************************************
 *  FUNCTION:		DoFreeBall
 *
 *  PARAMETERS:		short who
 *
 *  DESCRIPTION:	Free Ball Goody
 *
 *  RETURNS:		void
 *
 *********************************************************************/
void DoFreeBall(short who)
	{
	if(Player[who].BallCount < 99)
		Player[who].BallCount++;
	
	UpdateBallCount(who);
	}

/*********************************************************************
 *  FUNCTION:		DoSplitBall
 *
 *  PARAMETERS:		void
 *
 *  DESCRIPTION:	SplitBallGoody
 *
 *  RETURNS:		void
 *
 *********************************************************************/
void DoSplitBall(short who)
	{
	int sourceBall, split;

	if(Player[who].Ball[0].Active)
		{
		sourceBall = 0;
		split = 1;
		}
	else if (Player[who].Ball[1].Active)
		{
		sourceBall = 1;
		split = 0;
		}
	else
		return;

	if(Player[who].Ball[sourceBall].Style == ballEXPLODE)
		return;

	if(Player[who].Ball[split].Active)
		return;

   	Player[who].Ball[split] = Player[who].Ball[sourceBall];
   	Player[who].Ball[sourceBall].SpeedX += 4;
   	Player[who].Ball[split].SpeedX -= 4;
	}

/*********************************************************************
 *  FUNCTION:		DoSuperBall
 *
 *  PARAMETERS:		short who
 *
 *  DESCRIPTION:	Super Ball Goody
 *
 *  RETURNS:		void
 *
 *********************************************************************/
void DoSuperBall(short who)
	{
	int i;

	for(i=0; i<MAX_BALLS; i++)
		{
		switch(Player[who].Ball[i].Style)
			{
			case ballLAUNCH:
			case ballEXPLODE:
				return;
				break;
			
			default:
				Player[who].Ball[i].Style = ballSUPER;
				Player[who].BallStyle = ballSUPER;

				if(who==PLAYER_RED)
					DrawIcon(iconBLUEBALL, SUPERBALL_RED_X, SUPERBALL_RED_Y, ON);
				else
					DrawIcon(iconBLUEBALL, SUPERBALL_GREEN_X, SUPERBALL_GREEN_Y, ON);
				break;
			}
		}
	}

/*********************************************************************
 *  FUNCTION: 	   DoKillScreen
 *
 *  PARAMETERS:	   short who
 *
 *  DESCRIPTION:   Initiate Screen Revival Mode
 *
 *  RETURNS:	   void
 *
 *********************************************************************/
void DoKillScreen(short who)
	{
	AbortDroid();
	ExplodeGoody(who^1);
	BlankBall(PLAYER_RED,0);
	BlankBall(PLAYER_RED,1);
	BlankBall(PLAYER_GREEN,0);
	BlankBall(PLAYER_GREEN,1);

	Player[who].Tier = 2;
	Player[who].Col = 0;
	Player[who].Row = 0;
	Player[who].Mode = modeKILLSCREEN;
	}
			

/*********************************************************************
 *  FUNCTION: 	   DoReviveScreen
 *
 *  PARAMETERS:	   short who
 *
 *  DESCRIPTION:   Initiate Screen Revival Mode
 *
 *  RETURNS:	   void
 *
 *********************************************************************/
void DoReviveScreen(short who)
	{
	AbortDroid();
	Player[who].Col = 0;
	Player[who].Row = 0;
	Player[who].Mode = modeREVIVESCREEN;
	}
			

/*********************************************************************
 *  FUNCTION: 	   LaunchStinger
 *
 *  PARAMETERS:	   short who, x, y, z
 *
 *  DESCRIPTION:   Launches Stinger
 *
 *  RETURNS:	   void
 *
 *********************************************************************/
void LaunchStinger(short who, int X, int Y, int Z)
	{
	int xspeed;
	int yspeed;
	int zspeed;
	int DeltaX;
	int Steps;
	int volume;

	if(Player[who].Goody.Active)
		return;

	if(Player[who].Mode == modeREVIVESCREEN)
		return;

	if(!Player[who].Paddle.Active)
		return;

	Player[who].Goody.Style = goodySTINGER;

	Player[who].Goody.MaxSpeed = STINGER_MAX_SPEED;

	Player[who].Goody.Active = TRUE;
	
	zspeed = Player[who].Goody.MaxSpeed;

	Steps = ((TOTAL_RANGE-Z)/zspeed);
	DeltaX = Player[who].Paddle.X-X;

	xspeed = DeltaX/Steps;

	yspeed = (3200-abs(Y))/Steps;

	if(who == PLAYER_ON_TOP)
		yspeed = -yspeed;

	SetGoodySpeed(who, xspeed, yspeed, zspeed);
	SetGoodyLocation(who, X, Y, Z);

	Player[who].Goody.Width = STINGER_WIDTH;
	Player[who].Goody.Height = STINGER_HEIGHT;
	
	Player[who].Goody.OldWidth = STINGER_WIDTH;
	Player[who].Goody.OldHeight = STINGER_HEIGHT;

	Player[who].Goody.Phase = phaseLAUNCH;

	volume = Player[who].Goody.Z*127/10000;
	PlaySound(sndSTINGER_LAUNCH, volume, 0, 7);
	}


void DoStinger(short who)
	{
	WORD X;
	int player;
	int ball;
	int paddlewho;

	Player[who].Paddle.Cracks++;
	if(Player[who].Paddle.Catch)
		{
		Player[who].Paddle.Catch = FALSE;
 		DrawPaddleFace(who);

		for(player=0; player<2; player++)
		 	{
			for(ball=0; ball<2; ball++)
				{
				if(Player[player].Ball[ball].Sector == sectorTOP)
					paddlewho = PLAYER_ON_TOP;
				else
					paddlewho = PLAYER_ON_BOTTOM;

				if(who == paddlewho)
					{
					if(Player[player].Ball[ball].Caught)
						{
						Player[player].Ball[ball].Caught = FALSE;
						SetBallSpeed(player, ball, Player[who].Paddle.SpeedX, 0, 
										-BallMaxSpeedZ/2);
						}
					}
				}
		  	}
		}

	if(Player[who].Paddle.Cracks == MAX_CRACKS)
		{
		Player[who].Paddle.Step = 0;
		DrawPaddleFace(who);
		}
	else
		{
		X = MIN(PaddleWidthTab[Player[who].Paddle.Size]-15,
				MAX(2,(Player[who].Goody.ScreenX-Player[who].Paddle.ScreenX)+6));

		X &= 0xFFFC;
		Blit[BlitAddIndex].Type = blitBITMAP;

		Blit[BlitAddIndex].A1_Base  = (long)IMG_Paddle[who].Face;
		Blit[BlitAddIndex].A1_Flags = PITCH1 | PIXEL16 | WID64 | XADDPHR;
		Blit[BlitAddIndex].A1_Pixel = X;
		Blit[BlitAddIndex].A1_Step  = 0x00010000 | (-CRACK_WIDTH&0xFFFFL);

		Blit[BlitAddIndex].A2_Base  = (long)ROM_Shatter[0][Random(10)];
		Blit[BlitAddIndex].A2_Flags = PITCH1 | PIXEL16 | WID12 | XADDPHR;
		Blit[BlitAddIndex].A2_Pixel = 0x0L;
		Blit[BlitAddIndex].A2_Step  = Blit[BlitAddIndex].A1_Step;

		Blit[BlitAddIndex].B_Count  = (CRACK_HEIGHT<<16L) | CRACK_WIDTH;
		Blit[BlitAddIndex].B_Patd   = 0x0L;

		Blit[BlitAddIndex].B_Cmd    = LFU_REPLACE | UPDA1 | UPDA2 | SRCEN |
									  DSTEN | DCOMPEN;
		
		QueBlit();
		}
	
	DrawPaddleFace(who);
	}