/*******************************************************************
 *
 *	TITLE:			option.c
 *
 *	DESCRIPTION:	Option screen handling routines.
 *
 *	AUTHOR:			Mario Perdue  Richard Degler
 *
 *	HISTORY:		July 11, 1995		New Program
 *
 *
 *		   COPYRIGHT 1994, 1995 MP Graphics Systems		 
 *	  UNATHORIZED REPRODUCTION, ADAPTATION, DISTRIBUTION,
 *	  PERFORMANCE OR DISPLAY OF THIS COMPUTER PROGRAM OR   
 *	 THE ASSOCIATED AUDIOVISUAL WORK IS STRICTLY PROHIBITED.
 *						ALL RIGHTS RESERVED.			   
 *******************************************************************/

/** include files      **/
#include "\jaguar\include\jaguar.h"
#include "support.h"
#include "option.h"
#include "player.h"
#include "b2k_game.h"
#include "hiscore.h"
#include "displays.h"

extern	WORD	IMG_Playfield[];
extern	WORD	IMG_PlayfieldBuffer[];

extern	WORD	ROM_OptionGame[2][32][112];
extern  WORD	ROM_OptionText[10][15][64];
extern  WORD	ROM_OptionButton[7][256];
extern  WORD	ROM_OptionStart[32][96];
extern  WORD	ROM_OptionStartButton[2][512];
extern	WORD	ROM_OptionUsage[14][256];
extern	WORD	ROM_OptionPlayer[2][80][224];
extern	WORD	ROM_OptionSound[56][224];
extern  WORD	ROM_OptionSlide[2][2][56];

extern	DEF_PLAYER	Player[];

extern	DEF_OPTION	Option;

extern	ULONG	ticks;			 
extern	ULONG	DemoTimer;

extern 	int		Screen;
extern 	int		ScreenCount;
extern	char 	NUM_SCREENS;

extern	BYTE	PlayerOnTop;

extern	DEF_BLIT_DATA	Blit[];
extern	LONG			BlitAddIndex;
extern	LONG			BlitUseIndex;

extern	WORD			ROM_Panel[];

DEF_EEPROM		EepromBuffer;

/** public functions   **/

/*********************************************************************
 *  FUNCTION: 	   ReadOptions
 *
 *  PARAMETERS:	   void
 *
 *  DESCRIPTION:   Reads Options from EEPROM.  Sets default values on
 *                 bad read.
 *
 *  RETURNS:	   void
 *
 *********************************************************************/
void ReadOptions( void )
	{
	EepromBuffer.Magic = 0;

	if(eeReadBank(&EepromBuffer))
		{
		ResetOptions(TRUE);
		return;
		}

	if(EepromBuffer.Magic != MAGIC_NUMBER)
		{
		ResetOptions(TRUE);
		return;
		}

	Option.NumPlayers        = MIN(EepromBuffer.NumPlayers, 2);
	Option.Skill[0]          = MIN(EepromBuffer.Skill1, 2);
	Option.Skill[1]          = MIN(EepromBuffer.Skill2, 4);
	Option.PaddleResponse[0] = MIN(EepromBuffer.PaddleSpeed1, 2);
	Option.PaddleResponse[1] = MIN(EepromBuffer.PaddleSpeed2, 2);
	Option.ScreenWrapOn      = FALSE;
	Option.GameMode			 = EepromBuffer.GameMode;
	Option.Catch			 = EepromBuffer.Catch;
	Option.Breakthru		 = EepromBuffer.Breakthru;
	Option.EffectsVolume     = MIN(EepromBuffer.EffectVolume, 127);
	Option.MusicOn           = EepromBuffer.MusicOn;
	Option.MusicVolume       = MIN(EepromBuffer.MusicVolume, 127);

	SMainVolume(Option.MusicVolume <<8);
	SSFXVolume(Option.EffectsVolume <<8);

	ReadHighScores();
	}


/*********************************************************************
 *  FUNCTION: 	   ResetOptions
 *
 *  PARAMETERS:	   void
 *
 *  DESCRIPTION:   Sets default values for Options.
 *                  
 *
 *  RETURNS:	   void
 *
 *********************************************************************/
void ResetOptions( int write )
	{
	Option.NumPlayers        = 1;
	Option.Skill[0]          = skillNORMAL;
	Option.Skill[1]          = skillNONE;
	Option.PaddleResponse[0] = paddleMEDIUM;
	Option.PaddleResponse[1] = paddleMEDIUM;
	Option.ScreenWrapOn      = FALSE;
	Option.GameMode			 = game2000;
	Option.Catch			 = FALSE;
	Option.Breakthru		 = FALSE;
	Option.EffectsVolume     = 95;
	Option.MusicOn           = TRUE;
	Option.MusicVolume       = 95;

	SMainVolume(Option.MusicVolume <<8);
	SSFXVolume(Option.EffectsVolume <<8);

	ResetHighScores(write);

	if(write)
		WriteOptions();
	}

/*********************************************************************
 *  FUNCTION: 	   WriteOption
 *
 *  PARAMETERS:	   void
 *
 *  DESCRIPTION:   Writes Option to EEPROM.  Updates checksum.
 *
 *  RETURNS:	   void
 *
 *********************************************************************/
void WriteOptions( )
	{
	WORD *Buffer;

	Buffer = (WORD *)&EepromBuffer;

	EepromBuffer.NumPlayers	 = Option.NumPlayers;
	EepromBuffer.Skill1		 = Option.Skill[0];
	EepromBuffer.Skill2		 = Option.Skill[1];
	EepromBuffer.PaddleSpeed1= Option.PaddleResponse[0];
	EepromBuffer.PaddleSpeed2= Option.PaddleResponse[1];
	EepromBuffer.EffectVolume= Option.EffectsVolume;
    EepromBuffer.GameMode	 = Option.GameMode;
    EepromBuffer.Catch		 = Option.Catch; 
    EepromBuffer.Breakthru	 = Option.Breakthru;
	EepromBuffer.MusicOn	 = Option.MusicOn;
	EepromBuffer.MusicVolume = Option.MusicVolume;
	EepromBuffer.Magic		 = MAGIC_NUMBER;

	eeWriteWord(Buffer[FLAG_WORD], FLAG_WORD);
	eeWriteWord(Buffer[MAGIC_WORD], MAGIC_WORD);
	eeWriteWord(Buffer[VOLUME_WORD], VOLUME_WORD);

	eeUpdateChecksum();
	eeValidateChecksum();
	}


/*********************************************************************
 *  FUNCTION: 	   DoOptions
 *
 *  PARAMETERS:	   short who
 *
 *  DESCRIPTION:   Handle Option Screen
 *
 *  RETURNS:	   void
 *
 *********************************************************************/
void DoOptions()
	{
	int SelectedItem;
	int OldItem;
	int KeyPad;
	char Done;
	char Start;
	ULONG Time;
	ULONG ElapsedTime;

/*	ShowMessage(msgLOAD);*/

	while(BlitAddIndex != BlitUseIndex);

	Done = FALSE;
	Start = FALSE;

	Time = ticks;
	ElapsedTime = ticks;

	while(!Done)
	 	{
	  	SelectedItem = itemSTART;
	 	DisplayOptionScreen();

		/* Wait until all buttons are released */
		while(ReadButtons(PLAYER_RED) & BUTTON_ANY);

		/* Loop until the start key is pressed */
		while(!Start)
			{
			OldItem = SelectedItem;
			SelectedItem = GetSelectedItem(OldItem);

			if (SelectedItem != OldItem)
				{
				ElapsedTime = ticks;
				PlaySound(sndCRACK, 127, 0, 7);

				DisplayGameOption();
				DisplayButton(OPT_GAME_BUTTON_X, OPT_GAME_BUTTON_Y, btnBLUE, OFF);
				DisplayStartButton(OFF);
				DisplayButton(OPT_SKILL_RED_X, OPT_SKILL_RED_Y, btnRED, OFF);
				DisplayButton(OPT_PADDLE_RED_X, OPT_PADDLE_RED_Y, btnRED, OFF);
				if(Option.GameMode==game2000)
					{
					DisplayButton(OPT_SKILL_GREEN_X, OPT_SKILL_GREEN_Y, btnGREEN, OFF);
					DisplayButton(OPT_PADDLE_GREEN_X, OPT_PADDLE_GREEN_Y, btnGREEN, OFF);
					}
				else
					{
					DisplayButton(OPT_CATCH_X, OPT_CATCH_Y, btnRED, OFF);
					DisplayButton(OPT_BREAKTHRU_X, OPT_BREAKTHRU_Y, btnRED, OFF);
					}
				DisplayMeterLevel(OFF, OPT_EFFECTS_SLIDE_X, OPT_EFFECTS_SLIDE_Y, 
									Option.EffectsVolume);
				DisplayMeterLevel(OFF, OPT_MUSIC_SLIDE_X, OPT_MUSIC_SLIDE_Y, 
									Option.MusicVolume);

				switch(SelectedItem)
					{
					case itemGAMEMODE:
						DisplayGameOption();
						DisplayButton(OPT_GAME_BUTTON_X, OPT_GAME_BUTTON_Y, btnBLUE, ON);
						break;

					case itemSTART:
						DisplayStartButton(ON);
						break;

					case itemSKILL_RED:
						DisplayButton(OPT_SKILL_RED_X, OPT_SKILL_RED_Y, btnRED, ON);
						break;

					case itemPADDLE_RED:
						DisplayButton(OPT_PADDLE_RED_X, OPT_PADDLE_RED_Y, btnRED, ON);
						break;

					case itemSKILL_GREEN:
						if(Option.GameMode == game2000)
							DisplayButton(OPT_SKILL_GREEN_X, OPT_SKILL_GREEN_Y, btnGREEN, ON);
						else
							DisplayButton(OPT_CATCH_X, OPT_CATCH_Y, btnRED, ON);
						break;

					case itemPADDLE_GREEN:
						if(Option.GameMode == game2000)
							DisplayButton(OPT_PADDLE_GREEN_X, OPT_PADDLE_GREEN_Y, btnGREEN, ON);
						else
							DisplayButton(OPT_BREAKTHRU_X, OPT_BREAKTHRU_Y, btnRED, ON);
						break;

					case itemEFFECTS_VOL:
						DisplayMeterLevel(TRUE, OPT_EFFECTS_SLIDE_X, 
										OPT_EFFECTS_SLIDE_Y, Option.EffectsVolume);
						break;

					case itemMUSIC_VOL:
						DisplayMeterLevel(TRUE, OPT_MUSIC_SLIDE_X, 
										OPT_MUSIC_SLIDE_Y, Option.MusicVolume);
						break;
					}
				}

			if(ReadButtons(PLAYER_RED) & BUTTON_ANY)
				ElapsedTime = ticks;

			switch(SelectedItem)
				{
				case itemGAMEMODE:
					HandleGameOption();
					break;

				case itemSTART:
					if(ReadButtons(PLAYER_RED) & BUTTON_FIRE)
						{
						PlaySound(sndBRICK2, 127, 4, 7);
						Start = TRUE;
						Done = TRUE;
						}
					break;

				case itemSKILL_RED:
					HandleSkillOption(PLAYER_RED);
					break;

				case itemPADDLE_RED:
					HandlePaddleOption(PLAYER_RED);
					break;

				case itemSKILL_GREEN:
					if(Option.GameMode == game2000)
						HandleSkillOption(PLAYER_GREEN);
					else
						HandleCatchOption();
					break;

				case itemPADDLE_GREEN:
					if(Option.GameMode == game2000)
						HandlePaddleOption(PLAYER_GREEN);
					else
						HandleBreakthruOption();
					break;

				case itemEFFECTS_VOL:
					if((ticks-Time) > 1000)
						{
						Time = ticks;
						PlaySound(sndSHOOT, 127, 0, 7);
						}
					HandleEffectsVolOption();
					break;

				case itemMUSIC_VOL:
					HandleMusicVolOption();
					break;
				}
			
			KeyPad = ReadKeyPad(PLAYER_RED);
			if(KeyPad == KEYPAD_RESET)
				{
				ElapsedTime = ticks;
				while(abs(ticks-ElapsedTime) < 500);

				if(ReadButtons(PLAYER_RED) & BUTTON_OPTION)
					{
					StopMusic();
					ShowMessage(msgCLEAR);
					ResetOptions(TRUE);

				  	SelectedItem = itemSTART;
				 	DisplayOptionScreen();
					while(BlitAddIndex != BlitUseIndex);
					StartMusic(songTITLE);
					}
				else
					{
					StopMusic();
					Player[PLAYER_RED].Mode = modeTITLE;
					Player[PLAYER_GREEN].Mode = modeIDLE;
					return;
					}
				}
			
			if(ticks-ElapsedTime > 30000)
				{
				/* Start Demo Mode */
				Option.GameMode = game2000;

				Player[PLAYER_RED].AutoPilot = ON;
				Player[PLAYER_GREEN].AutoPilot = ON;

				DemoTimer = ticks;

				Screen = Random(NUM_SCREENS);

				ScreenCount = Screen;
				Option.NumPlayers = Random(MAX_PLAYERS)+1;
				if(Option.NumPlayers == 2)
					{
					SetPlayfieldBack(ROUND);
					Option.ScreenWrapOn = TRUE;
					}
				else
					{
					SetPlayfieldBack(FLAT);
					Option.ScreenWrapOn = FALSE;
					}

				Player[PLAYER_RED].Mode = modeSTARTGAME;
				Player[PLAYER_GREEN].Mode = modeSTARTGAME;
				StopMusic();
				return;
				}
			}
		}

	WriteOptions();

	Player[PLAYER_RED].Mode = modeSTARTGAME;
	Player[PLAYER_GREEN].Mode = modeSTARTGAME;

	if(Start)
		{
		StopMusic();

		Player[PLAYER_RED].AutoPilot = OFF;

		switch(Option.Skill[PLAYER_GREEN])
			{
			case skillCOMPUTER:
				Option.NumPlayers = 2;
				Player[PLAYER_GREEN].AutoPilot = ON;
				break;

			case skillNONE:
				Option.NumPlayers = 1;
				Player[PLAYER_GREEN].AutoPilot = OFF;
				break;

			default:
				Option.NumPlayers = 2;
				Player[PLAYER_GREEN].AutoPilot = OFF;
			}
			
		if(Option.NumPlayers == 2)
			{
			SetPlayfieldBack(ROUND);
			Option.ScreenWrapOn = TRUE;
			}
		else
			{
			SetPlayfieldBack(FLAT);
			Option.ScreenWrapOn = FALSE;
			}

		StopMusic();

		PlayerOnTop = PLAYER_GREEN;

		DemoTimer = 0;
		}
	}

/*********************************************************************
 *  FUNCTION: 	   GetSelectedItem
 *
 *  PARAMETERS:	   
 *
 *  DESCRIPTION:   
 *
 *  RETURNS:	   
 *
 *********************************************************************/
int GetSelectedItem(int CurrentItem)
	{
	int JoyStick;
	int SelectedItem;

	JoyStick = ReadJoyStick(PLAYER_RED);

	while(ReadJoyStick(PLAYER_RED) & (JOYPAD_UP | JOYPAD_DOWN));

	SelectedItem = CurrentItem;

	if(JoyStick & JOYPAD_ANY)
		{
		switch(JoyStick)
			{
			case JOYPAD_DOWN:
				switch (SelectedItem)
					{
					case itemGAMEMODE:
						SelectedItem = itemSKILL_RED;
						break;

					case itemSTART:
						SelectedItem = itemSKILL_GREEN;
						break;

					case itemSKILL_RED:
						SelectedItem = itemPADDLE_RED;
						break;

					case itemSKILL_GREEN:
						SelectedItem = itemPADDLE_GREEN;
						break;

					case itemPADDLE_RED:
					case itemPADDLE_GREEN:
						SelectedItem = itemMUSIC_VOL;
						break;
						
					case itemMUSIC_VOL:
						SelectedItem = itemEFFECTS_VOL;
						break;

					case itemEFFECTS_VOL:
						SelectedItem = itemGAMEMODE;
						break;
				 	}
				break;

			case JOYPAD_UP:
				switch (SelectedItem)
					{
					case itemGAMEMODE:
						SelectedItem = itemEFFECTS_VOL;
						break;

					case itemSTART:
						SelectedItem = itemEFFECTS_VOL;
						break;

					case itemSKILL_RED:
						SelectedItem = itemGAMEMODE;
						break;

					case itemSKILL_GREEN:
						SelectedItem = itemSTART;
						break;

					case itemPADDLE_RED:
						SelectedItem = itemSKILL_RED;
						break;

					case itemPADDLE_GREEN:
						SelectedItem = itemSKILL_GREEN;
						break;

					case itemMUSIC_VOL:
						SelectedItem = itemPADDLE_RED;
						break;

					case itemEFFECTS_VOL:
						SelectedItem = itemMUSIC_VOL;
						break;
				 	}
				break;

			case JOYPAD_LEFT:
				switch (SelectedItem)
					{
					case itemSTART:
						SelectedItem = itemGAMEMODE;
						break;

					case itemSKILL_GREEN:
						SelectedItem = itemSKILL_RED;
						break;

					case itemPADDLE_GREEN:
						SelectedItem = itemPADDLE_RED;
						break;
				 	}
				break;

			case JOYPAD_RIGHT:
				switch (SelectedItem)
					{
					case itemGAMEMODE:
						SelectedItem = itemSTART;
						break;

					case itemSKILL_RED:
						SelectedItem = itemSKILL_GREEN;
						break;

					case itemPADDLE_RED:
						SelectedItem = itemPADDLE_GREEN;
						break;
				 	}
				break;
			}
		}

	return(SelectedItem);
	}


/*********************************************************************
 *  FUNCTION: 	   DisplaySkillOption
 *
 *  PARAMETERS:	   
 *
 *  DESCRIPTION:   
 *
 *  RETURNS:	   
 *
 *********************************************************************/
void DisplaySkillOption(short state, short skillwho)
	{
	int text;
	int x1, y1;
	int x2, y2;
	int color;

	if(skillwho == PLAYER_RED)
		color = btnRED;
	else
		color = btnGREEN;

	text = Option.Skill[skillwho]+textNOVICE;

	if(skillwho == PLAYER_RED)
		{
		x1 = OPT_SKILL_RED_X;
		y1 = OPT_SKILL_RED_Y;

		x2 = x1+OPT_TEXT_OFFSET_X;
		y2 = y1+OPT_TEXT_OFFSET_Y;
		}
	else
		{
		x1 = OPT_SKILL_GREEN_X;
		y1 = OPT_SKILL_GREEN_Y;

		x2 = x1+OPT_TEXT_OFFSET_X;
		y2 = y1+OPT_TEXT_OFFSET_Y;
		}

	DisplayButton(x1, y1, color, state);
	DisplayOptionText(text, x2, y2);
	}



/*********************************************************************
 *  FUNCTION: 	   DisplayPaddleOption
 *
 *  PARAMETERS:	   
 *
 *  DESCRIPTION:   
 *
 *  RETURNS:	   
 *
 *********************************************************************/
void DisplayPaddleOption(short state, short paddlewho)
	{
	int text;
	int x1, y1;
	int x2, y2;
	int color;

	if(paddlewho == PLAYER_RED)
		color = btnRED;
	else
		color = btnGREEN;

	text = Option.PaddleResponse[paddlewho]+textSLOW;

	if(paddlewho == PLAYER_RED)
		{
		x1 = OPT_PADDLE_RED_X;
		y1 = OPT_PADDLE_RED_Y;

		x2 = x1+OPT_TEXT_OFFSET_X;
		y2 = y1+OPT_TEXT_OFFSET_Y;
		}
	else
		{
		x1 = OPT_PADDLE_GREEN_X;
		y1 = OPT_PADDLE_GREEN_Y;

		x2 = x1+OPT_TEXT_OFFSET_X;
		y2 = y1+OPT_TEXT_OFFSET_Y;
		}

	DisplayButton(x1, y1, color, state);
	DisplayOptionText(text, x2, y2);
	}

void DisplayCatchOption(int state)
	{
	int text;

	text = Option.Catch+textOFF;

	DisplayButton(OPT_CATCH_X, OPT_CATCH_Y, btnRED, state);
	DisplayOptionText(text, OPT_CATCH_X+OPT_TEXT_OFFSET_X, OPT_CATCH_Y+OPT_TEXT_OFFSET_Y);
	}

void DisplayBreakthruOption(int state)
	{
	int text;

	text = Option.Breakthru+textOFF;

	DisplayButton(OPT_BREAKTHRU_X, OPT_BREAKTHRU_Y, btnRED, state);
	DisplayOptionText(text, OPT_BREAKTHRU_X+OPT_TEXT_OFFSET_X, OPT_BREAKTHRU_Y+OPT_TEXT_OFFSET_Y);
	}

/*********************************************************************
 *  FUNCTION: 	   DisplayMasterVolOption
 *
 *  PARAMETERS:	   
 *
 *  DESCRIPTION:   
 *
 *  RETURNS:	   
 *
 *********************************************************************/
void DisplayMasterVolOption(short state, int volume)
	{
	if(volume > OPT_MAX_VOLUME)
		volume = OPT_MAX_VOLUME;
	if(volume < 0)
		volume = 0;

	DisplayMeterLevel(state, OPT_EFFECTS_SLIDE_X, OPT_EFFECTS_SLIDE_Y, volume);
	}



/*********************************************************************
 *  FUNCTION: 	   DisplayMusicVolOption
 *
 *  PARAMETERS:	   
 *
 *  DESCRIPTION:   
 *
 *  RETURNS:	   
 *
 *********************************************************************/
void DisplayMusicVolOption(short state, int volume)
	{
	if(volume > OPT_MAX_VOLUME)
		volume = OPT_MAX_VOLUME;
	if(volume < 0)
		volume = 0;

	DisplayMeterLevel(state, OPT_MUSIC_SLIDE_X, OPT_MUSIC_SLIDE_Y, volume);
	}


/*********************************************************************
 *  FUNCTION: 	   DisplayOptionScreen
 *
 *  PARAMETERS:	   void
 *
 *  DESCRIPTION:   Display Option screen
 *
 *  RETURNS:	   void
 *
 *********************************************************************/
void DisplayOptionScreen( void )
	{
	ExpandImage(IMG_PlayfieldBuffer, ROM_Panel);
	DisplayStartOption();
	DisplaySoundOptions();
	DisplayUsage();

	DisplayExpandedImage(IMG_Playfield, IMG_PlayfieldBuffer);
	DisplayGameOption();
	RefreshPlayerOptions(TRUE);

	DisplayButton(OPT_GAME_BUTTON_X, OPT_GAME_BUTTON_Y, btnBLUE, OFF);
	DisplayStartButton(ON);

	DisplayMasterVolOption(OFF, Option.EffectsVolume);
	DisplayMusicVolOption(OFF, Option.MusicVolume); 
	
	while(BlitAddIndex != BlitUseIndex);
	}


/*********************************************************************
 *  FUNCTION: 	   DisplayOptionText
 *
 *  PARAMETERS:	   int which
 *				   int x
 *                 int y
 *
 *  DESCRIPTION:   Display Text string on the option screen
 *
 *  RETURNS:	   void
 *
 *********************************************************************/
void DisplayOptionText( int which, int x, int y )
	{
	ULONG step;
	ULONG flags1;
	ULONG pixel1; 
	ULONG pixel2; 
	ULONG flags2;
	ULONG count; 
	ULONG cmd;

	/* Erase Panel area from ROM */
	flags1 = PLAYFIELD_FLAGS;
	step   = (1<<16L)|(-OPT_TEXT_WIDTH & 0x0FFFFL);

	pixel1 = (y<<16) | (x & 0x0FFFFL);

	count  = (OPT_TEXT_HEIGHT<<16L) | (OPT_TEXT_WIDTH & 0x0FFFFL);

	cmd   = LFU_REPLACE | UPDA1 | UPDA2 | SRCEN;

	Blit[BlitAddIndex].Type     = blitBITMAP;

	Blit[BlitAddIndex].A1_Base  = (long)IMG_Playfield;
	Blit[BlitAddIndex].A1_Flags = flags1;
	Blit[BlitAddIndex].A1_Pixel = pixel1;
	Blit[BlitAddIndex].A1_Step  = step;
	
	Blit[BlitAddIndex].A2_Base  = (long)IMG_PlayfieldBuffer;
	Blit[BlitAddIndex].A2_Flags = flags1;
	Blit[BlitAddIndex].A2_Pixel = pixel1;
	Blit[BlitAddIndex].A2_Step  = step;
	
	Blit[BlitAddIndex].B_Count  = count;

	Blit[BlitAddIndex].B_Cmd    = cmd;

	QueBlit();

	pixel2 = 0x0L;
	flags2 = OPT_TEXT_FLAGS;

	cmd   = LFU_REPLACE | UPDA1 | UPDA2 | SRCEN | DSTEN | DCOMPEN;

	Blit[BlitAddIndex].Type     = blitBITMAP;

	Blit[BlitAddIndex].A1_Base  = (long)IMG_Playfield;
	Blit[BlitAddIndex].A1_Flags = flags1;
	Blit[BlitAddIndex].A1_Pixel = pixel1;
	Blit[BlitAddIndex].A1_Step  = step;
	
	Blit[BlitAddIndex].A2_Base  = (long)ROM_OptionText[which];
	Blit[BlitAddIndex].A2_Flags = flags2;
	Blit[BlitAddIndex].A2_Pixel = pixel2;
	Blit[BlitAddIndex].A2_Step  = step;
	
	Blit[BlitAddIndex].B_Count  = count;
	Blit[BlitAddIndex].B_Patd   = 0x0L;

	Blit[BlitAddIndex].B_Cmd    = cmd;

	QueBlit();
	}

/*********************************************************************
 *  FUNCTION: 	   DisplayButton
 *
 *  PARAMETERS:	   char state
 *                 int x
 *                 int y
 *
 *  DESCRIPTION:   Display button on the option screen
 *
 *  RETURNS:	   void
 *
 *********************************************************************/
void DisplayButton( int x, int y, int color, int state )
	{
	ULONG step;
	ULONG flags1;
	ULONG pixel1; 
	ULONG flags2;
	ULONG count; 
	ULONG cmd;

	flags1 = PLAYFIELD_FLAGS;
	step   = (1<<16L)|(-OPT_BUTTON_WIDTH & 0x0FFFFL);

	pixel1 = (y<<16)|(x & 0x0FFFFL);

	count  = (OPT_BUTTON_HEIGHT<<16L) | (OPT_BUTTON_WIDTH & 0x0FFFFL);

	cmd   = LFU_REPLACE | UPDA1 | UPDA2 | SRCEN;

	Blit[BlitAddIndex].Type     = blitBITMAP;

	Blit[BlitAddIndex].A1_Base  = (long)IMG_Playfield;
	Blit[BlitAddIndex].A1_Flags = flags1;
	Blit[BlitAddIndex].A1_Pixel = pixel1;
	Blit[BlitAddIndex].A1_Step  = step;
	
	Blit[BlitAddIndex].A2_Base  = (long)IMG_PlayfieldBuffer;
	Blit[BlitAddIndex].A2_Flags = flags1;
	Blit[BlitAddIndex].A2_Pixel = pixel1;
	Blit[BlitAddIndex].A2_Step  = step;
	
	Blit[BlitAddIndex].B_Count  = count;
	Blit[BlitAddIndex].B_Cmd    = cmd;

	QueBlit();


	flags2 = OPT_BUTTON_FLAGS;

	cmd   = LFU_REPLACE | UPDA1 | UPDA2 | SRCEN | DSTEN | DCOMPEN;

	Blit[BlitAddIndex].Type     = blitBITMAP;

	Blit[BlitAddIndex].A1_Base  = (long)IMG_Playfield;
	Blit[BlitAddIndex].A1_Flags = flags1;
	Blit[BlitAddIndex].A1_Pixel = pixel1;
	Blit[BlitAddIndex].A1_Step  = step;
	
	Blit[BlitAddIndex].A2_Base  = (long)ROM_OptionButton[color+state];
	Blit[BlitAddIndex].A2_Flags = flags2;
	Blit[BlitAddIndex].A2_Pixel = 0x0L;
	Blit[BlitAddIndex].A2_Step  = step;
	
	Blit[BlitAddIndex].B_Patd   = 0x0L;
	Blit[BlitAddIndex].B_Count  = count;

	Blit[BlitAddIndex].B_Cmd    = cmd;

	QueBlit();
  	}

/*********************************************************************
 *  FUNCTION: 	   DisplayStartOption
 *
 *  PARAMETERS:	   char state
 *                 int x
 *                 int y
 *
 *  DESCRIPTION:   Display button on the option screen
 *
 *  RETURNS:	   void
 *
 *********************************************************************/
void DisplayStartOption()
	{
	Blit[BlitAddIndex].Type     = blitBITMAP;

	Blit[BlitAddIndex].A1_Base  = (long)IMG_PlayfieldBuffer;
	Blit[BlitAddIndex].A1_Flags = PITCH1 | PIXEL16 | WID320 | XADDPIX | ZOFFS1;
	Blit[BlitAddIndex].A1_Pixel = (OPT_START_Y<<16) | (OPT_START_X & 0x0FFFFL);
	Blit[BlitAddIndex].A1_Step  = (1<<16L)|(-OPT_START_WIDTH & 0x0FFFFL);
	
	Blit[BlitAddIndex].A2_Base  = (long)ROM_OptionStart;
	Blit[BlitAddIndex].A2_Flags = OPT_START_FLAGS;
	Blit[BlitAddIndex].A2_Pixel = 0x0L;
	Blit[BlitAddIndex].A2_Step  = (1<<16L)|(-OPT_START_WIDTH & 0x0FFFFL);
	
	Blit[BlitAddIndex].B_Patd   = 0x0L;
	Blit[BlitAddIndex].B_Count  = (OPT_START_HEIGHT<<16L) | (OPT_START_WIDTH & 0x0FFFFL);

	Blit[BlitAddIndex].B_Cmd    = LFU_REPLACE | UPDA1 | UPDA2 | SRCEN | DSTEN | DCOMPEN;

	QueBlit();
	}

/*********************************************************************
 *  FUNCTION: 	   DisplayStartButton
 *
 *  PARAMETERS:	   char state
 *
 *  DESCRIPTION:   Display start button on the option screen
 *
 *  RETURNS:	   void
 *
 *********************************************************************/
void DisplayStartButton( int state )
	{
	ULONG step;
	ULONG flags1;
	ULONG pixel1; 
	ULONG pixel2; 
	ULONG flags2;
	ULONG count; 
	ULONG cmd;

	flags1 = PLAYFIELD_FLAGS;
	step   = (1<<16L)|(-OPT_START_BUTTON_WIDTH & 0x0FFFFL);

	pixel1 = (OPT_START_BUTTON_Y<<16) | (OPT_START_BUTTON_X & 0x0FFFFL);

	count  = (OPT_START_BUTTON_HEIGHT<<16L) | (OPT_START_BUTTON_WIDTH & 0x0FFFFL);

	cmd   = LFU_REPLACE | UPDA1 | UPDA2 | SRCEN;

	Blit[BlitAddIndex].Type     = blitBITMAP;

	Blit[BlitAddIndex].A1_Base  = (long)IMG_Playfield;
	Blit[BlitAddIndex].A1_Flags = flags1;
	Blit[BlitAddIndex].A1_Pixel = pixel1;
	Blit[BlitAddIndex].A1_Step  = step;
	
	Blit[BlitAddIndex].A2_Base  = (long)IMG_PlayfieldBuffer;
	Blit[BlitAddIndex].A2_Flags = flags1;
	Blit[BlitAddIndex].A2_Pixel = pixel1;
	Blit[BlitAddIndex].A2_Step  = step;
	
	Blit[BlitAddIndex].B_Count  = count;
	Blit[BlitAddIndex].B_Cmd    = cmd;

	QueBlit();


	pixel2 = (state*OPT_START_BUTTON_HEIGHT)<<16L;
	flags2 = OPT_START_BUTTON_FLAGS;

	cmd   = LFU_REPLACE | UPDA1 | UPDA2 | SRCEN | DSTEN | DCOMPEN;

	Blit[BlitAddIndex].Type     = blitBITMAP;

	Blit[BlitAddIndex].A1_Base  = (long)IMG_Playfield;
	Blit[BlitAddIndex].A1_Flags = flags1;
	Blit[BlitAddIndex].A1_Pixel = pixel1;
	Blit[BlitAddIndex].A1_Step  = step;
	
	Blit[BlitAddIndex].A2_Base  = (long)ROM_OptionStartButton;
	Blit[BlitAddIndex].A2_Flags = flags2;
	Blit[BlitAddIndex].A2_Pixel = pixel2;
	Blit[BlitAddIndex].A2_Step  = step;
	
	Blit[BlitAddIndex].B_Patd   = 0x0L;
	Blit[BlitAddIndex].B_Count  = count;

	Blit[BlitAddIndex].B_Cmd    = cmd;

	QueBlit();
  	}

/*********************************************************************
 *  FUNCTION: 	   DisplayMeterLevel
 *
 *  PARAMETERS:	   int x
 *                 int y
 *                 int position
 *
 *  DESCRIPTION:   Display a meter on the Option screen
 *
 *  RETURNS:	   void
 *
 *********************************************************************/
void DisplayMeterLevel( int state, int x, int y, int position )
	{
	ULONG flags1, flags2;
	ULONG pixel1, pixel2; 
	ULONG step;
	ULONG count; 
	ULONG cmd;
	ULONG patd;

	int MeterLevel;

	int i;

	MeterLevel = MIN(32, position/4);


	for(i=0; i<32; i++)
		{
		if(i >= MeterLevel)
			patd = 0x0505L;
		else if (i >= 27)
			patd = 0xF0FFL;
		else if (i >= 20)
			patd = 0xFFFFL;
		else
			patd = 0x8FFFL;

	 	/* Pattern blit to fill led's of vu meter */
		flags1 = PLAYFIELD_FLAGS;
		step  = (1<<16L) | (-3 & 0xFFFFL);
		pixel1 = (y<<16L) | ((x+i*4) & 0xFFFFL);

		count  = (6<<16L) | (3 & 0xFFFFL);
		cmd    = PATDSEL | UPDA1;

		Blit[BlitAddIndex].Type     = blitPATTERN;

		Blit[BlitAddIndex].A1_Base  = (long)IMG_Playfield;
		Blit[BlitAddIndex].A1_Flags = flags1;
		Blit[BlitAddIndex].A1_Pixel = pixel1;
		Blit[BlitAddIndex].A1_Step  = step;
		
		Blit[BlitAddIndex].B_Count  = count;
		Blit[BlitAddIndex].B_Patd	  = patd;
		
		Blit[BlitAddIndex].B_Cmd    = cmd;

		QueBlit();
		}

	flags1 = PLAYFIELD_FLAGS;
	pixel1 = ((y+OPT_SLIDE_OFFSET_Y)<<16L) | ((x+OPT_SLIDE_OFFSET_X) & 0xFFFFL);
	step  = (1<<16L) | (-OPT_SLIDE_WIDTH & 0xFFFFL);

	flags2 = OPT_SLIDE_FLAGS;
	pixel2 = 0x0L;

	count  = (OPT_SLIDE_HEIGHT<<16L) | (OPT_SLIDE_WIDTH & 0xFFFFL);

	cmd   = LFU_REPLACE | UPDA1 | UPDA2 | SRCEN | DSTEN | DCOMPEN;

	Blit[BlitAddIndex].Type     = blitBITMAP;

	Blit[BlitAddIndex].A1_Base  = (long)IMG_Playfield;
	Blit[BlitAddIndex].A1_Flags = flags1;
	Blit[BlitAddIndex].A1_Pixel = pixel1;
	Blit[BlitAddIndex].A1_Step  = step;
	
	Blit[BlitAddIndex].A2_Base  = (long)ROM_OptionSlide[state];
	Blit[BlitAddIndex].A2_Flags = flags2;
	Blit[BlitAddIndex].A2_Pixel = pixel2;
	Blit[BlitAddIndex].A2_Step  = step;
	
	Blit[BlitAddIndex].B_Patd   = 0x0L;
	Blit[BlitAddIndex].B_Count  = count;

	Blit[BlitAddIndex].B_Cmd    = cmd;

	QueBlit();
	}

/*********************************************************************
 *  FUNCTION: 	   DisplayGameOption
 *
 *  PARAMETERS:	   char state
 *                 int x
 *                 int y
 *
 *  DESCRIPTION:   Display button on the option screen
 *
 *  RETURNS:	   void
 *
 *********************************************************************/
void DisplayGameOption()
	{
	Blit[BlitAddIndex].Type     = blitBITMAP;

	Blit[BlitAddIndex].A1_Base  = (long)IMG_PlayfieldBuffer;
	Blit[BlitAddIndex].A1_Flags = PITCH1 | PIXEL16 | WID320 | XADDPIX | ZOFFS1;
	Blit[BlitAddIndex].A1_Pixel = (OPT_GAME_Y<<16) | (OPT_GAME_X & 0x0FFFFL);
	Blit[BlitAddIndex].A1_Step  = (1<<16L)|(-OPT_GAME_WIDTH & 0x0FFFFL);
	
	Blit[BlitAddIndex].A2_Base  = (long)ROM_OptionGame[Option.GameMode];
	Blit[BlitAddIndex].A2_Flags = OPT_GAME_FLAGS;
	Blit[BlitAddIndex].A2_Pixel = 0x0L;
	Blit[BlitAddIndex].A2_Step  = (1<<16L)|(-OPT_GAME_WIDTH & 0x0FFFFL);
	
	Blit[BlitAddIndex].B_Patd   = 0x0L;
	Blit[BlitAddIndex].B_Count  = (OPT_GAME_HEIGHT<<16L) | (OPT_GAME_WIDTH & 0x0FFFFL);

	Blit[BlitAddIndex].B_Cmd    = LFU_REPLACE | UPDA1 | UPDA2 | SRCEN | DSTEN | DCOMPEN;

	QueBlit();

	Blit[BlitAddIndex].Type     = blitBITMAP;

	Blit[BlitAddIndex].A1_Base  = (long)IMG_Playfield;
	Blit[BlitAddIndex].A1_Flags = PLAYFIELD_FLAGS;
	Blit[BlitAddIndex].A1_Pixel = (OPT_GAME_Y<<16) | (OPT_GAME_X & 0x0FFFFL);
	Blit[BlitAddIndex].A1_Step  = (1<<16L)|(-OPT_GAME_WIDTH & 0x0FFFFL);
	
	Blit[BlitAddIndex].A2_Base  = (long)ROM_OptionGame[Option.GameMode];
	Blit[BlitAddIndex].A2_Flags = OPT_GAME_FLAGS;
	Blit[BlitAddIndex].A2_Pixel = 0x0L;
	Blit[BlitAddIndex].A2_Step  = (1<<16L)|(-OPT_GAME_WIDTH & 0x0FFFFL);
	
	Blit[BlitAddIndex].B_Patd   = 0x0L;
	Blit[BlitAddIndex].B_Count  = (OPT_GAME_HEIGHT<<16L) | (OPT_GAME_WIDTH & 0x0FFFFL);

	Blit[BlitAddIndex].B_Cmd    = LFU_REPLACE | UPDA1 | UPDA2 | SRCEN | DSTEN | DCOMPEN;

	QueBlit();
  	}


/*********************************************************************
 *  FUNCTION: 	   DisplayUsage
 *
 *  PARAMETERS:	   int x
 *                 int y
 *
 *  DESCRIPTION:   Display button on the option screen
 *
 *  RETURNS:	   void
 *
 *********************************************************************/
void DisplayUsage()
	{
	Blit[BlitAddIndex].Type     = blitBITMAP;

	Blit[BlitAddIndex].A1_Base  = (long)IMG_PlayfieldBuffer;
	Blit[BlitAddIndex].A1_Flags = PITCH1 | PIXEL16 | WID320 | XADDPIX | ZOFFS1;
	Blit[BlitAddIndex].A1_Pixel = (OPT_USAGE_Y<<16) | (OPT_USAGE_X & 0x0FFFFL);
	Blit[BlitAddIndex].A1_Step  = (1<<16L)|(-OPT_USAGE_WIDTH & 0x0FFFFL);
	
	Blit[BlitAddIndex].A2_Base  = (long)ROM_OptionUsage;
	Blit[BlitAddIndex].A2_Flags = OPT_USAGE_FLAGS;
	Blit[BlitAddIndex].A2_Pixel = 0x0L;
	Blit[BlitAddIndex].A2_Step  = (1<<16L)|(-OPT_USAGE_WIDTH & 0x0FFFFL);
	
	Blit[BlitAddIndex].B_Patd   = 0x0L;
	Blit[BlitAddIndex].B_Count  = (OPT_USAGE_HEIGHT<<16L) | (OPT_USAGE_WIDTH & 0x0FFFFL);

	Blit[BlitAddIndex].B_Cmd    = LFU_REPLACE | UPDA1 | UPDA2 | SRCEN | DSTEN | DCOMPEN;

	QueBlit();
  	}


/*********************************************************************
 *  FUNCTION: 	   RefreshPlayerOptions
 *
 *  PARAMETERS:	   int x
 *                 int y
 *
 *  DESCRIPTION:   Display button on the option screen
 *
 *  RETURNS:	   void
 *
 *********************************************************************/
void RefreshPlayerOptions(int DrawBkg)
	{
	if(DrawBkg)
		DisplayPlayerOptions();
	
	switch (Option.GameMode)
		{
		case gameCLASSIC:
			DisplaySkillOption(OFF, PLAYER_RED);
			DisplayPaddleOption(OFF, PLAYER_RED);
			DisplayBreakthruOption(OFF);
			DisplayCatchOption(OFF);
			break;

		case game2000:
			DisplaySkillOption(OFF, PLAYER_RED);
			DisplayPaddleOption(OFF, PLAYER_RED);
			DisplaySkillOption(OFF, PLAYER_GREEN);
			DisplayPaddleOption(OFF, PLAYER_GREEN);
			break;
		}
	}


/*********************************************************************
 *  FUNCTION: 	   DisplayPlayerOptions
 *
 *  PARAMETERS:	   int x
 *                 int y
 *
 *  DESCRIPTION:   Display button on the option screen
 *
 *  RETURNS:	   void
 *
 *********************************************************************/
void DisplayPlayerOptions()
	{
	Blit[BlitAddIndex].Type     = blitBITMAP;

	Blit[BlitAddIndex].A1_Base  = (long)IMG_PlayfieldBuffer;
	Blit[BlitAddIndex].A1_Flags = PITCH1 | PIXEL16 | WID320 | XADDPIX | ZOFFS1;
	Blit[BlitAddIndex].A1_Pixel = (OPT_PLAYER_Y<<16) | (OPT_PLAYER_X&0xFFFFL);
	Blit[BlitAddIndex].A1_Step  = (1<<16L)|(-OPT_PLAYER_WIDTH & 0x0FFFFL);
	
	Blit[BlitAddIndex].A2_Base  = (long)ROM_OptionPlayer[Option.GameMode];
	Blit[BlitAddIndex].A2_Flags = OPT_PLAYER_FLAGS;
	Blit[BlitAddIndex].A2_Pixel = 0x0L;
	Blit[BlitAddIndex].A2_Step  = (1<<16L)|(-OPT_PLAYER_WIDTH & 0x0FFFFL);
	
	Blit[BlitAddIndex].B_Patd   = 0x0L;
	Blit[BlitAddIndex].B_Count  = (OPT_PLAYER_HEIGHT<<16L) | (OPT_PLAYER_WIDTH & 0x0FFFFL);

	Blit[BlitAddIndex].B_Cmd    = LFU_REPLACE | UPDA1 | UPDA2 | SRCEN | DSTEN | DCOMPEN;

	QueBlit();

	Blit[BlitAddIndex].Type     = blitBITMAP;

	Blit[BlitAddIndex].A1_Base  = (long)IMG_Playfield;
	Blit[BlitAddIndex].A1_Flags = PLAYFIELD_FLAGS;
	Blit[BlitAddIndex].A1_Pixel = (OPT_PLAYER_Y<<16) | (OPT_PLAYER_X&0xFFFFL);
	Blit[BlitAddIndex].A1_Step  = (1<<16L)|(-OPT_PLAYER_WIDTH & 0x0FFFFL);
	
	Blit[BlitAddIndex].A2_Base  = (long)ROM_OptionPlayer[Option.GameMode];
	Blit[BlitAddIndex].A2_Flags = OPT_PLAYER_FLAGS;
	Blit[BlitAddIndex].A2_Pixel = 0x0L;
	Blit[BlitAddIndex].A2_Step  = (1<<16L)|(-OPT_PLAYER_WIDTH & 0x0FFFFL);
	
	Blit[BlitAddIndex].B_Patd   = 0x0L;
	Blit[BlitAddIndex].B_Count  = (OPT_PLAYER_HEIGHT<<16L) | (OPT_PLAYER_WIDTH & 0x0FFFFL);

	Blit[BlitAddIndex].B_Cmd    = LFU_REPLACE | UPDA1 | UPDA2 | SRCEN | DSTEN | DCOMPEN;

	QueBlit();
  	}


/*********************************************************************
 *  FUNCTION: 	   DisplaySoundOptions
 *
 *  PARAMETERS:	   int x
 *                 int y
 *
 *  DESCRIPTION:   Display button on the option screen
 *
 *  RETURNS:	   void
 *
 *********************************************************************/
void DisplaySoundOptions()
	{
	Blit[BlitAddIndex].Type     = blitBITMAP;

	Blit[BlitAddIndex].A1_Base  = (long)IMG_PlayfieldBuffer;
	Blit[BlitAddIndex].A1_Flags = PITCH1 | PIXEL16 | WID320 | XADDPIX | ZOFFS1;
	Blit[BlitAddIndex].A1_Pixel = (OPT_SOUND_Y<<16) | (OPT_SOUND_X&0xFFFFL);
	Blit[BlitAddIndex].A1_Step  = (1<<16L)|(-OPT_SOUND_WIDTH & 0x0FFFFL);
	
	Blit[BlitAddIndex].A2_Base  = (long)ROM_OptionSound;
	Blit[BlitAddIndex].A2_Flags = OPT_SOUND_FLAGS;
	Blit[BlitAddIndex].A2_Pixel = 0x0L;
	Blit[BlitAddIndex].A2_Step  = (1<<16L)|(-OPT_SOUND_WIDTH & 0x0FFFFL);
	
	Blit[BlitAddIndex].B_Patd   = 0x0L;
	Blit[BlitAddIndex].B_Count  = (OPT_SOUND_HEIGHT<<16L) | (OPT_SOUND_WIDTH & 0x0FFFFL);

	Blit[BlitAddIndex].B_Cmd    = LFU_REPLACE | UPDA1 | UPDA2 | SRCEN | DSTEN | DCOMPEN;

	QueBlit();
  	}


/*********************************************************************
 *  FUNCTION:	  HandleGameOption
 *
 *  PARAMETERS:	  
 *
 *  DESCRIPTION:  
 *
 *  RETURNS:	  
 *
 *********************************************************************/
void HandleGameOption()
	{
	int Button;

	Button = ReadButtons(PLAYER_RED);

	if(Button & (BUTTON_A | BUTTON_B))
		{
		switch(Button)
			{
			case BUTTON_A:
				PlaySound(sndBRICK2, 127, 4, 7);
				Option.GameMode += 1;
				if(Option.GameMode > 1)
					Option.GameMode = 0;
				break;

			case BUTTON_B:
				PlaySound(sndBRICK2, 127, 0, 7);
				Option.GameMode -= 1;
				if(Option.GameMode < 0)
					Option.GameMode = 1;
				break;
			}

		DisplayGameOption();
		DisplayButton(OPT_GAME_BUTTON_X, OPT_GAME_BUTTON_Y, btnBLUE, ON);
		RefreshPlayerOptions(TRUE);
		while(ReadButtons(PLAYER_RED) != BUTTON_NONE);
		}
	}


/*********************************************************************
 *  FUNCTION:	  HandleSkillOption
 *
 *  PARAMETERS:	  
 *
 *  DESCRIPTION:  
 *
 *  RETURNS:	  
 *
 *********************************************************************/
void HandleSkillOption(short skillwho)
	{
	int Button;
	int text;
	int x, y;
	int maxvalue;

	Button = ReadButtons(PLAYER_RED);

	if(skillwho == PLAYER_RED)
		maxvalue = 2;
	else
		maxvalue = 4;

	if(Button & (BUTTON_A | BUTTON_B))
		{
		switch(Button)
			{
			case BUTTON_A:
				PlaySound(sndBRICK2, 127, 4, 7);
				Option.Skill[skillwho] += 1;
				if(Option.Skill[skillwho] > maxvalue)
					Option.Skill[skillwho] = 0;
				break;

			case BUTTON_B:
				PlaySound(sndBRICK2, 127, 0, 7);
				Option.Skill[skillwho] -= 1;
				if(Option.Skill[skillwho] < 0)
					Option.Skill[skillwho] = maxvalue;
				break;
			}

		SetPaddleSize(PLAYER_RED, PADDLE_MAX_SIZE-Option.Skill[PLAYER_RED]);

		while(ReadButtons(PLAYER_RED) != BUTTON_NONE);

		text = Option.Skill[skillwho]+textNOVICE;

		if(skillwho == PLAYER_RED)
			{
			x = OPT_SKILL_RED_X+OPT_TEXT_OFFSET_X;
			y = OPT_SKILL_RED_Y+OPT_TEXT_OFFSET_Y;
			}
		else
			{
			x = OPT_SKILL_GREEN_X+OPT_TEXT_OFFSET_X;
			y = OPT_SKILL_GREEN_Y+OPT_TEXT_OFFSET_Y;
			}

		DisplayOptionText(text, x, y);
		}
	}

/*********************************************************************
 *  FUNCTION:	  HandlePaddleOption
 *
 *  PARAMETERS:	  
 *
 *  DESCRIPTION:  
 *
 *  RETURNS:	  
 *
 *********************************************************************/
void HandlePaddleOption(short paddlewho)
	{
	int Button;
	int text;
	int x, y;

	Button = ReadButtons(PLAYER_RED);

	if(Button & (BUTTON_A | BUTTON_B))
		{
		switch(Button)
			{
			case BUTTON_A:
				PlaySound(sndBRICK2, 127, 4, 7);
				Option.PaddleResponse[0] += 1;
				if(Option.PaddleResponse[0] > 2)
					Option.PaddleResponse[0] = 0;
				break;

			case BUTTON_B:
				PlaySound(sndBRICK2, 127, 0, 7);
				Option.PaddleResponse[0] -= 1;
				if(Option.PaddleResponse[0] < 0)
					Option.PaddleResponse[0] = 2;
				break;
			}

		Option.PaddleResponse[1] = Option.PaddleResponse[0];

		while(ReadButtons(PLAYER_RED) != BUTTON_NONE);

		text = Option.PaddleResponse[paddlewho]+textSLOW;

		if(paddlewho == PLAYER_RED)
			{
			x = OPT_PADDLE_RED_X+OPT_TEXT_OFFSET_X;
			y = OPT_PADDLE_RED_Y+OPT_TEXT_OFFSET_Y;
			}
		else
			{
			x = OPT_PADDLE_GREEN_X+OPT_TEXT_OFFSET_X;
			y = OPT_PADDLE_GREEN_Y+OPT_TEXT_OFFSET_Y;
			}

		DisplayOptionText(text, x, y);

		Player[PLAYER_RED].Paddle.MaxSpeedX = PADDLE_MAX_SPEED_X+200*
													Option.PaddleResponse[PLAYER_RED];
		}
	}

/*********************************************************************
 *  FUNCTION:	  HandleCatchOption
 *
 *  PARAMETERS:	  
 *
 *  DESCRIPTION:  
 *
 *  RETURNS:	  
 *
 *********************************************************************/
void HandleCatchOption()
	{
	int Button;

	Button = ReadButtons(PLAYER_RED);

	if(Button & (BUTTON_A | BUTTON_B))
		{
		switch(Button)
			{
			case BUTTON_A:
				PlaySound(sndBRICK2, 127, 4, 7);
				break;

			case BUTTON_B:
				PlaySound(sndBRICK2, 127, 0, 7);
				break;
			}

		if(Option.Catch)
			Option.Catch = FALSE;
		else
			Option.Catch = TRUE;

		DisplayCatchOption(ON);

		while(ReadButtons(PLAYER_RED) != BUTTON_NONE);
		}
	}

/*********************************************************************
 *  FUNCTION:	  HandleBreakthruOption
 *
 *  PARAMETERS:	  
 *
 *  DESCRIPTION:  
 *
 *  RETURNS:	  
 *
 *********************************************************************/
void HandleBreakthruOption()
	{
	int Button;

	Button = ReadButtons(PLAYER_RED);

	if(Button & (BUTTON_A | BUTTON_B))
		{
		switch(Button)
			{
			case BUTTON_A:
				PlaySound(sndBRICK2, 127, 4, 7);
				break;

			case BUTTON_B:
				PlaySound(sndBRICK2, 127, 0, 7);
				break;
			}

		if(Option.Breakthru)
			Option.Breakthru = FALSE;
		else
			Option.Breakthru = TRUE;

		DisplayBreakthruOption(ON);

		while(ReadButtons(PLAYER_RED) != BUTTON_NONE);
		}
	}


/*********************************************************************
 *  FUNCTION:	  HandleEffectsVolOption
 *
 *  PARAMETERS:	  
 *
 *  DESCRIPTION:  
 *
 *  RETURNS:	  
 *
 *********************************************************************/
void HandleEffectsVolOption()
	{
	int JoyStick;
	int Button;

	JoyStick = ReadJoyStick(PLAYER_RED);
	Button = ReadButtons(PLAYER_RED);

	if(Button & BUTTON_A)
		JoyStick = JOYPAD_RIGHT;
	if(Button & BUTTON_B)
		JoyStick = JOYPAD_LEFT;

	if(JoyStick & (JOYPAD_LEFT | JOYPAD_RIGHT))	
		{
		switch(JoyStick)
			{
			case JOYPAD_RIGHT:
				Option.EffectsVolume += 1;
				if(Option.EffectsVolume > OPT_MAX_VOLUME)
					Option.EffectsVolume = OPT_MAX_VOLUME;
				break;

			case JOYPAD_LEFT:
				Option.EffectsVolume -= 1;
				if(Option.EffectsVolume < 0)
					Option.EffectsVolume = 0;
				break;
			}

		SSFXVolume(Option.EffectsVolume <<8);

		DisplayMeterLevel(TRUE, OPT_EFFECTS_SLIDE_X, OPT_EFFECTS_SLIDE_Y, 
									Option.EffectsVolume);
		}
	}

/*********************************************************************
 *  FUNCTION:	  HandleMusicVolOption
 *
 *  PARAMETERS:	  
 *
 *  DESCRIPTION:  
 *
 *  RETURNS:	  
 *
 *********************************************************************/
void HandleMusicVolOption()
	{
	int JoyStick;
	int Button;

	JoyStick = ReadJoyStick(PLAYER_RED);
	Button = ReadButtons(PLAYER_RED);

	if(Button & BUTTON_A)
		JoyStick = JOYPAD_RIGHT;
	if(Button & BUTTON_B)
		JoyStick = JOYPAD_LEFT;

	if(JoyStick & (JOYPAD_LEFT | JOYPAD_RIGHT))
		{
		switch(JoyStick)
			{
			case JOYPAD_RIGHT:
				if(!Option.MusicOn)
					{
					Option.MusicOn = TRUE;
					StartMusic(songTITLE);
					}
				Option.MusicVolume += 1;
				if(Option.MusicVolume > OPT_MAX_VOLUME)
					Option.MusicVolume = OPT_MAX_VOLUME;
				break;

			case JOYPAD_LEFT:
				if(!Option.MusicOn)
					{
					Option.MusicOn = TRUE;
					StartMusic(songTITLE);
					}
				Option.MusicVolume -= 1;
				if(Option.MusicVolume < 0)
					Option.MusicVolume = 0;
				break;
			}

		SMainVolume(Option.MusicVolume <<8);

		DisplayMeterLevel(TRUE, OPT_MUSIC_SLIDE_X, OPT_MUSIC_SLIDE_Y, 
							Option.MusicVolume);
		}
	}

