////////////////////////////////////////////////////////////////////////////////////////////////////
// SLN - SubQMod's Linker for the Atari Jaguar Console System
// SLN.H - Application Header
// Copyright (c) SubQMod, 2008

#ifndef __SLH_H__
#define __SLH_H__

// --- Macro Definitions ---------------------------------------------------------------------------

// Requirements for Windows Compilation

#ifdef WIN32
#define _OPEN_FLAGS  _O_BINARY|_O_RDWR
#define _BACKSLASH   '\\'
#ifdef _MSC_VER
   #if _MSC_VER > 1000
      #pragma warning(disable:4996)
   #endif
#endif
#include <io.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <setjmp.h>
#endif

// Requirements for Mac OS-X or Linux Compilation

#ifdef __GCCUNIX__
#define _OPEN_FLAGS  O_RDWR
#define _BACKSLASH   '/'
#include <sys/fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <setjmp.h>
#include <unistd.h>
#endif

#define MAJOR        0                                      // Major version number
#define MINOR        1                                      // Minor version number
#define PATCH        4                                      // Patch release number

#ifdef WIN32
#define PLATFORM     "Win32"                                // Release platform - Windows
#else 
#ifdef __GCCUNIX__
#define PLATFORM     "OSX/Linux"                            // Release platform - MAC OSX or Linux
#else
#define PLATFORM     "Unknown"                              // Release platform - Not Specified 
#endif
#endif

// Command link flag, warning macro
#define warn(x, f) printf("Warning: repeated flag `%c'%s\n", x, f ? "; previous one(s) ignored." : ".")

// Macro for max: good because longs, shorts, or pointers can be compared
#ifndef max
#define max(a,b) ((a) > (b) ? (a) : (b))
#endif max

// Macro to swap the 16-bit words of a 32-bit integer
#define _SWAPWORD(x) (((unsigned)(x) >> 16) | ((unsigned)(x) << 16)) 

#define FARGSIZE     1024                                   // Number of chars in filename argument
#define FNLEN        1024                                   // Size of a file name 
#define NHANDLES     256                                    // Number of open file handles at once 
#define OST_BLOCK    0x400000                               // Output symbol table block (4MB)
#define DSTSEG_D     1                                      // Include file destination seg (DATA)
#define DSTSEG_T     2                                      // Include file destination seg (TEXT)
#define MAXARGS 256                                         // Max number of args in a command file

// --- Headers -------------------------------------------------------------------------------------
// Most of these structures reflect the actual format of the object in question, on a 68000: char 
// means one byte, int means two bytes, long means four.  If the host machine doesn't have this 
// same format (like a VAX), you will have to read the file into a buffer and stuff the values into 
// the structure (see slongio.c).

struct OHEADER {
   long magic;			                                       // 0x0107 for .o, 0x601b for abs
   long tsize;
   long dsize;
   long bsize;
   long ssize;
   union {
      struct {			                                       // For .o 
         long tsize;                                        // Text relocation size
         long dsize;                                        // Data relocation size
         char reserved[12];
      } reloc;
      struct {			                                       // For .abs 
         long stksize;		                                 // Unused 
         long tstart;                                       // Start of TEXT 
         long rbflag;                                        // -1 if no fixups at all 
         long dstart;                                       // Start of DATA 
         long bstart;                                       // Start of BSS
      } abs;
   } absrel;
   char *ostbase;                                           // Base of output symbol table 
   long fsize;                                              // Length of fixups
   char *fixups;                                            // Start of fixups 
};

#define new_oheader()   (struct OHEADER *)malloc((long)sizeof(struct OHEADER))

struct ARHEADER {
   char a_fname[14];
   long a_modti;
   char a_userid;
   char a_gid;
   int a_fimode;
   long a_fsize;
   int reserved;                                            // Two bytes zeroes btw header & file 
};

#define new_arheader()  (struct ARHEADER *)malloc((long)sizeof(struct ARHEADER))

// --- Object File Structure and Related Items -----------------------------------------------------

struct OFILE {
   char o_name[FNLEN];                                      // Fixed-length names
   char o_arname[FNLEN];                                    // Name of archive this is from
   struct OFILE *o_next;                                    // Next object file
   long o_tbase, o_dbase, o_bbase;                          // Computed bases for this ofile
   int o_symstart;                                          // First sym in image is nth in out
   int o_flags;                                             // Flags (see O_*)
   struct OHEADER o_header;                                 // Header of this file
   char *o_image;                                           // Image of this file
};

#define new_ofile()  (struct OFILE *)malloc((long)sizeof(struct OFILE))

// Flags in an Object File's o_flags field
// O_USED: means this ofile is used or is on the command line or in a -x 
#define O_USED	      0x0001
#define O_ARCHIVE    0x0002                                 // This is a dummy archive entry

// --- Symbol Record -------------------------------------------------------------------------------

// SYMREC: Used by builddir for the lists of exports and imports, and by the linker for the output
// symbol table (that's why there are type and value fields, unused in builddir)

#define SYMLEN       100                                    // Symbol name size (incl null)

struct SYMREC {
   char s_name[SYMLEN];                                     // Including null terminator 
   int s_type;
   long s_value;
   struct SYMREC *s_next;
};

#define new_symrec() (struct SYMREC *)malloc((long)sizeof(struct SYMREC))

// --- Hash Record ---------------------------------------------------------------------------------

// HREC: One item in a hash bucket, including a link to the next item. Commons and Globals share a 
// hash table, but their value fields are interpreted differently.  

struct HREC {
   char h_sym[SYMLEN];
   struct HREC *h_next;
   struct OFILE *h_ofile;
   long h_value;
   int h_type;
};

#define new_hrec()   (struct HREC *)malloc((long)sizeof(struct HREC))

#define NBUCKETS     1024                                   // Number of hash buckets

// Bit definitions for the type field of a symbol.
//
// There is a special case for the linker here: the bit T_OST is a flag
// meaning that the symbol's value field contains an INT (in host-machine
// format, in the first two bytes of the field) which is the index of the
// symbol in the output symbol table.
//
// If that field is -1, it means you have to look this symbol up in the
// ost to get its index.  This happens when the symbol was extern to this
// module and defined by a LATER module.
//
// The upshot is that a symbol which isn't in the ost has its type & value
// fields intact, while a symbol which is in the ost has T_OST set and
// its index in its value field (or -1 if you have to look it up).
// When producing the output fixups, you either output a symbol fixup with
// the new index (for a partial link), or resolve the symbol based on its
// type & value from the output symbol table.

#define ABST_DEFINED    0x8000
#define ABST_EQUATED    0x4000
#define ABST_GLOBAL     0x2000
#define ABST_REGISTER	0x1000
#define ABST_EXTERN     0x0800
#define ABST_DATA       0x0400	/* data-based relocatable */
#define ABST_TEXT       0x0200	/* text-based relocatable */
#define ABST_BSS        0x0100	/* bss-based relocatable  */
#define ABST_FILE       0x0080                                    // file symbol 
#define ABST_ARCHIVE    0x0040                                    // only when FILE set: archive file or no 
#define ABST_OST        0x0001                                    // private: "symbol is in ost": see above 
#define T_COMMON	(T_GLOBAL | T_EXTERN)
#define T_SEG		(T_DATA | T_TEXT | T_BSS)                 // segment bits 

// Symbol Table - Type Definitions

#define T_UNDF          0x00000000     // Undefined Symbol
#define T_EXT           0x01000000     // External Bit, ORed In
#define T_ABS           0x02000000     // Absolute Symbol
#define T_TEXT          0x04000000     // TEXT Segment
#define T_DATA          0x06000000     // DATA Segment
#define T_BSS           0x08000000     // BSS Segment

// These macros are used with the TYPE field of a SYMBOL.

#define iscommon(type) (((type) & T_EXT) == T_EXT)
#define isglobal(type) (((type) & T_EXT) == T_EXT)
#define isextern(type) (((type) & T_EXT) == T_EXT)
#define islocal(type)  (((type) & T_EXT) == 0)

// This macro is used to compare two symbols for equality. It depends on
// symcopy remaining as it is (copies two longs plus a null)

#define symcmp(a,b) ((*(long *)(a) == *(long *)(b)) && \
		              (*(long *)((a) + sizeof(long)) == \
		              *(long *)((b) + sizeof(long))))

// --- Function Prototypes -------------------------------------------------------------------------

int doargs(int, char *[]);
char *make_string(char *);
void put_name(struct OFILE *);
int flush_handles(void);
void symcopy(char *, char *);
int ost_add(char *,int, long);
int add_fixup(long);
void display_help(void);
void display_version(void);
int pladd(char *, char *);
char *path_tail(char *);
int dolist(void);
int segmentpad(FILE *, long, int);
int ost_lookup(char *);

#endif // __SLH_H__
